/* AstroPrincipal.m
 * Commander of the Astrology module
 *
 * Copyright (C) 2003-2005 by vhf interservice GmbH
 * Author:   Georg Fleischmann
 *
 * created:  2003-03-29
 * modified: 2005-11-18 (App*)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the vhf Public License as
 * published by vhf interservice GmbH. Among other things, the
 * License requires that the copyright notices and this notice
 * be preserved on all copies.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the vhf Public License for more details.
 *
 * You should have received a copy of the vhf Public License along
 * with this program; see the file LICENSE. If not, write to vhf.
 *
 * vhf interservice GmbH, Im Marxle 3, 72119 Altingen, Germany
 * eMail: info@vhf.de
 * http://www.vhf.de
 */

#include <AppKit/AppKit.h>
#include <VHFShared/VHFSystemAdditions.h>
#include "../Cenon/App.h"
#include "../Cenon/LayerObject.h"
#include "../Cenon/Graphics.h"
#include "../Cenon/functions.h"
#include "../Cenon/messages.h"
#include "AstroPrincipal.h"
#include "DocViewAstro.h"
#include "DocViewMap.h"
#include "AstroPanels.h"
#include "AstroChart.h"
#include "Astro.bproj/AstroController.h"
#include "CityManager.h"
#include "EventManager.h"
#include "TimeZoneManager.h"
#include "Map.h"
#include "astroMessages.h"

@interface AstroPrincipal(PrivateMethods)
@end

@implementation AstroPrincipal

/*  137.1234 -> 137 7'24"
 * -137.1234 -> -137 7'24"
 */
NSString *stringFromDeg(float deg)
{   int		d, m, s;
    NSString	*string;

    //if (deg < 0)
    //    NSLog(@"stringFromDeg(): does not properly support negative degrees !");
    d = (int)deg;
    m = (int)floor((deg-d) * 60.0);
    s = (int)floor((deg-d-m/60.0) * 3600.0 + 0.5);
    string = [NSString stringWithFormat:DEGREE_FORMAT,
                       [[NSString stringWithFormat:@"%d", d] stringWithLength:(deg < 0.0) ? -4 : -3],
                       [[NSString stringWithFormat:@"%d", m] stringWithLength:-2],
                       [[NSString stringWithFormat:@"%d", s] stringWithLength:-2] ];
    return string;
}

/*
 * created:  1999-03-15
 * modified: 2005-10-19 (ACFix, declination)
 * register the defaults
 */
+ (void)initialize
{   NSMutableDictionary	*registrationDict = [NSMutableDictionary dictionary];
    NSArray		*objects;

    [registrationDict setObject:vhfStringFromRGBColor([NSColor whiteColor]) forKey:@"astroBGColor"];
    [registrationDict setObject:@"YES" forKey:@"astroShowPanel"];
    [registrationDict setObject:@"YES" forKey:@"astroShowRetrograde"];
    [registrationDict setObject:@"%d.%m.%Y" forKey:@"astroDateFormat"];
    [registrationDict setObject:@"vhfAstro" forKey:@"astroFont"];
    //[registrationDict setObject:@"YES" forKey:@"astroShowHouseSize"];
    //[registrationDict setObject:@"YES" forKey:@"astroShowHouseMaxima"];
    [registrationDict setObject:@"YES" forKey:@"astroTopocentricPosition"];
    //[registrationDict setObject:@"YES" forKey:@"astroShowWaves"];
    [registrationDict setObject:@"NO" forKey:@"astroShowDeclination"];
    [registrationDict setObject:@"NO" forKey:@"astroACFix"];
    objects = [NSArray arrayWithObjects:@"Sun", @"Moon", @"Mercury", @"Venus", @"Mars", @"Jupiter", @"Saturn", @"Uranus", @"Neptune", @"Pluto", @"Chiron", @"True Node", @"True Apogee", nil];
    [registrationDict setObject:objects forKey:@"astroObjects"];

    [[NSUserDefaults standardUserDefaults] registerDefaults:registrationDict];
}

+ (id)instance
{   static id instance = nil;

    if (!instance)
    {   instance = [self alloc];
        [instance init];
    }
    return instance;
}

/* init
 */
- init
{   NSNotificationCenter	*notificationCenter = [NSNotificationCenter defaultCenter];
#if !defined(GNUSTEP_BASE_VERSION) && !defined(__APPLE__)	// OpenStep 4.2
    NSMenu			*toolMenu = [[[NSApp mainMenu] itemWithTag:MENU_TOOLS] target];
#else
    NSMenu			*toolMenu = [[[NSApp mainMenu] itemAtIndex:MENU_TOOLS-1] submenu];
#endif

    /* add observer for documents that open */
    [notificationCenter addObserver:self
                           selector:@selector(documentDidOpen:)
                               name:DocumentDidOpen
                             object:nil];

    /* add observer to calculate chart */
    [notificationCenter addObserver:self
                           selector:@selector(calculateChart:)
                               name:CalculateChartNotification
                             object:nil];

    /* add observer to load map image at date */
    [notificationCenter addObserver:self
                           selector:@selector(updateMapForDate:)
                               name:AstroDateSetNotification
                             object:nil];

    /* add a menu item for the AstroPanel */
    [toolMenu addItemWithTitle:@"Astro Panel ..."
                        action:@selector(showAstroPanel:)
                 keyEquivalent:@""];
    [[[toolMenu itemArray] objectAtIndex:[[toolMenu itemArray] count]-1] setTarget:self];
    if (Prefs_ShowAstroPanel)
        [self showAstroPanel:self];

    /* create event manager, city manager, time zone manager */
    eventManager = [EventManager new];
    cityManager = [CityManager new];
    timeZoneManager = [TimeZoneManager new];

    return self;
}



/* AstroPanel stuff
 */
- (void)showAstroPanel:sender
{
    if (!astroPanel)
    {   NSBundle	*bundle = [NSBundle bundleForClass:[AstroPanel class]];

        //if (![NSBundle loadModelNamed:@"AstroPanel" owner:self])
        if ( ![bundle loadNibFile:@"AstroPanel"
                externalNameTable:[NSDictionary dictionaryWithObject:self forKey:@"NSOwner"]
                         withZone:[self zone]] )
            NSLog(@"Cannot load AstroPanel interface file");
        [astroPanel init];
    }
    [astroPanel setFrameUsingName:@"AstroPanel"];
    [astroPanel setFrameAutosaveName:@"AstroPanel"];
    [astroPanel setBecomesKeyOnlyIfNeeded:YES];
/*#ifndef GNUSTEP_BASE_VERSION
    [astroPanel setFloatingPanel:YES];
#endif*/
    [astroPanel orderFront:sender];
}
- astroPanel
{
    return astroPanel;
}

- eventManager		{ return eventManager; }
- cityManager		{ return cityManager; }
- timeZoneManager	{ return timeZoneManager; }

- (Ephemeris*)ephemeris
{
    if (!ephemeris)
    {   NSFileManager	*fileManager = [NSFileManager defaultManager];
        NSArray		*paths;
        int		i;
        NSString	*path = nil;

        /* set path to ephemeris files */
        paths = [NSArray arrayWithObjects:
                         [[[NSBundle bundleForClass:[Ephemeris class]] resourcePath]
                          stringByAppendingPathComponent:@"Ephemeris"],
                         [userLibrary()  stringByAppendingPathComponent:@"Astro/Ephemeris"],
                         [localLibrary() stringByAppendingPathComponent:@"Astro/Ephemeris"], nil];
        for (i=0; i<(int)[paths count]; i++)
            if ([fileManager fileExistsAtPath:[paths objectAtIndex:i]] || i == (int)[paths count]-1)
            {   path = [paths objectAtIndex:i];
                break;
            }
        ephemeris = [Ephemeris new];
        [ephemeris setEphemerisPath:path];
    }
    return ephemeris;
}


/* notification that a new document has been opened
 */
- (void)documentDidOpen:(NSNotification*)notification
{   DocView	*view = [notification object];

    if ([view isMemberOfClass:[DocView class]])
    {
        /* these methods are called for every view.
         * We have to make sure that we do our special stuff for our documents only.
         */
        [view initMap];

        if ([view respondsToSelector:@selector(mapDocumentHasBeenSaved:)])
            [[NSNotificationCenter defaultCenter] addObserver:view
                                                     selector:@selector(mapDocumentHasBeenSaved:)
                                                         name:DocumentHasBeenSaved
                                                       object:nil];
    }
    else
        NSLog(@"AstroPrincipal, notification send from object %@, shoult be DocView!", view);
}

/* notification that we have to calculate a chart
 *
 * dict   name	    NSString       title of the chart
 *        date      NSCalendarDate date of the chart
 *        lat       NSNumber       latitude
 *        lon       NSNumber       longitude
 *        city      NSString       location
 *        composite NSDictionary   optional dictionary with entries like above for a composite chart
 */
- (void)calculateChart:(NSNotification*)notification
{   NSDictionary	*dict = [notification object];

    if ([dict isKindOfClass:[NSDictionary class]])
    {   int		i;
        Document	*doc;
        AstroChart	*astroChart;

        for ( i=[[NSApp windows] count]-1; i>=0; i-- )
        {   doc = [(App*)NSApp documentInWindow:[[(App*)NSApp windows] objectAtIndex:i]];
            if ( [[doc name] hasPrefix:CHART_PREFIX] )
                break;
        }
        if ( i < 0 )
        {   NSLog(@"AstroPrincipal: No document found with prefix '%@' to chart data!", CHART_PREFIX);
            return;
        }

  	astroChart = [[astroPanel windowAt:AP_CHART] astroChart];
        if (!astroChart)
            astroChart = [AstroChart astroChartWithView:[doc documentView]];
        [astroChart radix:[dict objectForKey:@"date"]
                longitude:[dict floatForKey:@"lon"]
                 latitude:[dict floatForKey:@"lat"]
                     city:[dict objectForKey:@"city"]
                    title:[dict objectForKey:@"name"]
                composite:[dict objectForKey:@"composite"]];
    }
    else
        NSLog(@"AstroPrincipal (calculateChart), notification send with an object not a dictionary!");
}

/* notification that the date was set, and we have to load the corresponding map image
 *
 * object   NSCalendarDate with date and time of map to load
 */
- (void)updateMapForDate:(NSNotification*)notification
{   NSCalendarDate	*date = [notification object];

    if ([date isKindOfClass:[NSCalendarDate class]])
    {   int		i;
        Document	*doc;
        Map		*map;
        DocView		*view;

        for ( i=[[NSApp windows] count]-1; i>=0; i-- )
        {   doc = [(App*)NSApp documentInWindow:[[(App*)NSApp windows] objectAtIndex:i]];
            if ( [[doc name] hasPrefix:MAP_PREFIX] )
                break;
        }
        if ( i < 0 )
            return;

        view = [doc documentView];
        map = [[view statusDict] objectForKey:@"map"];
        [map displayMapAtDate:date inView:view];
    }
    else
        NSLog(@"AstroPrincipal (updateMapForDate), notification send with an object not a date!");
}

@end
