/* CIAMap.m
 * Astrological map for Cenon based on the CIA maps
 *
 * Created by Ilonka Fleischmann, 2001
 * Rewritten from mapreader.c created by Steve Putz, 22-Aug-90
 *
 * created:  2001-07-30
 * modified: 2003-07-11
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the vhf Public License as
 * published by vhf interservice GmbH. Among other things, the
 * License requires that the copyright notices and this notice
 * be preserved on all copies.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the vhf Public License for more details.
 *
 * You should have received a copy of the vhf Public License along
 * with this program; see the file LICENSE. If not, write to vhf.
 */

#include <Foundation/Foundation.h> 
#include <stdio.h> 
#include <stdlib.h> 
#include <errno.h>
#include <string.h>
#include <ctype.h>		// for isdigit() macro
#include <math.h>
#include <sys/types.h>		// for sys/stat.h
#include <sys/stat.h>		// for stat()

#include <VHFShared/types.h>
#include <VHFShared/vhfCommonFunctions.h>
#include "../Cenon/App.h"
#include "../Cenon/Document.h"
#include "../Cenon/Graphics.h"
#include "../Cenon/locations.h"
#include "../Cenon/messages.h"
#include "../Cenon/functions.h"
#include "../DocViewMap.h"
#include "../astroLocations.h"
#include "CIAMap.h"

/* Private methods
 */
@interface CIAMap(PrivateMethods)
@end

@implementation CIAMap

void map_read_profile_entry(MapSet *ms, char *line);

/* constants
 */
#define DEF_CWIDTH	841 // A4
#define DEF_CHEIGHT	595 // A4
#define MAX_SEGLIST	10000		/* max vectors to draw at once */

/*****************************************************************************
 * routines for reading vector map data
 *
 * Created by Steve Putz, 22-Aug-90
 * Rewritten from mapbrowser.c created by Steve Putz, 5-Oct-89
 *****************************************************************************
 * Options:
 *	-map_dir dir
 *	-map_profile file
 *	-map_local_profile file
 *	-map_profentry 'profile entry'
 *	-map_maxopen N
 *	-map_maxmaps N
 *	-map_maxtypes N
 *	-map_maxscales N
 *	-map_maxcolors N
 *
 * Profile entry types:
 *	#comment
 *	-mapdir dirName
 *	-color colorIndex pattern and|or red green blue
 *	-background colorIndex
 *	-highlight colorIndex
 *	-grid colorIndex lineWidth
 *	-resolution scaleName minAutoRes minRes
 *	-type typeName
 *	-feature index colorIndex lineWidth feature_description
 *	-min_features binaryFeatureMask
 *	-suppress binaryFeatureMask
 *	-optional
 *	-map fileName [featureOffset]
 *	-readprofile filename
 *	-ll_center longitude latitude
 *	-ll_extent longitude latitude
 *
 * The mapdir defaults to the profile's directory.
 * The resolution values (in dots/degree) are used to select which set of maps
 * to use for a given resolution.  A smaller scale set is used when a
 * resolution of minAutoRes (if automatic scaling is enabled) or minRes
 * is exceeded.
 * The pattern is one of solid, dotted, dashed, dashdot, dashdotdotted,
 * longdashed.
 * Example profile:
 *	-color  0 solid 	and 0xff 0xff 0xff # white (background)
 *	-color  1 solid 	and 0x00 0x00 0x00 # black (coasts)
 *	-color  2 dashdot	or  0xc0 0x90 0x00 # brown (borders)
 *	-color  3 longdashed	and 0xc0 0xe0 0xff # blue (lakes)
 *	-color  4 dashdotdotted	or  0xff 0x00 0x00 # red (highlight color)
 *	-color  5 solid 	and 0xd0 0xd0 0xd0 # gray (grid)
 *	-background 0
 *	-highlight 3
 *	-grid 4 1
 *	-type coast
 *	-feature  1 1 1 coastlines
 *	-feature  2 3 1 lakes
 *	-min_features 0001
 *	-type borders
 *	-optional
 *	-feature  1 2 1 country boundary
 *	-feature  4 2 1 state boundary
 *	-resolution medium 0 0
 *	-type coast
 *	-map NorthAmerica/cil-medium.map
 *	-map SouthAmerica/cil-medium.map
 *	-type borders
 *	-suppress 0100
 *	-map NorthAmerica/bdy-medium.map
 *	-map NorthAmerica/pby-medium.map +3
 *	-map SouthAmerica/bdy-medium.map
 *	-resolution high 50 20
 *	-type coast
 *	-map NorthAmerica/cil.map
 *	-map SouthAmerica/cil.map
 *	-type borders
 *	-map NorthAmerica/bdy.map
 *	-map NorthAmerica/pby.map +3
 *	-map SouthAmerica/bdy.map
 *****************************************************************************
 * Variables ending with Lat or Lng are floating point values in degrees.
 * Variables ending with Sec are integer values in seconds.
 * Variables ending with U are integer values in units of a particular map.
 * Drawing units are integer pixel coordinates (upper left origin).
 * The point of interest (centerLng, centerLat) is centered
 * in the drawing space.  Lattitudes are normalized to this
 * rotated coordinate system using the NORMALIZE_LONGITUDE() macro.
 *****************************************************************************/

#define	EarthRadiusA	6378137.0
#define	EarthRadiusB	6356752.0
#define Eccentricity	(EarthRadiusA - EarthRadiusB)/EarthRadiusA
#define	OneDegInM	(2.0*Pi*EarthRadiusA / 360.0)	// meter -> degree (2*Pi*r/360 -> 1 degree in meter)

#define ENV_PROFDIR	"MAPBROWSER_DIR"
#define ENV_PROFNAME	"MAPBROWSER_PROFILE"
#define ENV_MAXMAPS	"MAPBROWSER_MAXMAPS"
#define ENV_MAXTYPES	"MAPBROWSER_MAXTYPES"
#define ENV_MAXSCALES	"MAPBROWSER_MAXSCALES"
#define ENV_MAXCOLORS	"MAPBROWSER_MAXCOLORS"

#define DEF_PROFNAME	"world_8f6.mapset"	// low res
#define MAX_FEATURE	32
#define MAX_MAGNIFY	8			// max factor beyond data resolution
#define MAX_LINELEN	1024
#define FILE_RESERVE	2
#define DEF_MAPTYPE	"map"
#define DEF_GRAT_FLAG	GRAT_ALL
#define GRAT_NONE	0
#define GRAT_OUTLINE	1
#define GRAT_ALL	2
#define GRAT_MERIDIANS	12
#define MAP_MARGIN	8
#define FNAME_STD	"-"
#define TOK_SEPS	" \t\n"
#define BG_COLOR	0
#define FG_COLOR	1

#define D_DEFAULT	1
#define D_VERBOSE	0x00001
#define D_READMAP	0x00002
#define D_PROFILE	0x00004
#define D_RESCALE	0x00008
#define D_FILES		0x00010

static MapFile		**filePool = NULL;
static int		poolSize = 0;

struct mystruct
{
    char   byte1;	// Mysterious 7-digit number
    char   byte2;	// Mysterious 7-digit number
    char   byte3;	// Mysterious 7-digit number
    char   byte4;	// Mysterious 7-digit number
    char   maxlat1;	// maxlat
    char   maxlat2;	// maxlat
    char   maxlat3;	// maxlat
    char   maxlat4;	// maxlat
};


#define GETSHORT(val, file) \
{   val = getc(file) << 8;\
    val = (short) (val | (getc(file) & 0xFF));\
}

#define GETLONG(val, file) \
{   val = getc(file) << 24;\
    val |= getc(file) << 16;\
    val |= getc(file) << 8;\
    val |= getc(file);\
}

#define UNPACK_COORDS(lx, ly, file) \
{   lx = (char)getc(file);\
    if (lx & SHORTBYTE) \
    {   if (lx > 0) lx &= ~SHORTBYTE;\
        ly = (char)getc(file);\
    } \
    else \
    {   if (lx < 0) lx |= SHORTBYTE;\
        lx <<= 24;\
        lx |= getc(file) << 16;\
        lx |= getc(file) << 8;\
        lx |= getc(file);\
        GETLONG(ly, file);\
    }\
}

#define MAGSHIFT		8	// fixed-point for scaled map data
#define FATTEN_THRESH		(1 << MAGSHIFT)

#define LAT_TO_MILES(deg)	((float) (deg)*60*1.15)
#define DEG_TO_SEC(deg)		((long) ((deg) * SEC_PER_DEG))
#define SEC_TO_DEG(sec)		((float) (sec) / SEC_PER_DEG)

#define DOTS_TO_SEC(ms,dots)	(((dots) * (ms)->msecPerDot) >> MAGSHIFT)
//#define SEC_TO_DOTS(ms,secs)	(((secs) << MAGSHIFT) / (ms)->msecPerDot)
#define SEC_TO_DOTS(ms,secs)	(((float)((secs) << MAGSHIFT)) / (float)(ms)->msecPerDot)
#define SEC_PER_DOT(ms)		((float) (ms)->msecPerDot / (1<<MAGSHIFT))
#define DOTS_PER_DEG(ms)	((float) SEC_PER_DEG / SEC_PER_DOT(ms))
#define DEG_PER_DOT(ms)		((float) SEC_PER_DOT(ms) / SEC_PER_DEG)
#define DOTS_TO_DEG(ms,dots)	((float) (dots) * DEG_PER_DOT(ms))

#define U_TO_SEC(map,units)	((units) << (map)->uSecShift)
#define SEC_TO_U(map,sec)	((sec) >> (map)->uSecShift)
#define U_PER_DEG(map)		((map)->uPerDeg)
#define DEG_TO_U(map,deg)	SEC_TO_U(map,DEG_TO_SEC(deg))
#define U_TO_DEG(map,units)	SEC_TO_DEG(U_TO_SEC(map,units))
#define DEG180_U(map)		SEC_TO_U(map,DEG_TO_SEC(180))
/* WRAP_THRESH_U is used to suppress strokes crossing "behind" the map */
#define WRAP_THRESH_U(map)	SEC_TO_U(map,DEG_TO_SEC(180))
//#define U_TO_DOTS(units,muPerDot) ( ((units) << MAGSHIFT) / muPerDot )
#define U_TO_DOTS(units,muPerDot) ( ((float)((units) << MAGSHIFT)) / muPerDot )

/* projection table macros */
#define PROJ_SHIFT_VAL		11	// for 11 bit projection table values
#define PROJ_SHIFT_SEC		8	// 21 bit secs to 11 bit proj table index
#define PROJ_SHIFT_U(map)	(PROJ_SHIFT_SEC - (map)->uSecShift)
#define SEC_TO_PROJ_IN(sec)	( (sec) >> PROJ_SHIFT_SEC )
#define XPROJ_SEC(table, sec)	( (table)[SEC_TO_PROJ_IN(sec)] )
#define XPROJ_U(table,units,SHFT) ( (table)[(units) >> SHFT] )
#define XPROJ_MUL(val, xp)	( ((val) * (xp)) >> PROJ_SHIFT_VAL )
#define XPROJ_DIV(val, xp)	( ((val) << PROJ_SHIFT_VAL) / (xp) )

#define abs(val)		(val < 0 ? -(val) : val)

#define LATLONGS_PRINTARGS(lat, lng) \
    abs(lat), (lat < 0 ? "S" : "N"), abs(lng), (lng < 0 ? "W" : "E")

#define NORM_MODULO(value, halfMod) \
    if (value < -halfMod) value += halfMod+halfMod; \
    else if (value > halfMod) value -= halfMod+halfMod;

#define NORMALIZE_LONGITUDE(lng, centerLng, deg180) \
{   lng -= centerLng; \
    NORM_MODULO(lng, deg180); \
}


+ (CIAMap*)map
{
    return [[[CIAMap allocWithZone:[self zone]] init] autorelease];
}

- init
{
    ms = 0;
    region.r_left = region.r_right = region.r_top = 0;
    region.r_width = 1500;	// screen width
    region.r_height = 1500;
    mp.x = region.r_width / 2.0;
    mp.y = region.r_height / 2.0;
    //ll.x = ll.y = MAXCOORD;
    //ur.x = ur.y = -MAXCOORD;
    return self;
}

/* static variables
 */

#define MAX_TEXTURES	6

//static short *Textures[] = {NULL, pr_tex_dotted, pr_tex_dashed, pr_tex_dashdot, pr_tex_dashdotdotted, pr_tex_longdashed};
//static short *Textures[] = {NULL, 1, 2, 3, 4, 5};
static char		*Pgm = "Cenon Astro";
static unsigned short	*XProj, *ProjTables[MAX_PROJ];
static int		generation = 0, openCount = 0, maxOpen = -1;

/* map_error
 */
void map_error(format, a1, a2, a3, a4, a5, a6, a7, a8, a9, a10)
    char *format, *a1, *a2, *a3, *a4, *a5, *a6, *a7, *a8, *a9, *a10;
//void map_error(char *format, char *a1, char *a2, char *a3, char *a4, char *a5, char *a6, char *a7, char *a8, char *a9, char *a10)
{
    printf(format, a1, a2, a3, a4, a5, a6, a7, a8, a9, a10);
    exit(1);
} // map_error

/* GetToken
 */
static char *GetToken(char *str, char *errStr)
{   char *tok;

    if ((tok = strtok(str, TOK_SEPS)) != NULL)
        return(tok);
    fprintf(stderr, "%s: profile error: %s\n", Pgm, errStr);
    return(NULL);
}

/* AllocStr - allocate and copy a string
 */
static char *AllocStr(char *str)
{   char *newStr;

    // newStr = strdup(str);
    newStr = (char*)malloc(strlen(str)+1);
    if (newStr == NULL)
        map_error("%s: out of memory\n", Pgm);
    else
        strcpy(newStr, str);
    return (newStr);
}

/* OpenFile
 */
static FILE *OpenFile(MapSet *ms, char *dir, char *fname, char *type)
{   char *fullname;
    FILE *fp;

    if (dir == NULL || fname[0] == '/')
        fullname = fname;
    else
    {
	fullname = (char *)malloc(strlen(dir)+strlen(fname)+2);
	if (fullname == NULL) map_error("%s: out of memory\n", Pgm);
	sprintf(fullname, "%s/%s", dir, fname);
    }
    errno = 0;
    fp = fopen(fullname, type);
    if (fp == NULL)
    {
	if (ms->debug & D_FILES)
        {   perror(fullname);
	    fprintf(stderr, "%s: error %d opening %s\n", Pgm, errno, fullname);
	}
	return (NULL);
    }
    if (fullname != fname) free(fullname);
    return (fp);
} // OpenFile


/* CloseFile
 */
void CloseFile(MapSet *ms, MapFile *mFile)
{
    if (mFile->file == NULL)
        return;
    if (ms->debug & D_FILES)
        fprintf(stderr, "closing file %s (%d/%d open)\n", mFile->name, openCount, maxOpen);
    //free(mFile->dir); // ms->mapDir
    free(mFile->name);
    fclose(mFile->file);
    mFile->file = NULL;		/* indicates file is closed */
}

/* CloseOldFile
 */
static void CloseOldFile(MapSet *ms)
{   MapFile	*oldM = NULL;
    int		i, oldG = generation;

    for (i = ms->mCount-1; i >= 0; --i)
    {
	if (ms->maps[i].mfile->file != NULL && ms->maps[i].mfile->used < oldG)
        {
	    oldG = ms->maps[i].mfile->used;
	    oldM = ms->maps[i].mfile;
	}
    }
    if (oldG == generation)
	map_error("%s: error: cannot find a file to close\n", Pgm);
    CloseFile(ms, oldM);
}

/* OpenMapFile
 */
static void OpenMapFile(MapSet *ms, MapFile *mFile)
{
    mFile->used = generation;
    if (mFile->file != NULL)
        return;

    while (mFile->file == NULL)
    {
	if (mFile->file == NULL && (maxOpen == -1 || openCount < maxOpen))
        {
	    mFile->file = OpenFile(ms, mFile->dir, mFile->name, "rb");
	    if (mFile->file != NULL)
            {
		++openCount;
		if (mFile->header.segcount == -1)
                {
		    GETLONG(mFile->header.magic, mFile->file);
		    GETLONG(mFile->header.dictaddr, mFile->file);
		    GETLONG(mFile->header.segcount, mFile->file);
		    GETLONG(mFile->header.segsize, mFile->file);
		    GETLONG(mFile->header.segmax, mFile->file);
		    if (mFile->header.segcount * sizeof (struct segdict) !=
				mFile->header.segsize)
                    {
			if (ferror(mFile->file))
                        {
			    perror(mFile->name);
			    if (errno == EBADF)
                            {
				/* apparently too many open files, even though fopen() succeeded! */
				maxOpen = openCount - FILE_RESERVE;
				fprintf(stderr, "%s: setting -map_maxopen %d\n", Pgm, maxOpen);
				CloseFile(ms, mFile);
				--openCount;
				mFile->header.segcount = -1;
				continue;
			    }
			}
			map_error("%s: incompatible file format: %s\n", Pgm, mFile->name);
		    }
		    if (ms->max_segcount < mFile->header.segcount)
			ms->max_segcount = mFile->header.segcount;
		    if (mFile->header.magic == CBD_MAGIC2)
                    {
			GETLONG(mFile->header.maxlat, mFile->file);
			GETLONG(mFile->header.minlat, mFile->file);
			GETLONG(mFile->header.maxlong, mFile->file);
			GETLONG(mFile->header.minlong, mFile->file);
			GETLONG(mFile->header.features, mFile->file);
			GETLONG(mFile->header.scale_shift, mFile->file);
			GETLONG(mFile->header.lat_offset, mFile->file);
			GETLONG(mFile->header.lng_offset, mFile->file);
		    }
		}
		if (ms->debug & D_FILES)
		    fprintf(stderr, "opened %s (%d/%d open)\n", mFile->name, openCount, maxOpen);
		return;
	    }
	    if (errno != EMFILE)
            {
		perror(mFile->name);
		map_error("%s: error %d opening %s\n", Pgm, errno, mFile->name);
	    }
	}
	/* try closing another open file */
	CloseOldFile(ms);
	--openCount;
	if (maxOpen == -1)
            maxOpen = openCount - FILE_RESERVE;
    }
}


/* InitMapFile
 */
static MapFile *InitMapFile(MapSet *ms, char *dir, char *name)
{   MapFile	*mFile;
    int		p;

    if (filePool == NULL)
	filePool = (MapFile **)calloc(ms->maxMaps, sizeof(MapFile *));
    if (filePool == NULL)
	map_error("%s: out of memory\n", Pgm);

    /* check pool for matching file */
    for (p = 0; p < poolSize; ++p)
    {
	mFile = filePool[p];
	if (!strcmp(mFile->name, name) && !strcmp(mFile->dir, dir))
        {
	    if (ms->debug & D_FILES)
	        fprintf(stderr, "found %s as pool file %d\n", name, p);
	    return(mFile);
	}
    }
    /* allocate new MapFile and add it to the pool */
    mFile = (MapFile *)calloc(1, sizeof(MapFile));	/* fill with zeros */
    mFile->dir = dir;
    mFile->name = name;
    mFile->used = generation;		/* LRU generation */
    mFile->file = NULL;			/* not open */
    mFile->header.dictaddr = 0;		/* not yet read */
    if (poolSize < ms->maxMaps)
	filePool[poolSize++] = mFile;
    return(mFile);
}

/* map_read_profile
 */
void map_read_profile(MapSet *ms, char *dirName, char *fileName)
{   FILE	*fp;
    char	line[MAX_LINELEN];

    if (!strcmp(fileName, FNAME_STD))
	fp = stdin;
    else
	fp = OpenFile(ms, dirName, fileName, "r");
    if (fp == NULL)
	map_error("unable to open file: %s\n", fileName);
    if (ms->debug & D_VERBOSE)
    {
	fprintf(stderr, "reading profile %s\n", fileName);
	fflush(stderr);
    }
    while (fscanf(fp, " %[^\n]\n", line) == 1)
	map_read_profile_entry(ms, line);
    fclose(fp);
} // end map_read_profile

/* map_read_profile_entry
 */
void map_read_profile_entry(MapSet *ms, char *line)
{   char	*option;
    int		i;

    if (line[0] == '#')
        return;
    option = GetToken(line, "option expected");
    if (!strcmp(option, "-mapdir"))
    {
	ms->mapDir = AllocStr(GetToken(NULL, option));
    }
    else if (!strcmp(option, "-resolution"))
    {
	char *scaleName = GetToken(NULL, option);
	for (ms->scaleI = 0; ms->scaleI < ms->scaleCount; ++ms->scaleI)
	    if (!strcmp(scaleName, ms->scaleNames[ms->scaleI]))
            {
		return;
	    }
	if (ms->scaleCount >= ms->maxScales)
            map_error("%s: too many scales, increase -map_maxscales (currently %d)\n",
		Pgm, ms->maxScales);
	else
        {
	    ms->scaleI = ms->scaleCount++;
	    ms->scaleNames[ms->scaleI] = AllocStr(scaleName);
	    ms->autoRes[ms->scaleI] = ms->minRes[ms->scaleI] = 0.0;
	    sscanf(GetToken(NULL, option), "%f", &ms->autoRes[ms->scaleI]);
	    sscanf(GetToken(NULL, option), "%f", &ms->minRes[ms->scaleI]);
	    if (ms->scaleI == 0 || ms->autoRes[ms->scaleI] < ms->minAuto)
            {
		ms->minAuto = ms->autoRes[ms->scaleI];
		ms->minAutoI = ms->scaleI;
	    }
	    if (ms->debug & D_PROFILE)
            {   fprintf(stderr, "autoRes %g minRes %g, min: %s %g\n",
			ms->autoRes[ms->scaleI], ms->minRes[ms->scaleI],
			ms->scaleNames[ms->minAutoI], ms->minAuto);
	    } // endif debug
	}
    }
    else if (!strcmp(option, "-type"))
    {
	char *typeName = GetToken(NULL, option);
	for (i = 0; i < ms->mTypeCount; ++i)
	    if (!strcmp(typeName, ms->mTypes[i]))
            {
		ms->mapType = i;
		return;
	    }
	if (ms->mTypeCount >= ms->maxTypes)
	    map_error("%s: too many map types, increase -map_maxtypes (currently %d)\n",
		Pgm, ms->maxTypes);
	else
	    ms->mTypes[ms->mapType = ms->mTypeCount++] = AllocStr(typeName);
    }
    else if (!strcmp(option, "-min_features"))
    {
        ms->min_featureMasks[ms->mapType] =
        strtol(GetToken(NULL, option), NULL, 2);
    }
    else if (!strcmp(option, "-suppress"))
    {
	if (ms->suppressMasks[ms->mapType] == NULL)
	    ms->suppressMasks[ms->mapType] = (long *)calloc(ms->maxScales, sizeof(long));
	ms->suppressMasks[ms->mapType][ms->scaleI] = strtol(GetToken(NULL, option), NULL, 2);
	ms->enforce_suppression[ms->mapType] = YES;
    }
    else if (!strcmp(option, "-optional"))
    {
	ms->optional[ms->mapType] = YES;
	ms->enableMasks[ms->mapType] = 0;	// disable by default
    }
    else if (!strcmp(option, "-color"))
    {
	int cnum;
	char *tok;
	cnum = atoi(GetToken(NULL, option)) & 0xFF;
	if (cnum >= ms->colors.length) ms->colors.length = cnum+1;
	tok = GetToken(NULL, option);
	if (!strcmp(tok, "dotted")) ms->patterns[cnum] = 1;
	else if (!strcmp(tok, "dashed")) ms->patterns[cnum] = 2;
	else if (!strcmp(tok, "dashdot")) ms->patterns[cnum] = 3;
	else if (!strcmp(tok, "dashdotdotted")) ms->patterns[cnum] = 4;
	else if (!strcmp(tok, "longdashed")) ms->patterns[cnum] = 5;
	else ms->patterns[cnum] = 0;
	ms->useColorPat[cnum] = !strcmp(GetToken(NULL, option), "and");
	ms->colors.map[0][cnum] = strtol(GetToken(NULL, option), NULL, 0);
	ms->colors.map[1][cnum] = strtol(GetToken(NULL, option), NULL, 0);
	ms->colors.map[2][cnum] = strtol(GetToken(NULL, option), NULL, 0);
    }
    else if (!strcmp(option, "-grid"))
	ms->gridColor = atoi(GetToken(NULL, option));
    else if (!strcmp(option, "-background"))
	ms->bgColor = atoi(GetToken(NULL, option));
    else if (!strcmp(option, "-highlight"))
	ms->highlight_color = atoi(GetToken(NULL, option));
    else if (!strcmp(option, "-fatten_scale"))
    {
	 float fatscale;
	sscanf(GetToken(NULL, option), "%f", &fatscale);
	ms->fatten_thresh = (long) ((1 << MAGSHIFT) / fatscale);
    }
    else if (!strcmp(option, "-feature"))
    {
	int featureNum = strtol(GetToken(NULL, option), NULL, 0);
	if (ms->features[ms->mapType] == NULL)
	    ms->features[ms->mapType] = (MapFeature *)calloc(MAX_FEATURE+1, sizeof(MapFeature));
	ms->features[ms->mapType][featureNum].rank = featureNum;
	ms->features[ms->mapType][featureNum].color = atoi(GetToken(NULL, option));
	ms->features[ms->mapType][featureNum].line_width = atoi(GetToken(NULL, option));
	ms->features[ms->mapType][featureNum].name = AllocStr(strtok(NULL, "\n"));
	if (ms->maxFeature[ms->mapType] < featureNum)
	    ms->maxFeature[ms->mapType] = featureNum;
    }
    else if (!strcmp(option, "-map"))
    {   MapStruct *map;
	char *str;

	if (ms->mCount >= ms->maxMaps)
	    map_error("%s: too many maps, increase -map_maxmaps (currently %d)\n",
		Pgm, ms->maxMaps);
	map = &ms->maps[ms->mCount];
	if (ms->mapType == -1)
	    ms->mTypes[ms->mapType = ms->mTypeCount++] = DEF_MAPTYPE;
	map->type = ms->mapType;
	map->scaleI = ms->scaleI;
	
	map->mfile = InitMapFile(ms, ms->mapDir,
		AllocStr(GetToken(NULL, option)));
	str = strtok(NULL, TOK_SEPS);
	if (str != NULL) map->rank_offset = atoi(str);
	map->uPerDeg = -1;			// not-loaded flag
	map->mfile->header.segcount = -1;	// not-read flag
	++ms->mCount;
    }
    else if (!strcmp(option, "-readprofile"))
	map_read_profile(ms, ms->mapDir, GetToken(NULL, option));
    else if (!strcmp(option, "-ll_center"))
    {
	sscanf(GetToken(NULL, option), "%f", &ms->def_centerLng);
	sscanf(GetToken(NULL, option), "%f", &ms->def_centerLat);
    }
    else if (!strcmp(option, "-ll_extent"))
    {
	sscanf(GetToken(NULL, option), "%f", &ms->def_showLng);
	sscanf(GetToken(NULL, option), "%f", &ms->def_showLat);
    }
} // end map_read_profile_entry


/* map_load	
 */
void map_load(MapSet *ms, int mapIndex)
{   MapStruct	*map = &ms->maps[mapIndex];

    ++generation;
    OpenMapFile(ms, map->mfile);

    map->uSecShift = -(map->mfile->header.scale_shift);
    map->uPerDeg = SEC_PER_DEG << map->uSecShift;
    if (map->mfile->header.magic != CBD_MAGIC2)
	map_error("%s: cannot read map format: %s\n", map->mfile->name);
    map->minSSec = U_TO_SEC(map, map->mfile->header.minlat);
    map->maxNSec = U_TO_SEC(map, map->mfile->header.maxlat);
    map->minWSec = U_TO_SEC(map, map->mfile->header.minlong);
    map->maxESec = U_TO_SEC(map, map->mfile->header.maxlong);
    map->minLng = U_TO_DEG(map,map->minWSec); /////////////////////// sec not units ???
    map->maxLng = U_TO_DEG(map,map->maxESec);
    map->minLat = U_TO_DEG(map,map->minSSec);
    map->maxLat = U_TO_DEG(map,map->maxNSec);
    if (map->maxNSec > ms->maxLatSec) ms->maxLatSec = map->maxNSec;
    if (map->minSSec < ms->minLatSec) ms->minLatSec = map->minSSec;
    if (map->maxESec > ms->maxLngSec) ms->maxLngSec = map->maxESec;
    if (map->minWSec < ms->minLngSec) ms->minLngSec = map->minWSec;
    if (ms->debug & D_READMAP)
    {   int i;

	fprintf(stderr,
	"%s contains %d blocks covering (%.2f%s %.2f%s) to (%.2f%s %.2f%s) features:",
		map->mfile->name, (int)map->mfile->header.segcount,
		LATLONGS_PRINTARGS(map->minLat, map->minLng),
		LATLONGS_PRINTARGS(map->maxLat, map->maxLng));
	for (i = 1; i <= 32; i++)
	    if (map->mfile->header.features & (1<<(i-1))) fprintf(stderr, " %d", i);
	fprintf(stderr, "\n");
    } // endif debug
}

/* map_load_all
 */
void map_load_all(MapSet *ms)
{   int i;

    ms->minLatSec = DEG90_SECS;
    ms->maxLatSec = -DEG90_SECS;
    ms->minLngSec = DEG180_SECS;
    ms->maxLngSec = -DEG180_SECS;

    for (i = 0; i < ms->mCount; ++i)
	map_load(ms, i);
}

/* rescale - rescale to show an area of size (showLng, showLat)
 * centered at (centerLng, centerLat) in (width x height) pixels.
 */
static void map_rescale(MapSet *ms, int width, int height)
{   float	canvasRatio, mapRatio, halfLat, halfLng, degMargin;
    long	halfHtSec, canvasSSec, canvasESec, canvasWSec;

    /* figure scale using inset width and height */
    width -= ms->margin * 2;
    height -= ms->margin * 2;

    /* choose new msecPerDot and adjust showLng */
    mapRatio = (float) (ms->maxLatSec - ms->minLatSec) / (ms->maxLngSec - ms->minLngSec);
    canvasRatio = (float) height / width;
    if (canvasRatio >= mapRatio)	// fit width
    {
	if (ms->showLng > ms->maxLng - ms->minLng)
            ms->showLng = ms->maxLng - ms->minLng;
	ms->msecPerDot = (long)DEG_TO_SEC(ms->showLng) << MAGSHIFT;
	ms->msecPerDot = ((ms->msecPerDot-1) / width) + 1;
    }
    else				// fit height
    {
	if (ms->showLat > ms->maxLat - ms->minLat)
            ms->showLat = ms->maxLat - ms->minLat;
	ms->msecPerDot = (long)DEG_TO_SEC(ms->showLat) << MAGSHIFT;
	ms->msecPerDot = ((ms->msecPerDot-1) / height) + 1;
    }

    if ((1 << MAGSHIFT) / ms->msecPerDot > MAX_MAGNIFY)
	ms->msecPerDot = (1 << MAGSHIFT) / MAX_MAGNIFY;
    if (ms->msecPerDot < 1)
        ms->msecPerDot = 1;

    ms->showLng = DOTS_TO_DEG(ms,width);
    ms->showLat = DOTS_TO_DEG(ms,height);

    if (ms->showLat > ms->maxLat - ms->minLat)
	ms->showLat = ms->maxLat - ms->minLat;
    if (ms->showLng > ms->maxLng - ms->minLng)
	ms->showLng = ms->maxLng - ms->minLng;

    /* apply margin only if at edge of map */
    degMargin = DOTS_TO_DEG(ms, ms->margin) / 2;
    width += ms->margin * 2;
    height += ms->margin * 2;
    ms->halfWidSec = DOTS_TO_SEC(ms, width / 2);

    halfHtSec = DOTS_TO_SEC(ms,height / 2);
    halfLng = SEC_TO_DEG(ms->halfWidSec);
    halfLat = SEC_TO_DEG(halfHtSec);

    if (halfLat * 2 >  ms->maxLat - ms->minLat)
	ms->centerLat = (ms->minLat + ms->maxLat) / 2;		/* center it */
    else if (ms->centerLat < ms->minLat + halfLat - degMargin)
	ms->centerLat = ms->minLat + halfLat - degMargin;	/* off bottom */
    else if (ms->centerLat > ms->maxLat - halfLat + degMargin)
	ms->centerLat = ms->maxLat - halfLat + degMargin;	/* off top */

    ms->cntrLngSec = (long)DEG_TO_SEC(ms->centerLng);
    ms->canvasNSec = (long)DEG_TO_SEC(ms->centerLat) + halfHtSec;
    canvasSSec = ms->canvasNSec - DOTS_TO_SEC(ms,height);

    canvasWSec = (long)DEG_TO_SEC(ms->centerLng) - ms->halfWidSec;
    canvasESec = (long)DEG_TO_SEC(ms->centerLng) + ms->halfWidSec;

    if (ms->debug & D_RESCALE)
    {
	fprintf(stderr, "%d x %d canvas, %g seconds/dot\n", width, height, (float) DOTS_TO_SEC(ms,1));
	fprintf(stderr, "center (%g,%g), showLng %g, (%d,%d) to (%d,%d)\n",
		ms->centerLng, ms->centerLat, ms->showLng, (int)canvasWSec, (int)ms->canvasNSec,
		(int)canvasESec, (int)canvasSSec);
    } // endif debug
    if (ms->debug & D_VERBOSE)
    {
	fprintf(stderr, "1 deg = %.4g dots, frame height = %d miles\n",
		(float) DOTS_PER_DEG(ms), (int) LAT_TO_MILES(halfLat*2));
	fprintf(stderr,
		"(%.2f%s %.2f%s) to (%.2f%s %.2f%s) center (%.2f%s %.2f%s)\n",
		LATLONGS_PRINTARGS(ms->centerLat + halfLat, ms->centerLng - halfLng),
		LATLONGS_PRINTARGS(ms->centerLat - halfLat, ms->centerLng + halfLng),
		LATLONGS_PRINTARGS(ms->centerLat, ms->centerLng));
	fprintf(stderr, "extent (%.2f by %.2f degrees)\n", halfLat*2, halfLng*2);
    }
} // map_rescale

/* y in seconds ! -> return degree !
 */
/*long computeMercatorYBack(long y)
{   float	yrad, sinSqrE2, yfloat, e = 2.718281828;

    // formular: yfloat = ln ( tan (pi/4 + L/2) * ( (1 - e * sin (L)) / (1 + e * sin (L))) ** (e/2) )
    yfloat = SEC_TO_DEG(y);
    yfloat *= OneDegInM;
    yfloat /= EarthRadiusA;

    yfloat = pow(e, yfloat); // log
    yfloat / sinSqrE2 = tan(Pi/4.0 + yrad/2.0);
    yfloat = RadToDeg(yrad);

// -------------------
    yrad = DegToRad(SEC_TO_DEG(y));

    sinSqrE2 = ( (1.0 - Eccentricity * sin(yrad)) / (1 + Eccentricity * sin(yrad)) );
    if ( sinSqrE2 <= 0.0 )
        return 0.0;
    sinSqrE2 = pow(sinSqrE2, Eccentricity/2.0);
    yfloat = log( tan(Pi/4.0 + yrad/2.0) * sinSqrE2 );

    yfloat *= EarthRadiusA;
    yfloat /= OneDegInM;

    return DEG_TO_SEC(yfloat);
}*/
float computeMercatorYBack(long y)
{   float	yrad, yfloat, e = 2.718281828;

    /* formular: yfloat = r * ln ( tan (pi/4 + L/2) * ( (1 - e * sin (L)) / (1 + e * sin (L))) ** (e/2) )
     */

    yfloat = SEC_TO_DEG(y);
    yfloat *= OneDegInM;
    yfloat /= EarthRadiusA;

    yfloat = pow(e, yfloat); // log -> e pow yfloat = ex log

    // yfloat = tan(Pi/4.0 + yrad/2.0);
    yrad = (atan(yfloat) - Pi/4.0) * 2.0;

    return RadToDeg(yrad);
}
/* y in seconds ! -> return seconds ! y is latitude
 */
/*long computeMercatorY(long y)
{   float	yrad, sinSqrE2, yfloat;

    // formular: yfloat = ln ( tan (pi/4 + L/2) * ( (1 - e * sin (L)) / (1 + e * sin (L))) ** (e/2)  )

    if ( Abs(y) >= 90*3600 )
        y = (y > 0) ? (89.9*3600) : (-89.9*3600);
    yrad = DegToRad(SEC_TO_DEG(y));

    sinSqrE2 = ( (1.0 - Eccentricity * sin(yrad)) / (1 + Eccentricity * sin(yrad)) );
    if ( sinSqrE2 <= 0.0 )
        return 0.0;
    sinSqrE2 = pow(sinSqrE2, Eccentricity/2.0);
    yfloat = log( tan(Pi/4.0 + yrad/2.0) * sinSqrE2 );

    yfloat *= EarthRadiusA;
    yfloat /= OneDegInM;

    return DEG_TO_SEC(yfloat);
}*/
long computeMercatorY(long y)
{   float	yrad, tangens, yfloat;

    /* formular: yfloat = r * ln ( tan (pi/4 + L/2) )
     */

    if ( Abs(y) >= 90*3600 )
        y = (y > 0) ? (89.99*3600) : (-89.99*3600);

    yrad = DegToRad(SEC_TO_DEG(y));
    tangens = tan(Pi/4.0 + yrad/2.0);
    yfloat = log( tangens );

    yfloat *= EarthRadiusA;
    yfloat /= OneDegInM;

    return DEG_TO_SEC(yfloat);
}

/* return point on map for given latitude and longitude
 */
NSPoint computeAzimut(float lat, float lon, MapSet *ms)
// - (NSPoint)pointOnMapForLat:(float)lat lon:(float)lon
{   NSPoint	center, p;
    double	deg, d, b, rad;
    /* stereographic map */
    double x = ms->centerLng;		// x center lon
    double y = ms->centerLat;		// y center lat
    double deg0 = -44.0;		// angle of 0 meridian on map
    //double scale = 40550000.0;	// scale of map weather map !

    /* northern hemisphere */
    center = NSMakePoint(x, y);
    //deg0 = [mapDict floatForKey:@"deg0"];

    /* get angle to rotate 0 meridian */
    deg = lon + deg0;

    /* radius of latitude = stereographic (azimut) projection */
    //d = 6356752.0 * 2.0 * 72000.0 / 25.0;	// earth diameter in point
    d = 6356752.0 * 2.0;	// earth diameter
    //scale = [mapDict doubleForKey:@"scale"];
    b = (90.0 - lat) / 2.0;
    //rad = Tan(b) * d / scale;
    rad = Tan(b) * d;

    p = vhfPointRotatedAroundCenter(NSMakePoint(center.x+rad, center.y), deg, center);
    p.x /= OneDegInM;
    p.y /= OneDegInM;
    return p; // in degree
}

/*
 * drawDestinyLines
 */
- (void) drawDestinyLines:(long)offsetESec :(long)offsetNSec
                         :(long)clipWSec :(long)clipNSec :(long)clipESec :(long) clipSSec :(SegList*)seglist
{   NSDictionary	*destinyDict = [NSDictionary dictionaryWithContentsOfFile:[NSString stringWithFormat:@"%@/MapLines.dict", userLibrary()]], *curDict;
    NSArray		*keyArray;
    struct pr_pos	*ptlist = seglist->ptlist;
    unsigned char	*mvlist = seglist->mvlist;
    long		y_inc, ux0, uy, dxN, dxS, dyN, dyS, cnt, uyn; // 1.34575 2.012
    int			nPts = 0, latSpace = DEG_TO_SEC(360.0/22.0/12.0);
    int			lonSpace = DEG_TO_SEC(360.0/15.0/12.0);
    float		deg, yf, x0f, x1f, zodiacDegLat = (360.0/22.0/12.0)/30.0;
    float		zodiacDegLng = (360.0/15.0/12.0)/30.0;
    int			uyStartLat = DEG_TO_SEC(48.0), uyStartLng = DEG_TO_SEC(12.0245098 - ms->centerLng);
    int			keyCnt=0, uyDiff, uxDiff, uyCur, i;

    if ( ms->proj != PROJ_MERCATOR || !destinyDict )
        return;

    seglist->op = 23; // red

    /* 48.0 Grad Nord latitude
       12.0245098 Grad Ost longitude

       - in der Schraege 0 Grad Widder (Loxodrome - Kurslinie der Schiffe) (23.4333)
       - in der Breite   0 Grad Loewe

       15 Tierkreisgrade = 1 Grad geographisch
       (kleines Gitter 50 m 1 Grad)
     */

    ms->gridDegrees = 0;

    // ost - west
    y_inc = DEG_TO_SEC(1);				// smaller vertical increment
    x0f = SEC_TO_DOTS(ms,clipWSec + offsetESec);	// constant parallel length
    x1f = SEC_TO_DOTS(ms,clipESec + offsetESec);

    for (uy = uyStartLat; uy < clipNSec; uy += latSpace)
    // uy = DEG_TO_SEC(startLat); // 48.0
    {
        uyDiff = Diff(uy, uyStartLat) / latSpace;
        while ( uyDiff >= 12 ) // from 0 - 11
            uyDiff -= 12.0;
        switch (uyDiff)
        {
            case 0: curDict = [destinyDict objectForKey:@"loewe"]; break;
            case 1: curDict = [destinyDict objectForKey:@"jungfrau"]; break;
            case 2: curDict = [destinyDict objectForKey:@"waage"]; break;
            case 3: curDict = [destinyDict objectForKey:@"skorpion"]; break;
            case 4: curDict = [destinyDict objectForKey:@"schuetze"]; break;
            case 5: curDict = [destinyDict objectForKey:@"steinbock"]; break;
            case 6: curDict = [destinyDict objectForKey:@"wassermann"]; break;
            case 7: curDict = [destinyDict objectForKey:@"fisch"]; break;
            case 8: curDict = [destinyDict objectForKey:@"widder"]; break;
            case 9: curDict = [destinyDict objectForKey:@"stier"]; break;
            case 10: curDict = [destinyDict objectForKey:@"zwillinge"]; break;
            case 11: curDict = [destinyDict objectForKey:@"krebs"]; break;
            default: // ??
                continue;
        }
        keyArray = [curDict allKeys];
        keyCnt = [keyArray count];
        for (i=0; i<keyCnt; i++)
        {
            deg = [[keyArray objectAtIndex:i] floatValue] * zodiacDegLat;
            uyCur = uy + DEG_TO_SEC(deg);

            uyn = computeMercatorY(uyCur);
            yf = SEC_TO_DOTS(ms,offsetNSec - uyn);
            ptlist[nPts].x = x0f;
            ptlist[nPts].y = yf;
            mvlist[nPts++] = 1;
            ptlist[nPts].x = x1f;
            ptlist[nPts].y = yf;
            mvlist[nPts++] = 0;
            [self addSegments:seglist :nPts];
            nPts = 0;
        }
    }
    for (uy = uyStartLat-latSpace; uy >= clipSSec; uy -= latSpace)
    {
        uyDiff = Diff(uy, uyStartLat) / latSpace;
        while ( uyDiff >= 12 ) // from 0 - 11
            uyDiff -= 12.0;
        switch (uyDiff)
        {
            case 0: curDict = [destinyDict objectForKey:@"loewe"]; break;
            case 1: curDict = [destinyDict objectForKey:@"krebs"]; break;
            case 2: curDict = [destinyDict objectForKey:@"zwillinge"]; break;
            case 3: curDict = [destinyDict objectForKey:@"stier"]; break;
            case 4: curDict = [destinyDict objectForKey:@"widder"]; break;
            case 5: curDict = [destinyDict objectForKey:@"fisch"]; break;
            case 6: curDict = [destinyDict objectForKey:@"wassermann"]; break;
            case 7: curDict = [destinyDict objectForKey:@"steinbock"]; break;
            case 8: curDict = [destinyDict objectForKey:@"schuetze"]; break;
            case 9: curDict = [destinyDict objectForKey:@"skorpion"]; break;
            case 10: curDict = [destinyDict objectForKey:@"waage"]; break;
            case 11: curDict = [destinyDict objectForKey:@"jungfrau"]; break;
            default: // ??
                continue;
        }
        keyArray = [curDict allKeys];
        keyCnt = [keyArray count];
        for (i=0; i<keyCnt; i++)
        {
            deg = [[keyArray objectAtIndex:i] floatValue] * zodiacDegLat;
            uyCur = uy + DEG_TO_SEC(deg);

            uyn = computeMercatorY(uyCur);
            yf = SEC_TO_DOTS(ms,offsetNSec - uyn);
            ptlist[nPts].x = x0f;
            ptlist[nPts].y = yf;
            mvlist[nPts++] = 1;
            ptlist[nPts].x = x1f;
            ptlist[nPts].y = yf;
            mvlist[nPts++] = 0;
            [self addSegments:seglist :nPts];
            nPts = 0;
        }
    }

    seglist->op = 24; // cyan

    // draw N-S meridians - lines with 23.4333 degree from N-S meridians
    uy = uyStartLat; // allways our start y
    uyn = computeMercatorY(uy);
    dyN = Diff(uy, computeMercatorY(clipNSec));
    cnt = dyN / y_inc; // 1 deg 3600 sec
    dyN = cnt * 3600;
    dxN = cnt * DEG_TO_SEC(Tan(23.4333));

    dyS = Diff(uy, computeMercatorY((clipSSec < 0) ? (clipSSec) : (clipSSec+(60.0*3600))));
    //dyS = Diff(uy, computeMercatorY(clipSSec));
    cnt = dyS / y_inc; // 1 deg 3600 sec
    dyS = cnt * 3600;
    dxS = cnt * DEG_TO_SEC(Tan(23.4333));

    for (ux0 = uyStartLng; ux0 <= clipESec+(20.0*3600); ux0 += lonSpace) //  -> um centerLng nach rechts verschoben
    {
        uxDiff = Diff(ux0, uyStartLng) / lonSpace;
        while ( uxDiff >= 12 ) // from 0 - 11
            uxDiff -= 12.0;
        switch (uxDiff)
        {
            case 0: curDict = [destinyDict objectForKey:@"widder"]; break;
            case 1: curDict = [destinyDict objectForKey:@"stier"]; break;
            case 2: curDict = [destinyDict objectForKey:@"zwillinge"]; break;
            case 3: curDict = [destinyDict objectForKey:@"krebs"]; break;
            case 4: curDict = [destinyDict objectForKey:@"loewe"]; break;
            case 5: curDict = [destinyDict objectForKey:@"jungfrau"]; break;
            case 6: curDict = [destinyDict objectForKey:@"waage"]; break;
            case 7: curDict = [destinyDict objectForKey:@"skorpion"]; break;
            case 8: curDict = [destinyDict objectForKey:@"schuetze"]; break;
            case 9: curDict = [destinyDict objectForKey:@"steinbock"]; break;
            case 10: curDict = [destinyDict objectForKey:@"wassermann"]; break;
            case 11: curDict = [destinyDict objectForKey:@"fisch"]; break;
            default: // ??
                continue;
        }
        keyArray = [curDict allKeys];
        keyCnt = [keyArray count];
        for (i=0; i<keyCnt; i++)
        {
            deg = [[keyArray objectAtIndex:i] floatValue] * zodiacDegLng;

            ptlist[nPts].x = SEC_TO_DOTS(ms,ux0 + DEG_TO_SEC(deg) + dxS + offsetESec); // mirrored later N-S changed
            ptlist[nPts].y = SEC_TO_DOTS(ms,offsetNSec - uyn + dyS);
            mvlist[nPts++] = 1;

            ptlist[nPts].x = SEC_TO_DOTS(ms,ux0 + DEG_TO_SEC(deg) - dxN + offsetESec);
            ptlist[nPts].y = SEC_TO_DOTS(ms,offsetNSec - uyn - dyN);
            mvlist[nPts++] = 0;
            [self addSegments:seglist :nPts];
            nPts = 0;
        }
    }

    for (ux0 = uyStartLng-lonSpace; ux0 >= clipWSec-(20.0*3600); ux0 -= lonSpace)
    {
        uxDiff = Diff(ux0, uyStartLng) / lonSpace;
        while ( uxDiff >= 12 ) // from 0 - 11
            uxDiff -= 12.0;
        switch (uxDiff)
        {
            case 0: curDict = [destinyDict objectForKey:@"fisch"]; break;
            case 1: curDict = [destinyDict objectForKey:@"wassermann"]; break;
            case 2: curDict = [destinyDict objectForKey:@"steinbock"]; break;
            case 3: curDict = [destinyDict objectForKey:@"schuetze"]; break;
            case 4: curDict = [destinyDict objectForKey:@"skorpion"]; break;
            case 5: curDict = [destinyDict objectForKey:@"waage"]; break;
            case 6: curDict = [destinyDict objectForKey:@"jungfrau"]; break;
            case 7: curDict = [destinyDict objectForKey:@"loewe"]; break;
            case 8: curDict = [destinyDict objectForKey:@"krebs"]; break;
            case 9: curDict = [destinyDict objectForKey:@"zwillinge"]; break;
            case 10: curDict = [destinyDict objectForKey:@"stier"]; break;
            case 11: curDict = [destinyDict objectForKey:@"widder"]; break;
            default: // ??
                continue;
        }
        keyArray = [curDict allKeys];
        keyCnt = [keyArray count];
        for (i=0; i<keyCnt; i++)
        {
            deg = [[keyArray objectAtIndex:i] floatValue] * zodiacDegLng;

            ptlist[nPts].x = SEC_TO_DOTS(ms,ux0 + DEG_TO_SEC(deg) + dxS + offsetESec); // mirrored later N-S changed
            ptlist[nPts].y = SEC_TO_DOTS(ms,offsetNSec - uyn + dyS);
            mvlist[nPts++] = 1;

            ptlist[nPts].x = SEC_TO_DOTS(ms,ux0 + DEG_TO_SEC(deg) - dxN + offsetESec);
            ptlist[nPts].y = SEC_TO_DOTS(ms,offsetNSec - uyn - dyN);
            mvlist[nPts++] = 0;
            [self addSegments:seglist :nPts];
            nPts = 0;
        }
    }
}

/*
 * drawZodiacLines
 */
- (void) drawZodiacLines:(long)offsetESec :(long)offsetNSec
                        :(long)clipWSec :(long)clipNSec :(long)clipESec :(long)clipSSec :(SegList*)seglist
{   struct pr_pos *ptlist = seglist->ptlist;
    unsigned char	*mvlist = seglist->mvlist;
    long		y_inc, ux0, uy, dxN, dxS, dyN, dyS, cnt, uyn; // 2.012
    int 		nPts = 0, latSpace = DEG_TO_SEC(360.0/22.0/12.0);
    int			lonSpace = DEG_TO_SEC(360.0/15.0/12.0); // 1.34246  1.9558823
    float		yf, x0f, x1f, startLng = 12.0245098, startLat = 48.0; // 12.0245098

    if ( ms->proj != PROJ_MERCATOR )
        return;

    seglist->op = 11; // 11 - widder 12 - stier ...

    /* 48.0 Grad Nord latitude
       12.0 Grad Ost longitude

       - in der Schraege 0 Grad Widder (Loxodrome - Kurslinie der Schiffe) (23.4333)
       - in der Breite   0 Grad Loewe

       15 Tierkreisgrade = 1 Grad geographisch
       (kleines Gitter 50 m 1 Grad)
    */
    ms->gridDegrees = 0;

    // ost - west
    y_inc = DEG_TO_SEC(1);				// smaller vertical increment
    x0f = SEC_TO_DOTS(ms,clipWSec + offsetESec);	// constant parallel length
    x1f = SEC_TO_DOTS(ms,clipESec + offsetESec);
    seglist->op = 15; // 11 - widder 12 - stier ... 15 - loewe

    for (uy = DEG_TO_SEC(startLat); uy < clipNSec; uy += latSpace)
    // uy = DEG_TO_SEC(startLat); // 48.0
    {
        if ( uy == DEG_TO_SEC(startLat) )
            seglist->op = 15;
        else if ( seglist->op + 1 > 22 )
            seglist->op = 11;
        else
            seglist->op += 1;

	/* draw parallel at uy */
        uyn = computeMercatorY(uy);
        yf = SEC_TO_DOTS(ms,offsetNSec - uyn);
	ptlist[nPts].x = x0f;
	ptlist[nPts].y = yf;
	mvlist[nPts++] = 1;
	ptlist[nPts].x = x1f;
	ptlist[nPts].y = yf;
	mvlist[nPts++] = 0;
        [self addSegments:seglist :nPts];
        nPts = 0;
    }
    for (uy = DEG_TO_SEC(startLat)-latSpace; uy >= clipSSec; uy -= latSpace)
    {
        if ( uy == DEG_TO_SEC(startLat)-latSpace ) seglist->op = 14;
        else if ( seglist->op - 1 < 11 ) seglist->op = 22;
        else seglist->op -= 1;

	/* draw parallel at uy */
        uyn = computeMercatorY(uy);
        yf = SEC_TO_DOTS(ms,offsetNSec - uyn);
	ptlist[nPts].x = x0f;
	ptlist[nPts].y = yf;
	mvlist[nPts++] = 1;
	ptlist[nPts].x = x1f;
	ptlist[nPts].y = yf;
	mvlist[nPts++] = 0;
        [self addSegments:seglist :nPts];
        nPts = 0;
    }

    // draw N-S meridians - lines with 23.4333 degree from N-S meridians
    uy = DEG_TO_SEC(startLat); // allways our start y
    uyn = computeMercatorY(uy);

    dyN = Diff(uy, computeMercatorY(clipNSec));
    cnt = dyN / y_inc; // 1 deg 3600 sec
    dyN = cnt * 3600;
    dxN = cnt * DEG_TO_SEC(Tan(23.4333));

    dyS = Diff(uy, computeMercatorY((clipSSec < 0) ? (clipSSec) : (clipSSec+(60.0*3600))));
    cnt = dyS / y_inc; // 1 deg 3600 sec
    dyS = cnt * 3600;
    dxS = cnt * DEG_TO_SEC(Tan(23.4333));

    /* draw in east direction */
    for (ux0 = DEG_TO_SEC(startLng - ms->centerLng); ux0 <= clipESec+(20.0*3600); ux0 += lonSpace)
    //ux0 = DEG_TO_SEC(startLng - ms->centerLng); // 12.0 // 6.083 -> um centerLng nach rechts verschoben
    {
        if ( ux0 == DEG_TO_SEC(startLng - ms->centerLng) ) seglist->op = 11;
        else if ( seglist->op + 1 > 22 ) seglist->op = 11;
        else seglist->op += 1;

	/* draw meridian at ux0 */
        ptlist[nPts].x = SEC_TO_DOTS(ms,ux0 + dxS + offsetESec); // mirrored later N-S changed
        ptlist[nPts].y = SEC_TO_DOTS(ms,offsetNSec - uyn + dyS);
        mvlist[nPts++] = 1;
        ptlist[nPts].x = SEC_TO_DOTS(ms,ux0 - dxN + offsetESec);
        ptlist[nPts].y = SEC_TO_DOTS(ms,offsetNSec - uyn - dyN);
        mvlist[nPts++] = 0;
        [self addSegments:seglist :nPts];
        nPts = 0;
    }
    /* draw in west direction */
    for (ux0 = DEG_TO_SEC(startLng - ms->centerLng)-lonSpace; ux0 >= clipWSec-(20.0*3600); ux0 -= lonSpace)
    {
        if ( ux0 == DEG_TO_SEC(startLng - ms->centerLng)-lonSpace ) seglist->op = 22;
        else if ( seglist->op - 1 < 11 ) seglist->op = 22;
        else seglist->op -= 1;

	/* draw meridian at ux0 */
        ptlist[nPts].x = SEC_TO_DOTS(ms,ux0 + dxS + offsetESec); // mirrored later N-S changed
        ptlist[nPts].y = SEC_TO_DOTS(ms,offsetNSec - uyn + dyS);
        mvlist[nPts++] = 1;
        ptlist[nPts].x = SEC_TO_DOTS(ms,ux0 - dxN + offsetESec);
        ptlist[nPts].y = SEC_TO_DOTS(ms,offsetNSec - uyn - dyN);
        mvlist[nPts++] = 0;
        [self addSegments:seglist :nPts];
        nPts = 0;
    }

    if (nPts > 0)
        [self addSegments:seglist :nPts];
}

/* Grid
 * drawGraticule - if ms->gridLines is zero just outline entire map
 */
- (void) drawGraticule:(long)offsetESec :(long)offsetNSec
                      :(long)clipWSec :(long)clipNSec :(long)clipESec :(long)clipSSec :(SegList*)seglist
{   struct pr_pos	*ptlist = seglist->ptlist;
    unsigned char	*mvlist = seglist->mvlist;
    long		spacing, y_inc, ux0, ux1, uy; // x0, x1, y;
    int 		nPts = 0, outlineOnly = (ms->gridLines == 0);
    float		xf = 0.0, yf = 0.0, x0f = 0.0, x1f = 0.0;

    //if (ms->proj == PROJ_AZIMUT)
    //    return;

    seglist->op = 10; // grid -> lightGray

    ms->gridDegrees = 0;
    if (ms->gridLines < 0)
	return;
    else if (outlineOnly)
	spacing = y_inc = DEG_TO_SEC(15);	// value not important
    else
    {	/* choose spacing so there are approximately ms->gridLines showing */
	float	maxdegs = ms->showLng > ms->showLat*2 ? ms->showLng : ms->showLat*2;
	int	degs = maxdegs / ms->gridLines + 1;

	/* choose spacing of 15, 10, 5, 3, 2, or 1 degrees */
	if (degs < 1 || maxdegs < 2)
        {
	    outlineOnly = YES;
	    degs = 1;				// value not important
	}
        else if (degs >= 12)
            degs = 15;
	else if (degs >= 8)
            degs = 10;
	else if (degs >= 4)
            degs = 5;
        degs = gridSize;
	if (!outlineOnly && (ms->debug & D_VERBOSE))
	    fprintf(stderr, "grid is %d degrees\n", degs);
	spacing = y_inc = DEG_TO_SEC(degs);
	if (!outlineOnly)
            ms->gridDegrees = degs;
    }

    if (ms->proj)
	y_inc = DEG_TO_SEC(1);			// smaller vertical increment
    else
    {
	x0f = SEC_TO_DOTS(ms,clipWSec + offsetESec);	// constant parallel length
	x1f = SEC_TO_DOTS(ms,clipESec + offsetESec);
    }

    for (uy = clipSSec - (clipSSec % spacing); uy < clipNSec && ms->proj != PROJ_AZIMUT; uy += spacing)
    {
	/* draw parallel at uy */
	if (outlineOnly && uy > -DEG90_SECS && uy < DEG90_SECS)
            continue;	// top/bottom only
	if (nPts+2 > seglist->maxpoints)
        {   [self addSegments:seglist :nPts];
	    nPts = 0;
	}
        if (ms->proj == PROJ_AZIMUT) // circle aroung 0, 90 with radius lat uy
        {   int	uyn;
	    x0f = SEC_TO_DOTS(ms,clipWSec + offsetESec);
	    x1f = SEC_TO_DOTS(ms,clipESec + offsetESec);
            uyn = computeMercatorY(uy);
            yf = SEC_TO_DOTS(ms,offsetNSec - uyn);
        }
        else if (ms->proj == PROJ_MERCATOR)
        {   int	uyn;
	    x0f = SEC_TO_DOTS(ms,clipWSec + offsetESec);
	    x1f = SEC_TO_DOTS(ms,clipESec + offsetESec);
            uyn = computeMercatorY(uy);
            yf = SEC_TO_DOTS(ms,offsetNSec - uyn);
        }
        else if (!ms->proj)
            yf = SEC_TO_DOTS(ms, offsetNSec - uy);
	else // if (ms->proj && ms->proj != PROJ_MERCATOR) 	// projection
	{   int xp = XPROJ_SEC(XProj, abs(uy));
	    ux0 = XPROJ_MUL(clipWSec, xp);
	    ux1 = XPROJ_MUL(clipESec, xp);
	    x0f = SEC_TO_DOTS(ms,ux0 + offsetESec);
	    x1f = SEC_TO_DOTS(ms,ux1 + offsetESec);
	}
	ptlist[nPts].x = x0f;
	ptlist[nPts].y = yf;
	mvlist[nPts++] = 1;
	ptlist[nPts].x = x1f;
	ptlist[nPts].y = yf;
	mvlist[nPts++] = 0;
    }

    /* draw N-S meridians */
    for (ux0 = clipWSec - (clipWSec % spacing); ux0 <= clipESec; ux0 += spacing)
    //for (ux0 = clipWSec - (clipWSec % spacing); ux0 < clipWSec - (clipWSec % spacing)+spacing; ux0 += spacing)
    {
        //if ( !(ms->proj == PROJ_AZIMUT && (!ux0 || ux0==324000 || ux0==648000 || ux0==-75600)) )
        //    continue;

	/* draw meridian at ux0 */
	if (outlineOnly && ux0 > -DEG180_SECS && ux0 < DEG180_SECS)
            continue;		// left/right only
	if (!ms->proj || ms->proj == PROJ_MERCATOR)
	{   float	y0f, y1f;

            xf = SEC_TO_DOTS(ms, ux0 + offsetESec - (ms->cntrLngSec % spacing));
            y0f = SEC_TO_DOTS(ms, offsetNSec - computeMercatorY(clipNSec));
            y1f = SEC_TO_DOTS(ms, offsetNSec - computeMercatorY(clipSSec));
            ptlist[nPts].x = xf;
            ptlist[nPts].y = y0f;
            mvlist[nPts++] = 1;
            ptlist[nPts].x = xf;
            ptlist[nPts].y = y1f;
            mvlist[nPts++] = 0;
        }
	else
            for (uy = clipSSec - (clipSSec % spacing); ;uy += y_inc)
            {
                /* draw meridian segment above uy */
                if (nPts >= seglist->maxpoints)
                {   [self addSegments:seglist :nPts];
                    if (uy == clipSSec)
                        nPts = 0;
                    else
                    {
                        ptlist[0].x = ptlist[nPts-1].x;
                        ptlist[0].y = ptlist[nPts-1].y;
                        mvlist[0] = 1;
                        nPts = 1;
                        }
                    }
                if (uy > clipNSec)
                    uy = clipNSec;
                if (ms->proj == PROJ_AZIMUT)
                {   NSPoint	p;
                    p = computeAzimut(SEC_TO_DEG(uy), SEC_TO_DEG(ux0), ms);
                    xf = SEC_TO_DOTS(ms, DEG_TO_SEC(p.x) + offsetESec);
                    yf = SEC_TO_DOTS(ms, offsetNSec - DEG_TO_SEC(p.y));
                }
                else if (ms->proj == PROJ_MERCATOR)
                {   int	uyn;
                    uyn = computeMercatorY(uy);
                    yf = SEC_TO_DOTS(ms, offsetNSec - uyn);
                }
                else if (!ms->proj)
                    yf = SEC_TO_DOTS(ms,offsetNSec - uy);
                else if (ms->proj)
                {   int xp = XPROJ_SEC(XProj, abs(uy));
                    ux1 = XPROJ_MUL(ux0, xp);
                    xf = SEC_TO_DOTS(ms,ux1 + offsetESec);
                    }
                ptlist[nPts].x = xf;
                ptlist[nPts].y = yf;
                mvlist[nPts++] = ((uy == clipSSec - (clipSSec % spacing)) ? 1 : 0);
                if (uy == clipNSec) break;
                }
    }

    if (nPts > 0)
        [self addSegments:seglist :nPts];
}

/*
 * map_getblock
 *
 * Reads nPts coordinates from the map file, transforms them and appends the
 * display coordinates to seglist.
 *
 * input parameters:
 *	*ms	 - MapSet structure
 *	*map	 - map structure
 *	offsetEU - x-coordinate offset (map units)
 *	offsetNU - y-coordinate offset (map units)
 *	ptStart	 - offset for first display point written to seglist
 *	nPts	 - number of coordinates to read from map file
 * input/output parameters:
 *	*seglist - array for storing display points and segment start flags
 *	*xU0, *yU0  - first coordinate on entry, set to last coord on exit
 * return value:
 *	number of points written to seglist (past ptStart)
 *
 * entry conditions:
 *	map->cntrLngU has been computed from ms->cntrLngSec
 *	map->muPerDot has been computed from ms->msecPerDot
 */
static int map_getblock(MapSet *ms, MapStruct *map, long offsetEU, long offsetNU, int ptStart,
                        int nPts, SegList *seglist, long *xU0, long *yU0)
{   FILE		*dFile = map->mfile->file;
    int			i;
    int			projShft = 0, xp;
    long		lxU, lyU;	/* previous coordinates (map units) */
    long		lrxU;	/* previous rotated x-coordinate (map units) */
    long		drxU;
    long		x, y;
    float		xf, yf;
    long		wrapThreshU = WRAP_THRESH_U(map);
    long		deg180U = DEG180_U(map);
    long		cntrLngU = map->cntrLngU;
    float		muPerDot = map->muPerDot; // long
    struct pr_pos	*ptlist;
    unsigned char	*mvlist;

    if (nPts <= 0) return(0);
    ptlist = seglist->ptlist + ptStart;
    mvlist = seglist->mvlist + ptStart;
    bzero(mvlist, nPts);			/* fill with zeros */

    x = lxU = *xU0;				/* get first coordinates */
    y = lyU = *yU0;

    NORMALIZE_LONGITUDE(x, cntrLngU, deg180U);	/* rotate x (map units) */
    lrxU = x;					/* save rotated x coord */

    mvlist[0] = 1;				/* indicate new segment */

    if (ms->proj == PROJ_AZIMUT)
    {   NSPoint	p;
        p = computeAzimut(U_TO_DEG(map, y), U_TO_DEG(map, x), ms);
        x = DEG_TO_U(map, p.x);
        y = DEG_TO_U(map, p.y);
            //y = SEC_TO_U(map, computeMercatorY(U_TO_SEC(map, y)));
    }
    else if (ms->proj == PROJ_MERCATOR)
        y = SEC_TO_U(map, computeMercatorY(U_TO_SEC(map, y)));
    else if (ms->proj)
    {				/* project x (map units) */
	projShft = PROJ_SHIFT_U(map);
	xp = XPROJ_U(XProj, abs(y), projShft);
	x = XPROJ_MUL(x, xp);
    }

    /* x is now rotated and projected */
    /* next translate and scale to display coordinates */

    ptlist[0].x = U_TO_DOTS(x + offsetEU, muPerDot);
    ptlist[0].y = U_TO_DOTS(offsetNU - y, muPerDot);

    for (i = 1; i < nPts; ++i)
    {
	UNPACK_COORDS(x, y, dFile);		// read deltas
	lxU += x;				// make absolute
	lyU += y;
	x = lxU;
	y = lyU;
	NORMALIZE_LONGITUDE(x, cntrLngU, deg180U);	// rotate x (map units)
	drxU = (lrxU > x ? lrxU - x : x - lrxU);	// stroke length
	if (drxU > wrapThreshU)
	    mvlist[i] = 1;	/* do not draw strokes crossing "behind" map */
	lrxU = x;				/* save rotated x coord */

        if (ms->proj == PROJ_AZIMUT)
        {   NSPoint	p;
            p = computeAzimut(U_TO_DEG(map, y), U_TO_DEG(map, x), ms);
            x = DEG_TO_U(map, p.x);
            y = DEG_TO_U(map, p.y);
        }
        else if (ms->proj == PROJ_MERCATOR)
            y = SEC_TO_U(map, computeMercatorY(U_TO_SEC(map, y)));
	else if (ms->proj && ms->proj < PROJ_MERCATOR)
        {   // project x (map units)
	    xp = XPROJ_U(XProj, abs(y), projShft);
	    x = XPROJ_MUL(x, xp);
	}    

	xf = U_TO_DOTS(x + offsetEU, muPerDot);
	yf = U_TO_DOTS(offsetNU - y, muPerDot);
	if (mvlist[i] && mvlist[i-1])
        {		// collapse repeated move
	    ptlist[i-1].x = xf;
	    ptlist[i-1].y = yf;
	    --i;		
	    --nPts;
	}
        else if (xf == ptlist[i-1].x && yf == ptlist[i-1].y && mvlist[i] == 0)
        {   // ignore repeated points
	    --i;
	    --nPts;
	}
        else
        {
	    ptlist[i].x = xf;
	    ptlist[i].y = yf;
	}
    }
    if (nPts > 0 && mvlist[nPts-1])
	nPts--;					/* ignore final move */
    *xU0 = lxU;					/* return last coordinates */
    *yU0 = lyU;
    return(nPts);
} /* map_getblock */

long changeBitDir32(int intVal)
{   long longVal;
    int	v1, v2, v3, v4;

    v1 = intVal & 0xFF; // intel first is smallest byte (sun is last byte)
    v2 = intVal & 0xFF00;
    v3 = intVal & 0xFF0000;
    v4 = intVal & 0xFF000000;

    longVal  = (v1 >> 0) << 24; // << 0; // now third -> must be move 24 to left
    longVal += (v2 >> 8) << 16; // now third -> must be move 16 to left
    longVal += (v3 >> 16) << 8; // now second -> must be move 8 to left
    longVal += (v4 >> 24); // now first - nothing to move

    if ( intVal < 0 ) longVal += 256;

    return longVal;
}

long changeBitDir16(int intVal)
{   long longVal;
    int	v1, v2;

    v1 = intVal & 0xFF; // intel first is  smallest byte (sun is last byte)
    v2 = intVal & 0xFF00;

    longVal  = (v1 >> 0) << 8; // << 0; // now second -> must be move 8 to left
    longVal += (v2 >> 8); // now first - nothing to move

//if ( intVal < 0 )
//    printf("und hier ? ");
    return longVal;
}

/*
 * drawTheMap
 */
- (void)drawTheMap:(int)mapIndex :(long)featureMask :(SegList*)seglist // :(MRect*) region
{   MapStruct	*map;
    MapFeature	*feature;
    int 	blockCount = 0, vecCount = 0, ptCount = 0;
    int 	rank, nPts, newPts;
    int 	new_color, new_op;
    int 	new_line_width, fatten = 0;
    short	*new_pattern = NULL;
    long	clipNSec, clipSSec, clipESec, clipWSec;
    long	offsetNU, offsetEU;
    long	clipNU, clipSU;
    long	normWU, normEU;
    long	nU, sU, eU, wU;
    long	xU0, yU0;
    long	nItems, segid, segid2, segpos, nBytes; //  prevSegpos = 50;
    struct 	segdict *dictptr;
    long	featuresSeen = 0;

    offsetEU = ms->halfWidSec;
    offsetNU = ms->canvasNSec;
    clipNSec = ms->canvasNSec - DOTS_TO_SEC(ms,region.r_top);
    clipSSec = clipNSec - DOTS_TO_SEC(ms, region.r_height);

    if (clipNSec > DEG90_SECS) clipNSec = DEG90_SECS;
    if (clipSSec < -DEG90_SECS) clipSSec = -DEG90_SECS;

    clipWSec = -ms->halfWidSec + DOTS_TO_SEC(ms, region.r_left);
    clipESec = clipWSec + DOTS_TO_SEC(ms, region.r_width);
    //clipESec = clipWSec + DEG_TO_SEC(ms->showLng);	// georg 2003-07-16 00:17

    if (ms->proj && ms->proj < PROJ_MERCATOR)
    {
	/* find maximum longitude ranges in projected area (in seconds) */
	long absN = abs(clipNSec);
	long absS = abs(clipSSec);
	long maxNS = absN > absS ? absN : absS;
	long minNS;
	if (clipNSec >= 0 && clipSSec <= 0) minNS = 0;
	else minNS = absN < absS ? absN : absS;
	if (clipWSec >= 0)	// use W longitude at minNS
        {
	    if (minNS >= DEG90_SECS)
                clipWSec = -DEG180_SECS;
	    else
                clipWSec = XPROJ_DIV(clipWSec, XPROJ_SEC(XProj, minNS));
	}
        else			// use W longitude at maxNS
        {
	    if (maxNS >= DEG90_SECS)
                clipWSec = -DEG180_SECS;
	    else
                clipWSec = -XPROJ_DIV(-clipWSec, XPROJ_SEC(XProj, maxNS));
	}
	if (clipESec >= 0)	// use E longitude at maxNS
        {
	    if (maxNS >= DEG90_SECS)
                clipESec = DEG180_SECS;
	    else
                clipESec = XPROJ_DIV(clipESec, XPROJ_SEC(XProj, maxNS));
	}
        else			// use E longitude at minNS
        {
	    if (minNS >= DEG90_SECS)
                clipESec = DEG180_SECS;
	    else
                clipESec = -XPROJ_DIV(-clipESec, XPROJ_SEC(XProj, minNS));
	}
    }
    if (clipESec > DEG180_SECS)
        clipESec = DEG180_SECS;
    if (clipWSec < -DEG180_SECS)
        clipWSec = -DEG180_SECS;

    if (mapIndex < 0 || mapIndex >= ms->mCount)
    {
	[self drawGraticule:offsetEU :offsetNU :clipWSec :clipNSec :clipESec :clipSSec :seglist];
        if ( drawZodiac )
            [self drawZodiacLines:offsetEU :offsetNU :clipWSec :clipNSec :clipESec :clipSSec :seglist];
        if ( drawDestiny )
            [self drawDestinyLines:offsetEU :offsetNU :clipWSec :clipNSec :clipESec :clipSSec :seglist];
	return;
    }

    map = &ms->maps[mapIndex];
    if (map->uPerDeg == -1)
        map_load(ms, mapIndex);
    if (map->minSSec > clipNSec || map->maxNSec < clipSSec)
        return;

    map->cntrLngU = SEC_TO_U(map,ms->cntrLngSec);	/* cache this */
    map->muPerDot = SEC_TO_U(map,ms->msecPerDot);	/* cache this */
    if (map->muPerDot <= 0)
    {
	fprintf(stderr, "map resolution exceeded, expect rounding errors\n");
	map->muPerDot = 1;
    }
    /* normalize frame coordinates back to Greenwich */
    normWU = ( clipWSec < 0 ) ? (-SEC_TO_U(map,-clipWSec)) : (SEC_TO_U(map,clipWSec));
    normEU = ( clipESec < 0 ) ? (-SEC_TO_U(map,-clipESec)) : (SEC_TO_U(map,clipESec));

    //if (clipESec != DEG180_SECS || clipWSec != -DEG180_SECS)	// original
    if (clipESec != DEG_TO_SEC(ms->maxLng) || clipWSec != -DEG_TO_SEC(ms->maxLng))	// georg 2003-07-11
    {
	NORMALIZE_LONGITUDE(normWU, -map->cntrLngU, DEG180_U(map));
	NORMALIZE_LONGITUDE(normEU, -map->cntrLngU, DEG180_U(map));
    }
    /* check if frame crosses +/-180 */
    if (normWU > normEU)
    {
	if (map->minWSec > U_TO_SEC(map,normEU) && map->maxESec < U_TO_SEC(map,normWU))
	    return;
    }
    else if (map->minWSec > U_TO_SEC(map,normEU) || map->maxESec < U_TO_SEC(map,normWU))
        return;
    offsetNU = SEC_TO_U(map, offsetNU);
    offsetEU = SEC_TO_U(map, offsetEU);
    clipNU = SEC_TO_U(map, clipNSec);
    clipSU = SEC_TO_U(map, clipSSec);

    ++generation;
    OpenMapFile(ms, map->mfile);
    fseek(map->mfile->file, map->mfile->header.dictaddr, 0);
    if (ms->dictbuf != NULL && ms->dictbuf_len < map->mfile->header.segcount)
    {
	free(ms->dictbuf);
	ms->dictbuf_len = 0;
	ms->dictbuf = NULL;
    }
    if (ms->dictbuf == NULL)
    {
	ms->dictbuf = (struct segdict *)malloc(ms->max_segcount * sizeof(struct segdict));
	if (ms->dictbuf == NULL)
        {
	    fprintf(stderr, "%s: out of memory\n", Pgm);
	    return;
	}
	ms->dictbuf_len = ms->max_segcount;
    }
    seglist -> line_width = -1;			/* get new values below */
    if (map->muPerDot < ms->fatten_thresh)
        fatten = 1;

    nItems = fread(ms->dictbuf, sizeof(struct segdict), map->mfile->header.segcount, map->mfile->file);
    for (dictptr = ms->dictbuf; nItems-- > 0; dictptr++)
    {
#ifdef __LITTLE_ENDIAN__
        segid = changeBitDir32(dictptr->segid);
        nU = changeBitDir32(dictptr->maxlat);
        sU = changeBitDir32(dictptr->minlat);
        eU = changeBitDir32(dictptr->maxlong);
        wU = changeBitDir32(dictptr->minlong);
        segpos = changeBitDir32(dictptr->absaddr);
        nBytes = changeBitDir16(dictptr->nbytes);
        rank = changeBitDir16(dictptr->rank) + map->rank_offset;
#else
        segid = dictptr->segid;
        nU = dictptr->maxlat;
        sU = dictptr->minlat;
        eU = dictptr->maxlong;
        wU = dictptr->minlong;
        segpos = dictptr->absaddr;
        nBytes = dictptr->nbytes;
        rank = dictptr->rank + map->rank_offset;
#endif
        featuresSeen |= 1 << (rank-1);
        if (!((1 << (rank-1)) & featureMask))
            continue;	// skip this segment
        if (sU > clipNU || nU < clipSU
            || ((normWU > normEU)  && (wU > normEU && eU < normWU))
            || ((normWU <= normEU) && (wU > normEU || eU < normWU)))
            continue;			// segment out of the clip region

        if (rank <= 0 || rank > MAX_FEATURE)
            rank = 0;
        if (ms->features[map->type] == NULL)
            feature = &ms->default_feature;
        else
            feature = &ms->features[map->type][rank];
        if (feature->name == NULL)
            feature = &ms->default_feature;
        new_line_width = feature->line_width + fatten;
        new_op = new_color = feature->highlight ? ms->highlight_color : feature->color;

        /*	if (ms->useColor)
        {
            new_op = PIX_SRC | PIX_COLOR(new_color);
            if (ms->useColorPat[new_color])
                new_pattern = Textures[ms->patterns[new_color]];
            else
                new_pattern = NULL;
        }
        else
        {
            new_op = PIX_SRC | PIX_COLOR(FG_COLOR);
            new_pattern = Textures[ms->patterns[new_color]];
        }
        */
        if (new_op != seglist->op || new_pattern != seglist->pattern ||
            new_line_width != seglist->line_width)
        {
            if (ptCount > 0)	// draw buffered segments before changing color
            {   [self addSegments:seglist :ptCount];
                ptCount = 0;
            }
            seglist->op = new_op;
            seglist->pattern = new_pattern;
            seglist->line_width = new_line_width;
        }
        fseek(map->mfile->file, segpos, 0);
        GETLONG(xU0, map->mfile->file);		// first segment coordinates
        GETLONG(yU0, map->mfile->file);
        GETLONG(segid2, map->mfile->file);
        if (segid != segid2)
        {
            fprintf(stderr, "warning: segment format error @%d: segid %d should be %d\n", (int)(dictptr-ms->dictbuf), (int)segid2, (int)segid);
            continue;
        }

        GETSHORT(nPts, map->mfile->file);	// number of vectors in seg
        getc(map->mfile->file);			// skip padding
        getc(map->mfile->file);			// skip padding
        nPts++;					// number of points in seg
        blockCount++;
        while (ptCount + nPts > seglist->maxpoints)	 // split long segments
        {   int splitCount = seglist->maxpoints - ptCount;

            newPts = map_getblock(ms, map, offsetEU, offsetNU, ptCount, splitCount, seglist, &xU0, &yU0);
            ptCount += newPts;
            vecCount += newPts;
            nPts -= splitCount;
            if (ptCount > 0)
            {   [self addSegments:seglist :ptCount];
                ptCount = 0;
            }
            /* start next segment by repeating last point (xU0, yU0) */
        }
        newPts = map_getblock(ms, map, offsetEU, offsetNU, ptCount, nPts, seglist, &xU0, &yU0);
        ptCount += newPts;
        vecCount += newPts;
    }
    if (ptCount > 0)
        [self addSegments:seglist :ptCount];
}

/* ComputeEllipse
 */
void ComputeEllipse(MapSet *ms, unsigned short *xTable)
{   int uy;
    double xProj;

    for (uy = 0; uy <= SEC_TO_PROJ_IN(DEG90_SECS); ++uy)
    {
	/* elliptical projection */
	xProj = (double) uy / SEC_TO_PROJ_IN(DEG90_SECS);
	xProj = sqrt((double)(1 - xProj*xProj));
	xTable[uy] = xProj * (1 << PROJ_SHIFT_VAL);
    }
} // ComputeEllipse

/* ComputeSine
 */
void ComputeSine(MapSet *ms, unsigned short *xTable)
{   int uy;
    double xProj;

    for (uy = 0; uy <= SEC_TO_PROJ_IN(DEG90_SECS); ++uy)
    {
	/* sinusoidal projection */
	xProj = (double) uy * M_PI_2 / SEC_TO_PROJ_IN(DEG90_SECS);
	xProj = cos(xProj);		// input to cos in radians
	xTable[uy] = xProj * (1 << PROJ_SHIFT_VAL);
    }
} // ComputeSine

/* ComputeMercator
 */
void ComputeMercator(MapSet *ms, unsigned short *xTable)
{   int		uy;
    double	xProj;

return;
    for (uy = 0; uy <= SEC_TO_PROJ_IN(DEG90_SECS); ++uy)
    {
	/* elliptical projection */
	xProj = (double) uy / SEC_TO_PROJ_IN(DEG90_SECS);
	xProj = sqrt((double)(1 - xProj*xProj));
	xTable[uy] = xProj * (1 << PROJ_SHIFT_VAL);
    }
} // ComputeMercator

/* shift_args
 */
/*static void shift_args(int n, int *argc, char *argv[])
{
    while (*argv != NULL)
    {
	*argv = argv[n];
	++argv;
    }
    *argc -= n;
}
*/

/*
 * map_init
 */
MapSet *map_init(NSString *mapset)
{   MapSet		*ms;
    char		*defProfile;
    int			i;
    NSString		*path;
    NSFileManager	*fileManager = [NSFileManager defaultManager];

    Pgm++;
    ms = (MapSet*)calloc(1, sizeof(MapSet));

    /* get and set the name of the map set, which is in the main bundle */
    if ( ![mapset length] )
    {   NSLog(@"CIAMap: No 'mapset' entry, using default 'world_8f6.mapset'");
        mapset = @"world_8f6.mapset";
    }
    defProfile = calloc([mapset cStringLength]+1, sizeof(char));
    [mapset getCString:defProfile];

    ms->maxMaps = MAX_MAPS;
    ms->maxTypes = MAX_TYPES;
    ms->maxScales = MAX_SCALES;
    ms->maxColors = MAX_COLORS;

    /* set the path to the map folder */
    path = [[[NSBundle bundleForClass:[CIAMap class]] resourcePath] stringByAppendingPathComponent:MAP_FOLDER];
    ms->mapDir = calloc([path cStringLength]+1, sizeof(char));
    [path getCString:ms->mapDir];
    path = [path stringByAppendingPathComponent:mapset];
    if (![fileManager fileExistsAtPath:path])
        NSLog(@"map_init(): file does not exist '%@'", path);

    if (ms->maxColors > MAX_COLORS) ms->maxColors = MAX_COLORS;
/*
    ms->mTypes = (char **)calloc(ms->maxTypes, sizeof(char *));
    ms->features = (MapFeature **)calloc(ms->maxTypes, sizeof(char *));
    ms->enableMasks = (long *)calloc(ms->maxTypes, sizeof(long));
    ms->min_featureMasks = (long *)calloc(ms->maxTypes, sizeof(long));
    ms->optional = (char *)calloc(ms->maxTypes, sizeof(char));
    ms->enforce_suppression = (char *)calloc(ms->maxTypes, sizeof(char));
    ms->suppressMasks = (long **)calloc(ms->maxTypes, sizeof(long *));
    ms->maxFeature = (char *)calloc(ms->maxTypes, sizeof(char));
    ms->scaleNames = (char **)calloc(ms->maxScales, sizeof(char *));
    ms->maps = (MapStruct *)calloc(ms->maxMaps, sizeof(MapStruct));
    ms->autoRes = (float *)calloc(ms->maxMaps, sizeof(float));
    ms->minRes = (float *)calloc(ms->maxMaps, sizeof(float));
*/
    ms->mTypeCount = 0;
    ms->scaleCount = 0;
    ms->mCount = 0;

    ms->showLng = UNDEF_DEGS;
    ms->showLat = UNDEF_DEGS;
    ms->centerLng = UNDEF_DEGS;
    ms->centerLat = UNDEF_DEGS;

    ms->def_showLng = UNDEF_DEGS;
    ms->def_showLat = UNDEF_DEGS;
    ms->def_centerLng = UNDEF_DEGS;
    ms->def_centerLat = UNDEF_DEGS;

    ms->minLat = UNDEF_DEGS;
    ms->minLng = UNDEF_DEGS;
    ms->maxLat = UNDEF_DEGS;
    ms->maxLng = UNDEF_DEGS;

    ms->minLatSec = -DEG90_SECS;	/* assume whole world until map_load_all */
    ms->maxLatSec = DEG90_SECS;
    ms->minLngSec = -DEG180_SECS;
    ms->maxLngSec = DEG180_SECS;

    ms->scaleI = 0;
    ms->minAutoI = 0;
    ms->proj = PROJ_RECT;
    ms->margin = MAP_MARGIN;
    ms->gridLines = GRAT_MERIDIANS;
    ms->gridWidth = 1;
    ms->gridColor = 1;
    ms->bgColor = 0;
    ms->highlight_color = 1;

    ms->mapType = -1;
    ms->debug = D_DEFAULT;

    ms->colors.type = 1; // RMT_EQUAL_RGB;
    ms->colors.length = 0;
/*
    ms->colors.map[0] = (unsigned char *)malloc(ms->maxColors);
    ms->colors.map[1] = (unsigned char *)malloc(ms->maxColors);
    ms->colors.map[2] = (unsigned char *)malloc(ms->maxColors);
    ms->patterns = (unsigned char *)malloc(ms->maxColors);
    ms->useColorPat = (unsigned char *)malloc(ms->maxColors);
*/
    ms->useColor = NO;
    ms->fatten_thresh = FATTEN_THRESH;

    ms->default_feature.rank = 0;
    ms->default_feature.name = "unknown map feature";
    ms->default_feature.line_width = 1;
    ms->default_feature.color = 1;

    for (i = 0; i < ms->maxTypes; i++)
	ms->enableMasks[i] = -1;		// enable all maps by default

    if (defProfile != NULL)
	map_read_profile(ms, ms->mapDir, defProfile);

    return(ms);
} // map_init

/*
 * map_dots_to_deg
 */
float map_dots_to_deg(MapSet *ms, int dots)
{
    return (DOTS_TO_DEG(ms,dots));
} // map_dots_to_deg

/*
 * map_project_longitude
 */
float map_project_longitude(MapSet *ms, float longitudeDeg, float latitudeDeg)
{   long	absNSec, lngSec;

    if (!ms->proj) 
	return (longitudeDeg);

    absNSec = (long)DEG_TO_SEC(latitudeDeg);
    absNSec = abs(absNSec);
    if (absNSec >= DEG90_SECS)
	lngSec = DEG180_SECS;
    else
	lngSec = XPROJ_DIV((long)DEG_TO_SEC(abs(longitudeDeg)), XPROJ_SEC(XProj, absNSec));
    if (longitudeDeg < 0)
        lngSec = -lngSec;
    longitudeDeg = SEC_TO_DEG(lngSec);
    return(longitudeDeg);
} // map_project_longitude

/* map_setup
 */
int map_setup(MapSet *ms, int width, int height)
{
    ms->minLng = SEC_TO_DEG(ms->minLngSec);
    ms->maxLng = SEC_TO_DEG(ms->maxLngSec);
    ms->minLat = SEC_TO_DEG(ms->minLatSec);
    ms->maxLat = SEC_TO_DEG(ms->maxLatSec);

    if (ms->proj < 0 || ms->proj > MAX_PROJ) ms->proj = PROJ_RECT;

    if (ms->proj)
    {
	XProj = ProjTables[ms->proj-1];
	if (XProj == NULL)
        {
	    XProj = (unsigned short *)
		malloc(SEC_TO_PROJ_IN(DEG90_SECS) * sizeof(unsigned short));
	    ProjTables[ms->proj-1] = XProj;
	    if (XProj == NULL)
            {
		fprintf(stderr, "unable to allocate projection table\n");
		ms->proj = PROJ_RECT;
	    }
	    switch (ms->proj)
            {
                case PROJ_ELLIPSE:
                    ComputeEllipse(ms, XProj);
                    break;
                case PROJ_SINE:
                    ComputeSine(ms, XProj);
                    break;
                case PROJ_MERCATOR:
                    ComputeMercator(ms, XProj);
                    break;
                case PROJ_AZIMUT:
                    ComputeMercator(ms, XProj);
                    break;
                case PROJ_RECT:
                default:
                    ms->proj = PROJ_RECT;
                    break;
            }
	}
    }

    if (ms->showLat == UNDEF_DEGS) ms->showLat = ms->def_showLat;
    if (ms->showLng == UNDEF_DEGS) ms->showLng = ms->def_showLng;
    if (ms->centerLat == UNDEF_DEGS) ms->centerLat = ms->def_centerLat;
    if (ms->centerLng == UNDEF_DEGS) ms->centerLng = ms->def_centerLng;

    if (ms->showLat == UNDEF_DEGS)
	ms->showLat = ms->maxLat - ms->minLat;
    if (ms->showLng == UNDEF_DEGS)
	ms->showLng = ms->maxLng - ms->minLng;
    if (ms->centerLat == UNDEF_DEGS)
	ms->centerLat = (ms->maxLat + ms->minLat) / 2;
    if (ms->centerLng == UNDEF_DEGS)
	ms->centerLng = (ms->maxLng + ms->minLng) / 2;

    NORM_MODULO(ms->centerLng, 180);	// put into +/180 range

    map_rescale(ms, width, height);

    ms->scaleI = 0;
    if (ms->scaleCount > 0)
    {	// find appropriate scale
	float dRes = (float) DOTS_PER_DEG(ms);
	float scaleRes = ms->minAuto;
	int i;
	ms->scaleI = ms->minAutoI;
	for (i = 0; i < ms->scaleCount; ++i)
        {
	    if (ms->autoRes[i] > scaleRes && ms->autoRes[i] <= dRes)
            {
		ms->scaleI = i;
		scaleRes = ms->autoRes[i];
	    }
	}
	if (ms->debug & D_VERBOSE)
	    fprintf(stderr, "auto scale = %s (%g)\n", ms->scaleNames[ms->scaleI], scaleRes);
    }
    return(0);
} // map_setup

/*
 * drawMap
 */
- (void)drawMap:(int)mapIndex :(SegList*)seglist // :(MRect*) region
{   MapStruct	*map;
    long	fmask;

    if (mapIndex == -1)			// Graticule
    {
	if (ms->gridLines < 0)
            return;	// disabled
	seglist->line_width = ms->gridWidth;
	//seglist->pattern = Textures[ms->patterns[ms->gridColor]];
/*	if (ms->useColor)
        {
	    seglist->op = PIX_SRC|PIX_COLOR(ms->gridColor);
	    if (!ms->useColorPat[ms->gridColor]) seglist->pattern = NULL;
	}
        else
	    seglist->op = PIX_SRC|PIX_COLOR(FG_COLOR);
*/
//	DrawMap(ms, -1, -1, region, seglist, drawProc, list);
	[self drawTheMap:-1 :-1 :seglist]; // :region
	return;
    }
    if (mapIndex < 0 || mapIndex >= ms->mCount)
        return;	// invalid map

    map = &ms->maps[mapIndex];
    if (map->uPerDeg == -1)
        map_load(ms, mapIndex);
    fmask = ms->enableMasks[map->type];
    if (ms->enforce_suppression[map->type])
	fmask &= ~ ms->suppressMasks[map->type][map->scaleI];
    if (map->rank_offset == 0)
	fmask &= map->mfile->header.features;	// can't mask if offset

    if (fmask == 0)
        return;
    [self drawTheMap:mapIndex :fmask :seglist]; // :region
}

/*
 *
 */

- (void)addSegments:(SegList*)seglist :(int)ptCount
{   int 		i, ptcnt;
    float		width = 0.0;
    NSPoint		p0 = {0.0, 0.0};
    NSColor		*col = [NSColor blackColor];
    VPolyLine		*polyLine = nil;
    NSMutableArray	*list = nil;

    switch ( seglist->op )
    {
        case 24: col = [NSColor darkGrayColor]; break; // destiny horicontal
        case 23: col = [NSColor darkGrayColor]; break; // destiny vertical
        case 22: col = [NSColor colorWithCalibratedRed:0.5 green:0.5 blue:1.0 alpha:1.0]; break; // fisch
        case 21: col = [NSColor colorWithCalibratedRed:0.0 green:1.0 blue:1.0 alpha:1.0]; break; // wassermann
        case 20: col = [NSColor colorWithCalibratedRed:0.333 green:0.05 blue:0.0 alpha:1.0]; break; // steinbock
        case 19: col = [NSColor colorWithCalibratedRed:0.0 green:0.5 blue:0.0 alpha:1.0]; break; // schuetze
        case 18: col = [NSColor colorWithCalibratedRed:0.5 green:0.0 blue:1.0 alpha:1.0]; break; // skorpion
        case 17: col = [NSColor colorWithCalibratedRed:0.0 green:0.0 blue:1.0 alpha:1.0]; break; // waage
        case 16: col = [NSColor colorWithCalibratedRed:1.0 green:0.75 blue:0.75 alpha:1.0]; break; //jungfrau
        case 15: col = [NSColor colorWithCalibratedRed:1.0 green:1.0 blue:0.0 alpha:1.0]; break; // loewe
        case 14: col = [NSColor colorWithCalibratedRed:0.5 green:1.0 blue:0.5 alpha:1.0]; break; // krebs
        case 13: col = [NSColor colorWithCalibratedRed:1.0 green:0.5 blue:0.0 alpha:1.0]; break; // zwilling
        case 12: col = [NSColor colorWithCalibratedRed:0.0 green:0.0 blue:0.0 alpha:1.0]; break; // stier
        case 11: col = [NSColor colorWithCalibratedRed:1.0 green:0.0 blue:0.0 alpha:1.0]; break; // widder
        case 10:
            col = [NSColor colorWithCalibratedWhite:0.6 alpha:1.0];	// grid
            list = [listDict objectForKey:@"Grid"];
            break;
        case 9: col = [NSColor redColor]; break; // highlight color
        case 8: col = [NSColor darkGrayColor]; break; // canals
        case 7:
            col = [NSColor colorWithCalibratedRed:0.0 green:0.2 blue:1.0 alpha:1.0]; break; // intermittent rivers
        case 6: col = [NSColor colorWithCalibratedRed:0.0 green:0.27 blue:0.667 alpha:1.0]; break; // intermit lakes
        case 5: col = [NSColor grayColor]; break; // reefs, salt pans
        case 4:
            col = waterColor;		// rivers
            list = [listDict objectForKey:@"Rivers"];
            break;
        case 3: col = [NSColor cyanColor]; break; // ice glacies
        case 2:
            col = [NSColor purpleColor]; width = 0.1;	// borders brownColor
            list = [listDict objectForKey:@"Borders"];
            break;
        default:
            col = [NSColor blackColor]; // coasts, lakes
            list = [listDict objectForKey:@"Coasts"];
    }
    if (!list)
        return;

    ptcnt = 0;
    for (i=0 ; i<ptCount; i++)
    {	NSPoint	p, p1;

        // seglist->ptlist[i].y = mp.y - (seglist->ptlist[i].y - mp.y); // mirror pts !!!
        p.x = seglist->ptlist[i].x;
        p.y = mp.y - (seglist->ptlist[i].y - mp.y); // mirror pts !!!

        /*if ( seglist->op == 10 ) // ( seglist->op != 3 && seglist->op != 9 && seglist->op != 11 )
        {
            if ( p.x < ll.x ) ll.x = p.x;
            if ( p.y < ll.y ) ll.y = p.y;
            if ( p.x > ur.x ) ur.x = p.x;
            if ( p.y > ur.y ) ur.y = p.y;
        }*/
        if ( seglist->mvlist[i] )
        {
            p0 = p;
            ptcnt = 1;
            polyLine = nil;

            //if ( i+2 < ptCount && !(seglist->mvlist[i+2]) ) // our next pt is no move and no list end -> polyline
            {   VPolyLine	*lastPolyLine = nil;

                /* check if p0 fits to endpoint of the last PolyLine */
                if ([list count])
                    lastPolyLine = [list objectAtIndex:[list count]-1];
                if (lastPolyLine && [lastPolyLine isKindOfClass:[VPolyLine class]])
                {   NSPoint	endP = [lastPolyLine pointWithNum:MAXINT];

                    if (DiffPoint(endP, p0) <= TOLERANCE)
                        polyLine = lastPolyLine;
                }
                if (!polyLine)
                {   polyLine = [VPolyLine polyLine];
                    [polyLine setWidth:width];
                    [polyLine setColor:col];
                    [polyLine addPoint:p0];
                }
            }
        }
        else
        {
            p1.x = p.x;
            p1.y = p.y;

            if ( ![polyLine numPoints] &&
                 ptcnt == 1 && (i+1 >= ptCount || seglist->mvlist[i+1]) ) // next pt is a move or list end -> line
            {   VLine	*line = [VLine line];

                [line setWidth:width];
                [line setColor:col];
                [line setVertices:p0 :p1];
                [list addObject:line];
            }
            else // if ( polyLine )
            {   [polyLine addPoint:p1];
                ptcnt++;
                if ( i+1 >= ptCount || seglist->mvlist[i+1] ) // next pt is a move or list end
                {
                    [polyLine truncate];
                    if (![list containsObject:polyLine])
                        [list addObject:polyLine];
                }
            }
            p0 = p1;
        }
    }
}

- (NSMutableDictionary*)generateMapListWithCenter:(NSPoint)center size:(NSSize)size grid:(float)grid
                                                 :(BOOL)zodiacLines :(BOOL)destinyLines
                                           mapset:(NSString*)mapset
{   Document		*curDoc = nil;
    SegList		seglist;
    char 		**useTypeNames;
    int			n_useTypes = 0;
    int			i, j, w, allTypes = NO, preload = NO;
    long		clipNSec, clipSSec, clipESec, clipWSec;
    NSRect		bRect = NSZeroRect;
    NSArray		*allKeys;
    NSMutableArray	*list;

    listDict = [NSMutableDictionary dictionary];
    [listDict setObject:[NSMutableArray array] forKey:@"Ocean"];
    [listDict setObject:[NSMutableArray array] forKey:@"Coasts"];
    [listDict setObject:[NSMutableArray array] forKey:@"Rivers"];
    [listDict setObject:[NSMutableArray array] forKey:@"Borders"];
    [listDict setObject:[NSMutableArray array] forKey:@"Grid"];
    allKeys = [listDict allKeys];

    waterColor = [NSColor colorWithCalibratedRed:0.5 green:0.85 blue:1.0 alpha:1.0];

    ms = map_init(mapset);

    // path to maps ? -> allways world  -> fix ! -> Map.m

    drawZodiac = zodiacLines;
    drawDestiny = destinyLines;
    gridSize = grid;

    /* size of view */
    for ( i=[[NSApp windows] count]-1; i>=0; i-- )
    {   Document *doc = [(App*)NSApp documentInWindow:[[(App*)NSApp windows] objectAtIndex:i]];

        if ( [[doc name] hasPrefix:MAP_PREFIX] )
        {   bRect = [[doc documentView] bounds];
            curDoc = doc;
            region.r_width  = bRect.size.width;		// set view size
            region.r_height = bRect.size.height;
            break;
        }
    }
    if ( !curDoc )
        return nil;
    mp.x = region.r_width  / 2.0;
    mp.y = region.r_height / 2.0;

    ms->centerLng = center.x;	// (width) lng in grad - Mitte des Bereiches der dargestellt werden soll
    ms->centerLat = center.y;	// (height) lat in grad - Mitte des Bereiches der dargestellt werden soll
    ms->showLng = size.width;	// (width?) longitude in grad - des Bereiches der dargestellt werden soll
    ms->showLat = size.height;	// (height?) latitude in grad - des Bereiches der dargestellt werden soll

    ms->proj = PROJ_MERCATOR;	// PROJ_MERCATOR, PROJ_AZIMUT, PROJ_RECT, PROJ_ELLIPSE, PROJ_SINE
    ms->margin = 0;
    ms->gridLines = 15;		// -1 = no grid (graticule), 0 = only outline?, 234.. acount of inside view

    allTypes = YES;
    preload = YES;

    if (ms->mCount <= 0)
        map_error("no maps in profile\n");

    if (allTypes)
    {
	for (j = 0; j < ms->mTypeCount; ++j)
	    ms->enableMasks[j] = -1;
    }
    else
    {   useTypeNames = (char**)calloc(ms->mTypeCount, sizeof(char*));

	for (i = 0; i < n_useTypes; ++i)
        {   int iok = 0;

	    for (j = 0; j < ms->mTypeCount; ++j)
		if (!strcmp(useTypeNames[i], ms->mTypes[j]))
                {
		    ms->enableMasks[j] = iok = -1;
		    break;
		}
	    if (!iok)
		fprintf(stderr, "warning: unknown map type: %s\n", useTypeNames[i]);
	}
    }

    if (region.r_width == -1)
    {
	region.r_width = DEF_CWIDTH;
	region.r_height = DEF_CHEIGHT;
    }

    seglist.maxpoints = MAX_SEGLIST;
    seglist.ptlist = (struct pr_pos *) calloc(seglist.maxpoints, sizeof(struct pr_pos));
    seglist.mvlist = (unsigned char *) calloc(seglist.maxpoints, 1);

    ms->useColor = 1;	// else bw

    if (preload)
	map_load_all(ms);

    map_setup(ms, region.r_width, region.r_height);

    [self drawMap:-1 :&seglist]; // :&region

    for (i = ms->mCount-1; i >= 0; --i)
	if (ms->maps[i].scaleI == ms->scaleI)
            [self drawMap:i :&seglist];	// :&region

    free(seglist.ptlist);
    free(seglist.mvlist);
    if (ms->dictbuf != NULL)
    {   free(ms->dictbuf);
	ms->dictbuf_len = 0;
	ms->dictbuf = NULL;
    }
    for (i=0; i<MAX_TYPES; i++)
        if ( ms->suppressMasks[i] != NULL )
        {   free(ms->suppressMasks[i]);
            ms->suppressMasks[i] = NULL;
        }
    for (i=0; i<MAX_TYPES; i++)
    {   for (j=1; j<ms->maxFeature[i]; j++) // 1 !
            free(ms->features[i][j].name);
        ms->maxFeature[i] = 0;
    }
    for (i=0; i<MAX_TYPES; i++)
        if ( ms->features[i] != NULL )
        {   free(ms->features[i]);
            ms->features[i] = NULL;
        }

    for (i=0; i<ms->scaleCount; i++)
        free(ms->scaleNames[i]);
    ms->scaleCount = 0;

    for (i=0; i<ms->mTypeCount; i++)
        free(ms->mTypes[i]);
    ms->mTypeCount = 0;

    for (i = ms->mCount-1; i >= 0; --i)
    {   CloseFile(ms, ms->maps[i].mfile);
        free(ms->maps[i].mfile);
        ms->maps[i].mfile = NULL;
    }
    free(filePool);
    filePool = NULL;
    poolSize = 0;

    clipNSec = ms->canvasNSec - DOTS_TO_SEC(ms,region.r_top);
    clipSSec = clipNSec - DEG_TO_SEC(ms->showLat);
    if ( clipNSec == DEG90_SECS )  clipNSec = 80.0*3600;
    if ( clipSSec == -DEG90_SECS ) clipSSec = -80.0*3600;

    clipWSec = -ms->halfWidSec + DOTS_TO_SEC(ms, region.r_left);
    clipESec = clipWSec + DEG_TO_SEC(ms->showLng);

#if 0
    if (ms->proj == PROJ_MERCATOR)
    {   ll.x = SEC_TO_DOTS(ms, clipWSec + ms->halfWidSec);
        ur.x = SEC_TO_DOTS(ms, clipESec + ms->halfWidSec);
        ll.y = SEC_TO_DOTS(ms, ms->canvasNSec - computeMercatorY(clipSSec));
        ur.y = SEC_TO_DOTS(ms, ms->canvasNSec - computeMercatorY(clipNSec));
        ll.y = mp.y - (ll.y - mp.y); // mirror pts
        ur.y = mp.y - (ur.y - mp.y); // mirror pts
    }

    for (l=0; l<[allKeys count]; l++)
    {   NSMutableArray	*list = [listDict objectForKey:[allKeys objectAtIndex:l]];
        NSRect		rect;

        /* clip */
        rect.origin.x = rect.origin.y = 0.0;
        rect.size.width = ur.x - ll.x;
        rect.size.height = ur.y - ll.y;
        for (i=[list count]-1; i>=0; i--)
        {   VGraphic		*gr = [list objectAtIndex:i], *ng;
            NSAutoreleasePool	*pool = [NSAutoreleasePool new];

            if ( (ng = [gr clippedWithRect:rect]) )
            {
                if ([ng isMemberOfClass:[VGroup class]])
                {   NSMutableArray	*glist;
                    int			c, cCnt;

                    [list removeObjectAtIndex:i];
                    glist = [(VGroup*)ng list];
                    for (c=0, cCnt = [glist count]; c<cCnt; c++)
                        [list insertObject:[glist objectAtIndex:c] atIndex:i];
                }
                else
                {   [list removeObjectAtIndex:i];
                    [list insertObject:ng atIndex:i];
                }
            }
            else
                [list removeObjectAtIndex:i]; // graphic not inside rect
            [pool release];
        }
    }
#endif

    /* concatenate segments of coasts */
    list = [listDict objectForKey:@"Coasts"];
    for (w=0; w<2; w++)
    {   float	dist = (!w) ? TOLERANCE : 0.4;

        for (i=0; i<(int)[list count]-1; i++)
        {   VPolyLine	*g = [list objectAtIndex:i];
            NSPoint	begP, endP;
            int		j, removed = 0;

            if (![g isKindOfClass:[VPolyLine class]] || [g filled])
                continue;
            begP = [g pointWithNum:0];
            endP = [g pointWithNum:MAXINT];

            for (j=i+1; j<(int)[list count]; j++)
            {   VPolyLine	*g1 = [list objectAtIndex:j];
                NSPoint		begP1, endP1;

                if ( [g filled] )
                    continue;
                /* we expect only lines or polylines here */
                begP1 = [g1 pointWithNum:0];
                endP1 = [g1 pointWithNum:MAXINT];
                if (   DiffPoint(endP, begP1) <= dist || DiffPoint(endP, endP1) <= dist
                    || DiffPoint(begP, endP1) <= dist || DiffPoint(begP, begP1) <= dist )
                {
                    [g join:g1];
                    [list removeObjectAtIndex:j];
                    j--;
                    begP = [g pointWithNum:0];
                    endP = [g pointWithNum:MAXINT];
                    if (DiffPoint(begP, endP) <= TOLERANCE)
                    {   [g setFilled:YES];
                        [g setFillColor:[NSColor lightGrayColor]];
                        [g setColor:[NSColor blackColor]];
                        [g setWidth:0.1];
                    }
                    removed++;
                }
            }
            if (removed)
                i--;
        }
    }
    /* fill everything */
    for (i=0; i<(int)[list count]; i++)
    {   VPolyLine	*g = [list objectAtIndex:i];

        if (![g isKindOfClass:[VPolyLine class]] || [g filled])
            continue;

        [g setFilled:YES];
        [g setFillColor:[NSColor lightGrayColor]];
        [g setColor:[NSColor blackColor]];
        [g setWidth:0.1];
    }
    /* make lakes blue (PolyLines inside PolyLines are all lakes) */
    for (i=0; i<(int)[list count]-1; i++)
    {   VPath	*g = [list objectAtIndex:i];
        NSRect	r = [g coordBounds], r1;

        if ( ![g filled] || [g fillColor] != [NSColor lightGrayColor]
             || r.size.width * r.size.height < 1000.0 )
            continue;

        for (j=i+1; j<(int)[list count]; j++)
        {   VPolyLine		*g1 = [list objectAtIndex:j];
            NSAutoreleasePool	*pool;

            if (i == j || ![g1 filled])
                continue;
            r1 = [g1 coordBounds];
            pool = [NSAutoreleasePool new];
            if ([g isPointInside:[g1 pointWithNum:0]])
            {
                [g1 setFillColor:waterColor];
                [g1 setWidth:0.0];
            }
            [pool release];
        }
    }

    /* create the ocean */
    {   VRectangle	*g;

        list = [listDict objectForKey:@"Ocean"];
        g = [VRectangle rectangleWithOrigin:NSZeroPoint size:bRect.size];
        [g setFilled:YES];
        [g setFillColor:waterColor];
        [list addObject:g];
    }

    return listDict;
}

- (int)convertScreenPointToLng:(float*)lng :(float*)lat :(NSPoint)pt
{   NSPoint	p = pt;
    float	la, lg;

    *lat = 50.783; *lng = 6.083;

    // mirror around mp
    p.y = mp.y - (p.y - mp.y); // mirror pts !!!

    // dots to sec
    p.x = (((int)(p.x * ms->msecPerDot)) >> MAGSHIFT);
    p.y = (((int)(p.y * ms->msecPerDot)) >> MAGSHIFT);

    // offset
    p.x -= ms->halfWidSec;
    p.y = ms->canvasNSec - p.y;

    // rewind mercator for y
    la = computeMercatorYBack(p.y); // 48.0 - must be 197491 befor
    lg = SEC_TO_DEG(p.x) + ms->centerLng;

    if ( la > 90.0 || la <-90.0 || lg > 180.0 || lg < -180.0 )
        return NO;

    *lat = la;
    *lng = lg;

    return YES;
}

- (float)mercatorLatitude:(float)latitude
{
    return SEC_TO_DEG(computeMercatorY(DEG_TO_SEC(latitude)));
}


/* return 0 lat/lon in dots
 */
- (NSPoint)origin
{   NSPoint	origin;

    origin.x = SEC_TO_DOTS(ms, ms->halfWidSec - DEG_TO_SEC(ms->centerLng));
    origin.y = SEC_TO_DOTS(ms, ms->canvasNSec - computeMercatorY(0.0));
    origin.y = mp.y - (origin.y - mp.y);	// mirror
    return origin;
}
/* scale - earth/map
 */
- (double)scale
{   double	scale = (OneDegInM / 25.4 * 72000.0) / DOTS_PER_DEG(ms);

    NSLog(@"1DegInM = %f,  dotsPerDeg = %f, scale = %f %f", OneDegInM, DOTS_PER_DEG(ms), (OneDegInM * 72000.0 / 25.4) / DOTS_PER_DEG(ms), scale);
    return scale;
}


- (void)dealloc
{
    free(ms->mapDir);
    free(ms);
    [super dealloc];
}

@end
