static char rcsid[] = "$Id: ciamap.c,v 3.9 1992/03/03 01:15:01 putz Exp $";
/*****************************************************************************
 * ciamap.c - This program reads a map database file in the CIA World Base
 *	format and produces a netmap-format output file.
 *
 * Originally created by Brian Reid (reid@pa.dec.com)
 * Modified March 1991 by Steve Putz (putz@parc.xerox.com)
 *****************************************************************************/

#include <stdio.h>
#include <math.h>
#include <ctype.h>
#include <strings.h>
#include <sys/file.h>
#include <sys/param.h>		/* for MAXPATHLEN */
#include "cbdmap.h"

#define	BIT32	int		/* change these definitions to suit your */
#define	BIT16	short		/* own machine				 */
#define	BIT8	char
#define	TRUE	1
#define	FALSE	0
/*
 * The data-compression scheme uses integer seconds to represent lat/long,
 * and stores each stroke as a [dx,dy] from the previous point. If dy will
 * fit into 8 bits and dx into 7 bits, then the entire [dx,dy] is stored in a
 * 16-bit field with bit 0x4000 turned on as a flag. If either value is too
 * large for that scheme, then both are stored as 32-bit values, with the
 * 0x40000000 bit turned off (even in negative numbers) in the first of them.
 */

#define	MAXSEG	20000		/* maximum strokes in a segment */
#define	MAXSEGS 10000

#define abs(x)	(x<0 ? -(x) : x)
#define sign(x) ((x>=0) ? 1 : -1)

struct seghead  sb;

BIT16           segbuf[MAXSEG];

struct segdict  sd[MAXSEGS];

int             MapDetail = 99;

int             rightShift = 0;
int             filterBits = 0;
int             paethFlag = 0;
int             debug = 0;

#define PUTSHORT(val, file) { putc((val)>>8, file); putc((val)&0xFF, file); }
#define PUTCOORD(seconds, file) PUTSHORT((seconds)/20, file)

/* ------------------------------------------------------------------------ */

main(argc, argv)
    int             argc;
    char          **argv;
{
    extern int      optind;
    extern char    *optarg;
    extern char    *strcpy();
    int             c, initdone, badopts;

    initdone = 0;
    badopts = 0;
    while ((c = getopt(argc, argv, "D:d:ps:f:m")) != EOF)
	switch (c) {

	    /* -D: debug. Argument is an integer debuging flags mask. */
	case 'D':
	    debug = strtol(optarg, NULL, 0);
	    break;
	    /*
	     * -d: detail. Argument is an integer giving map detail to
	     * include.
	     */
	case 'd':
	    sscanf(optarg, "%d", &MapDetail);
	    break;
	    /* -p: Paeth format. Produce Paeth format map files.  */
	case 'p':
	    paethFlag = TRUE;
	    break;
	    /*
	     * -s: Scale data. Argument is number of bits to right shift
	     * coordinate data.
	     */
	case 's':
	    sscanf(optarg, "%d", &rightShift);
	    break;
	    /* -f: Filter data. Argument is number of bits to smooth. */
	case 'f':
	    sscanf(optarg, "%d", &filterBits);
	    break;
	    /* -m: Ascii map files follow. "-m f2 f2 f3"			 */
	case 'm':
	    for (; optind < argc && argv[optind][0] != '-'; optind++) {
		if (!initdone);
		initdone++;
		TranslateMap(argv[optind]);
	    }
	case '?':
	    badopts++;
	}
    fflush(stdout);
    if (badopts) {
	exit(2);
    }
}


/* ------------------------------------------------------------------------ */

TranslateMap(name)
    char           *name;
{
    FILE           *df;
    char            line[100], of[MAXPATHLEN], *ext,
		    lad,	/* 'N' or 'S' */
                    lod;	/* 'E' or 'W' */
    int             Ofile = -1, first, lsn, irnk, nop, ladg, lam, las,
                    lodg, lom, los,
		    ipn,	/* polygon number */
                    big = 0, small = 0, medium = 0, skippedSegs = 0,
		    totalStrokes = 0, skippedStrokes = 0, lnum, nc,
                    i, j, strokecount, segmax, segsize, segcount, bytecount;
    BIT32           jlat,	/* integer latitude in seconds */
                    jlong,	/* integer longitude in seconds */
                    haveLat = 0, haveLong = 0, dx, dy;
    struct cbdhead  header;
    FILE           *paethIndex = NULL, *paethData = NULL;
    int             paethCount, paethTotal = 0;
    int             headerSize = CBD_HEADSIZE2;
    long            roundBit = 1 << (rightShift - 1);

    if (!(df = fopen(name, "r"))) {
	perror(name);
	return;
    }
    strcpy(of, name);
    ext = index(of, '.');
    if (ext)
	*ext = '\0';
    else
	ext = of + strlen(of);

    if (paethFlag) {
	strcpy(ext, ".index");
	paethIndex = fopen(of, "w");
	strcpy(ext, ".map");
	paethData = fopen(of, "w");
	if (paethIndex == NULL || paethData == NULL) {
	    perror(of);
	    return;
	}
    } else {
	if (rightShift) {
	    sprintf(ext, "%+d", -rightShift);
	    ext += strlen(ext);
	}
	if (filterBits) {
	    sprintf(ext, "f%d", filterBits);
	    ext += strlen(ext);
	}
	strcpy(ext, ".cbd");
	Ofile = open(of, O_RDWR | O_CREAT | O_TRUNC, 0666);
	if (Ofile < 0) {
	    perror(of);
	    return;
	}
    }

    fprintf(stderr, "Processing map file %s\n", name);

    bzero(&header, sizeof(header));
    /* note that header.magic is zero until we are done */
    header.scale_shift = -rightShift;
    header.minlat = DEG90_SECS;
    header.maxlat = -DEG90_SECS;
    header.minlong = DEG180_SECS;
    header.maxlong = -DEG180_SECS;

    if (Ofile >= 0) {		/* reserve space for header */
	write(Ofile, &header, headerSize);
    }
    segcount = 0;
    segmax = 0;

    /* The header line tells us  the serial number, rank, and path size */
    lnum = 0;
    while (fgets(line, 60, df)) {
	lnum++;
	for (nc = 0; nc < 18 && line[nc] != NULL; nc++);

	if (nc < 18)
	    continue;

	sscanf(line, "%7d%2d%6d", &lsn, &irnk, &nop);

	if (irnk > MapDetail) {
	    for (i = 1; i <= nop; i++) {
		fgets(line, 60, df);
		lnum++;
	    }
	} else {
	    sb.id = lsn;
	    first = FALSE;
	    if (segcount > MAXSEGS) {
		fprintf(stderr, "Segment count overflow.\n");
		close(Ofile);
		fclose(df);
		return;
	    }
	    if (Ofile >= 0)
		sd[segcount].absaddr = tell(Ofile);
	    sd[segcount].segid = lsn;
	    sd[segcount].rank = irnk;

	    sd[segcount].minlat = DEG90_SECS;
	    sd[segcount].maxlat = -DEG90_SECS;
	    sd[segcount].minlong = DEG180_SECS;
	    sd[segcount].maxlong = -DEG180_SECS;

	    bytecount = 0;
	    segsize = 0;
	    paethCount = 0;
	    for (i = 1; i <= nop; i++) {
		fgets(line, 60, df);
		lnum++;
		for (j = 0; j <= 19; j++) {
		    if (line[j] == ' ')
			line[j] = '0';
		}
#define DIG(x) ((int)(x)-48)
		ladg = 10 * DIG(line[0]) + DIG(line[1]);
		lam = 10 * DIG(line[2]) + DIG(line[3]);
		las = 10 * DIG(line[4]) + DIG(line[5]);
		lad = line[6];
		lodg = 100 * DIG(line[7]) + 10 * DIG(line[8]) + DIG(line[9]);
		lom = 10 * DIG(line[10]) + DIG(line[11]);
		los = 10 * DIG(line[12]) + DIG(line[13]);
		lod = line[14];
		ipn = 1000 * DIG(line[16]) + 100 * DIG(line[17]) + 10 *
			DIG(line[18]) + DIG(line[19]);

		if (ipn != i) {
		    fprintf(stderr, "Data error at sequence %d, segment %d: line %d expected but %d found:\n%s\n",
			lnum, lsn, i, ipn, line);
		    if (ipn > 0 && ipn < nop)
			i = ipn;
		}
		jlat = 3600 * ladg + 60 * lam + las;
		if (lad == 'S')
		    jlat = -jlat;
		jlong = 3600 * lodg + 60 * lom + los;
		if (lod == 'W')
		    jlong = -jlong;
		if (debug & 0x04)
		    fprintf(stderr, "(%.4f %.4f) = (%d %d) [%x %x]",
		     jlong / 3600.0, jlat / 3600.0, jlong, jlat, jlong, jlat);
		if (rightShift > 0) {
		    if (jlong & roundBit) {
			jlong = (jlong >> rightShift) + 1;
			if (debug & 0x20) fprintf(stderr, " x%+1 ");
		    } else {
			jlong >>= rightShift;
		    }
		    if (jlat & roundBit) {
			jlat = (jlat >> rightShift) + 1;
			if (debug & 0x20) fprintf(stderr, " y%+1 ");
		    } else {
			jlat >>= rightShift;
		    }
		    if (debug & 0x10)
			fprintf(stderr, "-> (%d %d) [%x %x]", jlong, jlat, jlong, jlat);
		}
		if (debug & 0x04)
		    fprintf(stderr, "\n");
		if (sd[segcount].maxlat < jlat)
		    sd[segcount].maxlat = jlat;
		if (sd[segcount].minlat > jlat)
		    sd[segcount].minlat = jlat;
		if (sd[segcount].maxlong < jlong)
		    sd[segcount].maxlong = jlong;
		if (sd[segcount].minlong > jlong)
		    sd[segcount].minlong = jlong;
		if (!first) {
		    first = TRUE;
		    sb.orgx = jlong;
		    sb.orgy = jlat;
		    strokecount = 0;
		} else {
		    dx = jlong - haveLong;
		    dy = jlat - haveLat;
		    if (dx == 0 && dy == 0 ||
				i < nop && abs(dx) + abs(dy) <= filterBits) {
			skippedStrokes++;
			if (debug & 0x02)
			    fprintf(stderr, "skipped: (%d %d)\n", dx, dy);
			continue;	/* don't set haveLat, haveLong */
		    } else {
			totalStrokes++;
			strokecount++;
			if (abs(dx) > MAX8x || abs(dy) > MAX8y) {
			    /*
			     * Encode this transition as a pair of 32-bit
			     * values.
			     */
			    big++;
			    if (abs(dx) <= 0x3FFF && abs(dy) <= 0x7FFF)
				medium++;
			    bytecount += 8;
			    if (segsize + 4 > MAXSEG) {
				fprintf(stderr, "segsize overflow.\n");
				close(Ofile);
				fclose(df);
				return;
			    }
			    segbuf[segsize++] = (dx >> 16) & ~SHORTFLAG;
			    segbuf[segsize++] = dx & 0xFFFF;
			    segbuf[segsize++] = (dy >> 16);
			    segbuf[segsize++] = dy & 0xFFFF;
			    if (debug & 0x02)
				fprintf(stderr, "%4d: long (%d %d): %04x %04x %04x %04x\n", strokecount, dx, dy, segbuf[segsize - 4] & 0xFFFF, segbuf[segsize - 3] & 0xFFFF, segbuf[segsize - 2] & 0xFFFF, segbuf[segsize - 1] & 0xFFFF);
			} else {
			    /*
			     * Encode this transition as a pair of 8-bit
			     * values.
			     */
			    small++;
			    bytecount += 2;
			    if (segsize + 1 > MAXSEG) {
				fprintf(stderr, "segsize overflow.\n");
				close(Ofile);
				fclose(df);
				return;
			    }
			    segbuf[segsize++] =
				SHORTFLAG |
				(dx << 8) |
				(dy & 0xFF);
			    if (debug & 0x02)
				fprintf(stderr, "%4d: short (%d %d): %04x\n", strokecount, dx, dy, segbuf[segsize - 1] & 0xFFFF);
			}	/* short encoding */
			if (paethFlag) {
			    PUTCOORD(jlat, paethData);
			    PUTCOORD(jlong, paethData);
			    paethCount++;
			}
		    }		/* dx,dy != 0 */
		}		/* not first */
		haveLat = jlat;
		haveLong = jlong;
	    }

	    if (strokecount == 0) {
		skippedSegs++;	/* reuse same segment */
	    } else if (Ofile >= 0) {
		sd[segcount].nbytes = bytecount;
		sb.nstrokes = strokecount;
		write(Ofile, &sb, sizeof sb);
		write(Ofile, segbuf, bytecount);
		if (debug & 0x01)
		    fprintf(stderr, "seg %d: (%.4f %.4f)=(%d %d) %d strokes %d bytes\n", sb.id, (sb.orgx << rightShift) / 3600.0, (sb.orgy << rightShift) / 3600.0, sb.orgx, sb.orgy, sb.nstrokes, bytecount);
		if (debug & 0x01)
		    fprintf(stderr, "seg NSEW = (%.4f %.4f %.4f %.4f)\n",
			    (sd[segcount].maxlat << rightShift) / 3600.0,
			    (sd[segcount].minlat << rightShift) / 3600.0,
			    (sd[segcount].maxlong << rightShift) / 3600.0,
			    (sd[segcount].minlong << rightShift) / 3600.0);
		header.features |= (1 << (sd[segcount].rank - 1));
		if (header.maxlat < sd[segcount].maxlat)
		    header.maxlat = sd[segcount].maxlat;
		if (header.minlat > sd[segcount].minlat)
		    header.minlat = sd[segcount].maxlat;
		if (header.maxlong < sd[segcount].maxlong)
		    header.maxlong = sd[segcount].maxlong;
		if (header.minlong > sd[segcount].minlong)
		    header.minlong = sd[segcount].minlong;
		segcount++;
	    } else if (paethFlag) {
		PUTCOORD(sd[segcount].maxlat, paethIndex);
		PUTCOORD(sd[segcount].minlat, paethIndex);
		PUTCOORD(sd[segcount].maxlong, paethIndex);
		PUTCOORD(sd[segcount].minlong, paethIndex);
		PUTSHORT(paethCount, paethIndex);
		paethTotal += paethCount;
	    }
	}
	/* end of each segment */

	if (segsize > segmax)
	    segmax = segsize;
    }
    /* end of file */
    header.magic = headerSize == CBD_HEADSIZE2 ? CBD_MAGIC2 : CBD_MAGIC;
    header.dictaddr = tell(Ofile);
    header.segcount = segcount;
    header.segsize = segcount * sizeof(struct segdict);
    header.segmax = segmax;

    fprintf(stderr,
	    "Segdict has %d entries (%d skipped), %d bytes, starts at %d.\n\
	%d strokes (%d skipped)\n\
	Largest segment has %d entries (%d bytes)\n\
	%d small entries, %d big (%.2f%% of space), %d medium\n",
	    segcount, skippedSegs, header.segsize, header.dictaddr,
	    totalStrokes, skippedStrokes,
	    segmax, 2 * segmax,
	    small, big, (double) 100 * big * 4 / (big * 4 + small), medium);
    {	int i;
	fprintf(stderr, "features: ");
	for (i = 1; i <= 32; i++)
	    if (header.features & (1<<(i-1))) fprintf(stderr, " %d", i);
	fprintf(stderr, "\n");
    }
    if (paethFlag) {
	fprintf(stderr, "\t%d paeth points\n", paethTotal);
	fclose(paethData);
	fclose(paethIndex);
    } else {
	write(Ofile, sd, (int) header.segsize);
	lseek(Ofile, 0L, L_SET);
	write(Ofile, &header, headerSize);
	close(Ofile);
    }
    fclose(df);
}
