/* APMap.m
 * Controls the generation of maps
 *
 * Copyright (C) 2001-2007 by vhf interservice GmbH
 * Author:   Georg Fleischmann
 *
 * Created:  2001-05-18
 * Modified: 2007-09-03
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the vhf Public License as
 * published by vhf interservice GmbH. Among other things, the
 * License requires that the copyright notices and this notice
 * be preserved on all copies.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the vhf Public License for more details.
 *
 * You should have received a copy of the vhf Public License along
 * with this program; see the file LICENSE. If not, write to vhf.
 *
 * vhf interservice GmbH, Im Marxle 3, 72119 Altingen, Germany
 * eMail: info@vhf.de
 * http://www.vhf.de
 */

#include <VHFShared/vhfCommonFunctions.h>
#include <VHFShared/VHFDictionaryAdditions.h>
#include "../Cenon/App.h"
#include "../DocViewMap.h"
#include "../Cenon/locations.h"
#include "../Cenon/messages.h"
#include "../Cenon/functions.h"
#include "../AstroPrincipal.h"
#include "../Astro.bproj/AstroController.h"	// Preferences
#include "../AstroDate.h"			// date formats
#include "../astroCommon.h"
#include "../astroLocations.h"
#include "../astroMessages.h"
#include "AstroPanel.h"
#include "APMap.h"

@interface APMap(PrivateMethods)
- (void)loadMap;
@end

@implementation APMap

- init
{
    [self setDelegate:self];

    objectList = [[NSArray alloc] initWithObjects:@"Sun", @"Moon", @"Mercury", @"Venus", @"Mars",
                                                  @"Jupiter", @"Saturn", @"Uranus", @"Neptune", @"Pluto",
                                                  @"True Node", @"True Apogee", @"Chiron", @"MC", nil];
    [self loadMap];

    return [super init];
}

- (NSCalendarDate*)dateWithString:(NSString*)string
{   NSString		*dateFormat = [[AstroDate sharedInstance] dateFormatForString:string];
    NSCalendarDate	*date = [NSCalendarDate dateWithString:string calendarFormat:dateFormat];

    if (!date)
        NSRunAlertPanel(@"", WRONGDATEFORMAT_STRING, OK_STRING, nil, nil, astroErrorDate());
    return date;
}

/* created: 2006-02-20
 */
- (void)setSwitch:(id)sender
{   BOOL	enableTimeTrack = NO;

    if ([(NSCell*)[showSwitches cellAtRow:SWITCH_SHOWTRACKS column:0] state])
        enableTimeTrack = YES;
    [earthSwitch setEnabled:enableTimeTrack];
    [dateFrField setEnabled:enableTimeTrack];
    [dateToField setEnabled:enableTimeTrack];
}

- (void)set:(id)sender
{   Document            *doc = [(App*)NSApp currentDocument];
    DocView             *view;
    NSDictionary        *mDict = [mapDict objectForKey:[regionPopup titleOfSelectedItem]];
    NSPoint             center;
    NSSize              size;
    float               gridSize;
    NSMutableDictionary *features = [NSMutableDictionary dictionary];
    NSMutableArray      *objects = [NSMutableArray array];
    int                 objIx = (int)[objectPopup indexOfSelectedItem] - 1, i;

    /* search a map or load default map */
    if ( ![[doc name] hasPrefix:MAP_PREFIX] )
    {   int	i;

        for ( i=[[NSApp windows] count]-1; i>=0; i-- )
        {   doc = [(App*)NSApp documentInWindow:[[(App*)NSApp windows] objectAtIndex:i]];
            if ( [[doc name] hasPrefix:MAP_PREFIX] )
                break;
        }
        if ( i < 0 )
        {   NSFileManager	*fileManager = [NSFileManager defaultManager];
            NSString		*path = nil, *relPath;
            NSString		*file = [MAP_FILE stringByAppendingPathExtension: DOCUMENT_EXT];

            relPath = vhfPathWithPathComponents(@"Astro", @"Maps", file, nil);
            for (i=0; i<3; i++)
            {
                switch (i)
                {
                    case 0:	// Home Library
                        path = vhfPathWithPathComponents(userLibrary(),  relPath, nil);
                        break;
                    case 1:	// Local Library
                        path = vhfPathWithPathComponents(localLibrary(), relPath, nil);
                        break;
                    default:
                        NSLog(@"APMap: No document found with prefix '%@'!", MAP_PREFIX);
                        return;
                }
                if ( [fileManager fileExistsAtPath:path] )
                    break;
            }
            [(App*)NSApp openFile:path];	// open default Flow Chart
            doc = [(App*)NSApp openedDocument];
        }
    }
    view = [doc documentView];


    /* center lat/long , w/h degree -> mapDict with selected item of mapPopUp */
    center.x    = [mDict floatForKey:@"centerLng"];
    center.y    = [mDict floatForKey:@"centerLat"];
    size.width  = [mDict floatForKey:@"widthLng"];
    size.height = [mDict floatForKey:@"heightLat"];

    /* grid size -> gridPopUp selected item */
    gridSize = [[gridPopup titleOfSelectedItem] floatValue];

    /* stuff to display */
    if ([(NSCell*)[showSwitches cellAtRow:SWITCH_SHOWACMC column:0] state])
        [features setObject:@"1" forKey:@"featureACMC"];
    if ([(NSCell*)[showSwitches cellAtRow:SWITCH_SHOWHOUSES column:0] state])
        [features setObject:@"1" forKey:@"featureHouses"];
    if ([(NSCell*)[showSwitches cellAtRow:SWITCH_SHOWZODIAC column:0] state])
        [features setObject:@"1" forKey:@"featureZodiac"];
    if ([(NSCell*)[showSwitches cellAtRow:SWITCH_SHOWDEGREES column:0] state])
        [features setObject:@"1" forKey:@"featureDegrees"];
    if ([(NSCell*)[showSwitches cellAtRow:SWITCH_SHOWTRACKS column:0] state])
        [features setObject:@"1" forKey:@"featureTimeTracks"];

    if ([(NSButton*)earthSwitch state])
        [features setObject:@"1" forKey:@"earthRev"];

    /* objects to display
     * if object2 == Node or Apogee, we have to check, if true or mean
     */
    if (objIx < 0 )	// All objects
    {
        for (i=0; i<[objectList count]; i++)
            [objects addObject:[objectList objectAtIndex:i]];
    }
    else if (objIx < [objectList count])    // single object
        [objects addObject:[objectList objectAtIndex:objIx]];
    /* set mean or true node/apogee */
    for (i=0; i<[objects count]; i++)
    {   NSString	*object = [objects objectAtIndex:i];
        BOOL		objChanged = NO;

        if ( ! [Prefs_Objects containsObject:object] )
        {
            if ([object isEqual:@"True Node"])
                object = @"Node",   objChanged = YES;	// Mean Node
            else if ([object isEqual:@"True Apogee"])
                object = @"Apogee", objChanged = YES;	// Mean Apogee
            if (objChanged)
                [objects replaceObjectAtIndex:i withObject:object];
        }
    }
    [features setObject:objects forKey:@"objects"];	// objects we have to map

    /* date range */
    [features setObject:[self dateWithString:[dateFrField stringValue]] forKey:@"dateFr"];
    [features setObject:[self dateWithString:[dateToField stringValue]] forKey:@"dateTo"];

    [view createMapWithCenter:center size:size grid:gridSize features:features];
}

#if 0
- (void)valuesToRadixAC:(float)radixAC atLat:(float)lat lon:(float)lon
{   int			sign;
    float		dlon23, dlatM, dlon, mLat, m48, latDiff, lonDiff;
    float		dia, horc, diff, cval, degree; // zodiac values of lat lon
    NSDictionary	*destinyDict = [NSDictionary dictionaryWithContentsOfFile:[NSString stringWithFormat:@"%@/MapLines.dict", userLibrary()]], *curDict=nil;

printf("Latitude: %.2f\n", lat);
printf("Longitude: %.2f\n", lon);
    // 48.0 -> lion horc   48.0 - ((360.0/22.0/12.0)/30.0 * 120) -> 0 Widder = 42.545454545454
    // (360.0/22.0/12.0) = 1.363636...grad = 30 Tgrade
    // (360.0/22.0/12.0) * 12 = 1 Tierkreis = 16.363636 drad
    latDiff = (lat >= 0.0) ? (Diff(48.0, lat)) : (-lat + 48.0);
    horc = latDiff / ((360.0/22.0/12.0)/30.0); // lat grad to Tgrad
    horc = ( lat >= 48.0 ) ? (horc + 120.0) : (120.0 - horc); //  + 0 Lion (120.0)
    while (horc > 360.0) horc -= 360.0;
    while (horc < 0.0)   horc += 360.0;
printf("Breite: %.2f\n", horc);
    // dia 12.0 -> widder
    // dy 1 - dx tan(23.433)
    dia = 1.0;
    mLat = [[[[(App*)NSApp currentDocument] documentView] ciaMap] mercatorLatitude:lat];
    m48  = [[[[(App*)NSApp currentDocument] documentView] ciaMap] mercatorLatitude:48.0];

    dlatM = (lat >= 0.0) ? (Diff(mLat, m48)) : (-mLat + 48.0);
    dlon = (lon >= 0.0) ? (Diff(lon, 12.0245098)) : (-lon + 12.0245098);
    dlon23 = dlatM * Tan(23.4333);
    lonDiff = ( (lon < 12.0245098  && lon + dlon23 < 12.0245098+(360.0/15.0/12.0) && lat > 48.0) ||
                (lon >= 12.0245098 && lat < 48.0) ) ? (Diff(dlon, dlon23)):(dlon+dlon23);
    dia = lonDiff / ((360.0/15.0/12.0)/30.0); // lon grad to Tgrad
    while (dia > 360.0) dia -= 360.0;
    if ( (lat > 48.0 && lon + dlon23 < 12.0245098) || (lat < 48.0 && lon - dlon23 < 12.0245098) )
        dia = 360.0 - dia; //  other direction
printf("Schraege: %.2f\n", dia);

    // search  sign angle
    sign = (int)(dia/30.0);
    degree = dia;
    while (degree > 30.0)
        degree -= 30.0;
    [diagonal setStringValue:[NSString stringWithFormat:@"%.1f %@", degree, zodiacSign(sign)]]; //@"Astro"
    // search  sign angle
    sign = (int)(horc/30.0);
    degree = horc;
    while (degree > 30.0)
        degree -= 30.0;
    [horicontal setStringValue:[NSString stringWithFormat:@"%.1f %@", degree, zodiacSign(sign)]];

    // difference
    diff = Max(horc, dia) - Min(horc, dia);
    if (diff > 180.0)
        diff = 360.0 - diff;
    // search  sign angle
    // cval/30.0 < 1 -> widder
    sign = (int)(diff/30.0);
    degree = diff;
    while (degree > 30.0)
        degree -= 30.0;
    [difference setStringValue:[NSString stringWithFormat:@"%.1f %@", degree, zodiacSign(sign)]];

    // cityVal
    cval = Max(diff, radixAC) - Min(diff, radixAC);
    if (cval > 180.0)
        cval = 360.0 - cval;
    // search  sign angle
    // cval/30.0 < 1 -> widder
    sign = (int)(cval/30.0);
    degree = cval;
    while (degree > 30.0)
        degree -= 30.0;
    [cityValue setStringValue:[NSString stringWithFormat:@"%.1f %@", degree, zodiacSign(sign)]];

    // criticalDegree
    [criticalDegree setStringValue:[NSString stringWithFormat:@""]]; // empty textfield
    switch (sign)
    {
        case  0: curDict = [destinyDict objectForKey:@"widder"];     break;
        case  1: curDict = [destinyDict objectForKey:@"stier"];      break;
        case  2: curDict = [destinyDict objectForKey:@"zwillinge"];  break;
        case  3: curDict = [destinyDict objectForKey:@"krebs"];      break;
        case  4: curDict = [destinyDict objectForKey:@"loewe"];      break;
        case  5: curDict = [destinyDict objectForKey:@"jungfrau"];   break;
        case  6: curDict = [destinyDict objectForKey:@"waage"];      break;
        case  7: curDict = [destinyDict objectForKey:@"skorpion"];   break;
        case  8: curDict = [destinyDict objectForKey:@"schuetze"];   break;
        case  9: curDict = [destinyDict objectForKey:@"steinbock"];  break;
        default:
        case 10: curDict = [destinyDict objectForKey:@"wassermann"]; break;
        case 11: curDict = [destinyDict objectForKey:@"fisch"];      break;
    }
    if (curDict)
    {   int	i, keyCnt;
        NSArray	*keyArray;

        // search in curDict for a value like degree (1 diff)
        keyArray = [curDict allKeys];
        keyCnt = [keyArray count];
        for (i=0; i<keyCnt; i++)
        {   float	deg = [[keyArray objectAtIndex:i] floatValue];

            if (Diff(deg, degree) < 1.0)
            {   [criticalDegree setStringValue:[NSString stringWithFormat:@"%@", [curDict objectForKey:[keyArray objectAtIndex:i]]]];
                break;
            }
        }
    }
}
#endif

- (void)loadMap
{   NSString	*path = [NSString stringWithFormat:@"%@/%@/%@", userLibrary(), MODULENAME, MAP_INFOFILE];
    int		i, cnt;
    NSArray	*keys;

    [mapDict release];
    mapDict = [[NSDictionary dictionaryWithContentsOfFile:path] retain];
    if ( !mapDict )
    {   NSMutableDictionary	*dict = [NSMutableDictionary dictionary];

	//NSRunAlertPanel(@"", CANTLOADFILE_STRING, OK_STRING, nil, nil, path);
        [dict setObject: @"11.0" forKey:@"centerLng"];
        [dict setObject:  @"0.0" forKey:@"centerLat"];
        [dict setObject:@"360.0" forKey:@"widthLng"];
        [dict setObject:@"180.0" forKey:@"widthLat"];
        mapDict = [[NSDictionary dictionaryWithObject:dict forKey:@"World"] retain];
    }

    keys = [mapDict allKeys];

    /* update pop up */
    [regionPopup selectItemAtIndex:0];		// select surviving item (GNUstep)
    [regionPopup removeAllItems];
    for (i=0, cnt=[keys count]; i<cnt; i++)
    {
        if ([[keys objectAtIndex:i] isEqual:@"mapset"])
            continue;
        [regionPopup addItemWithTitle:[keys objectAtIndex:i]];
    }
    [regionPopup selectItemAtIndex:0];
}

/* created:  2001-05-18
 * modified: 2006-02-19
 */
- (void)dealloc
{
    [objectList release];
    [mapDict release];
    [super dealloc];
}

@end
