/* TopocentricCycle.m
 * Topocentric houses and related functions
 *
 * This file uses some functions from the
 * Astrolog free Astrology software by Walter D. Pullen
 * (http://www.magitech.com/~cruiser1/astrolog.htm)
 *
 * The extensions are
 * Copyright (C) 1999-2004 by vhf interservice GmbH
 * Author:   Georg Fleischmann
 *
 * created:  2003-08-11
 * modified: 2004-06-26
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the vhf Public License as
 * published by vhf interservice GmbH. Among other things, the
 * License requires that the copyright notices and this notice
 * be preserved on all copies.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the vhf Public License for more details.
 *
 * You should have received a copy of the vhf Public License along
 * with this program; see the file LICENSE. If not, write to vhf.
 *
 * vhf interservice GmbH, Im Marxle 3, 72119 Altingen, Germany
 * eMail: info@vhf.de
 * http://www.vhf.de
 */

#include <math.h>
#include <VHFShared/types.h>
#include "astroCommon.h"
#include "TopocentricCycle.h"

#define RFract(r)	((r) - floor(r))

@implementation TopocentricCycle

+ (TopocentricCycle*)topocentricCycleAtDate:(NSCalendarDate*)utc
                                   latitude:(float)lat longitude:(float)lon
{   TopocentricCycle	*cycle = [[self alloc] initWithDate:utc latitude:lat longitude:lon];
    return [cycle autorelease];
}

- (id)initWithDate:(NSCalendarDate*)utc latitude:(float)lat longitude:(float)lon
{
    [super init];
    [self setDate:utc latitude:lat longitude:lon];
    return self;
}

/* Determine the sign of a number: -1 if value negative, +1 if value
 * positive, and 0 if it's zero.
 */
static double RSgn(double r)
{
  return r == 0.0 ? 0.0 : ((r) < 0.0 ? -1.0 : 1.0);
}

/* Modulus function to bring the given
 * parameter into the range of 0 to 360
 */
static double Mod(double d)
{
    if (d >= 360.0)
        d -= 360.0;
    else if (d < 0.0)
        d += 360.0;
    if (d >= 0 && d < 360.0)
        return d;
    return (d - floor(d/360.0)*360.0);
}
/* Modulus function for the range of 0 to 2 Pi */
static double ModRad(double r)
{
    while (r >= Pi*2.0)
        r -= Pi*2.0;
    while (r < 0.0)
        r += Pi*2.0;
    return r;
}

/* Given an x and y coordinate, return the angle formed by a line from the
 * origin to this coordinate. This is just converting from rectangular to
 * polar coordinates; however, we don't determine the radius here
 */
static double Angle(double x, double y)
{   double a;

    if (x != 0.0)
    {
        if (y != 0.0)
            a = atan(y/x);
        else
            a = (x < 0.0) ? Pi : 0.0;
    }
    else
        a = (y < 0.0) ? -Pi/2.0 : Pi/2.0;
    if (a < 0.0)
        a += Pi;
    if (y < 0.0)
        a += Pi;
    return a;
}

//#define doubleSmall	(1.7453E-09)
/* Convert polar to rectangular coordinates. */
/*static void PolToRec(double A, double R, double *X, double *Y)
{
    if (A == 0.0)
        A = doubleSmall;
    *X = R*cos(A);
    *Y = R*sin(A);
}*/

/* Convert rectangular to polar coordinates. */
/*static void RecToPol(double X, double Y, double *A, double *R)
{
    if (Y == 0.0)
        Y = doubleSmall;
    *R = sqrt(X*X + Y*Y);
    *A = Angle(X, Y);
}*/

static double CuspTopocentric(TOPO topo, double deg)
{   double oa = ModRad(topo.RA + DegToRad(deg));
    double X  = atan(tan(topo.lat) / cos(oa));
    double lo = atan(cos(X) * tan(oa) / cos(X+topo.OB));

    if (lo < 0.0)
        lo += Pi;
    if (sin(oa) < 0.0)
        lo += Pi;
    return lo;
}

static long MdyToJulian(int mon, int day, int yea)
{   long im, j;

    im = 12*((long)yea+4800)+(long)mon-3;
    j = (2*(im%12) + 7 + 365*im)/12;
    j += (long)day + im/48 - 32083;
    if (j > 2299171)			// Take care of dates in
        j += im/4800 - im/1200 + 38;	// Gregorian calendar
    return j;
}

/* The min distance value returned will either be positive or
 * negative based on whether the second value is ahead or behind
 * the first one in a circular zodiac
 */
static double MinDifference(double deg1, double deg2)
{   double i;

    i = deg2 - deg1;
    if (fabs(i) < 180.0)
        return i;
    return RSgn(i)*(fabs(i) - 360.0);
}

/* MC
 */
- (double)mc
{   double MC;

    MC = atan(tan(topo.RA)/cos(topo.OB));
    if (MC < 0.0)
        MC += Pi;
    if (topo.RA > Pi)
        MC += Pi;
    return Mod(RadToDeg(MC));
}

/* AC
 */
- (double)ac
{   double asc;

    asc = Angle(-sin(topo.RA)*cos(topo.OB)-tan(topo.lat)*sin(topo.OB), cos(topo.RA));
    return Mod(RadToDeg(asc));
}

/* return topocentric houses with variable number of houses
 * cnt%4 == 0
 * created: 1999-11-30
 */
- (double*)houses:(int)cnt
{   double		tl, savedLat = topo.lat, m, m1 = 0.0;
    static double	houses[360];
    int			i;

    houses[cnt/4] = ModRad(DegToRad(topo.MC + 180.0));
    tl = tan(topo.lat);

    /* 0 - cnt/4     -> 90 - 180, 1 - 0
     * cnt/4 - cnt/2 ->  0 -  90, 1 - 0
     */
    for ( i=0; i<cnt/2; i++ )
    {   double beg = 0.0;

        if ( i == cnt/4 )
            continue;
        m = (double)(i%(cnt/4)) / (double)(cnt/4);	// for cnt == 12: 0, 1/3, 2/3, 0, ...
        if ( i >= 0 && i<= cnt/4 )		// Q1
            beg = 90.0, m1 = 1-m;
        else if ( i >= cnt/4 && i<= cnt/2 )	// Q2
            beg = 0.0, m1 = m;

        topo.lat = atan(tl*m1);			// devide lat: lat, 1/3 lat, 2/3 lat
        houses[i] = CuspTopocentric(topo, beg+90.0*m);
        if ( i >= cnt/4 && i<= cnt/2 )		// 2nd quadrant
            houses[i] += Pi;
    }
    topo.lat = savedLat;

    /* houses of the 2nd half */
    for ( i = 0; i < cnt/2; i++ )
    {
        houses[i] = Mod(RadToDeg(houses[i]));
        houses[i+cnt/2] = Mod(houses[i] + 180.0);
    }

    return houses;
}

/* init parameters
 * created: 2003-08-11
 */
- (void)setDate:(NSCalendarDate*)utc latitude:(float)lat longitude:(float)lon
{   int	yymmdd, hhmm;

    yymmdd = [utc yearOfCommonEra]*10000 + [utc monthOfYear]*100 + [utc dayOfMonth];
    hhmm   = [utc hourOfDay]*100 + [utc minuteOfHour];
    return [self setDate:yymmdd time:hhmm latitude:lat longitude:lon];
}

- (void)setDate:(int)yymmdd time:(int)hhmm latitude:(float)lat longitude:(float)lon
{
    topo.yea = yymmdd/10000;
    topo.mon = yymmdd/100-yymmdd/10000*100;
    topo.day = yymmdd-yymmdd/100*100;
    topo.tim = hhmm/100 + (float)(hhmm-hhmm/100*100)/100.0;
    topo.tim = RSgn(topo.tim)*floor(fabs(topo.tim))+RFract(fabs(topo.tim))*100.0/60.0;

    topo.lat = DegToRad( Max( Min(lat, 89.9999), -89.9999) );
    topo.lon = -lon;

    {   double	jd = (double)MdyToJulian(topo.mon, topo.day, topo.yea);
        double	t = (jd + topo.tim/24.0 - 2415020.5) / 36525.0;	// Julian time for chart

        /* Compute angle that the ecliptic is inclined to the Celestial Equator */
        topo.OB = DegToRad(23.452294 - 0.0130125*t);

        topo.RA = DegToRad(Mod((6.6460656+2400.0513*t+2.58E-5*t*t+topo.tim)*15.0-topo.lon));

        /*{   double R = 1.0, X, Y, A = topo.RA;

            PolToRec(A, R, &X, &Y);
            X *= cos(-topo.OB);
            RecToPol(X, Y, &A, &R);
            topo.MC = Mod(RadToDeg(A));           // Midheaven
        }*/
    }

    topo.MC = [self mc];       // Calculate Ascendant & Midheaven
    topo.AC = [self ac];
    /* Flip the Ascendant if it falls in the wrong half of the zodiac. */
    if (MinDifference(topo.MC, topo.AC) < 0.0)
        topo.AC = Mod(topo.AC + 180.0);
}

- (void)setLatitude:(float)lat longitude:(float)lon
{
    topo.lat = DegToRad( Max( Min(lat, 89.9999), -89.9999) );
    topo.lon = -lon;

    {   double	jd = (double)MdyToJulian(topo.mon, topo.day, topo.yea);
        double	t = (jd + topo.tim/24.0 - 2415020.5) / 36525.0;	// Julian time for chart

        /* Compute angle that the ecliptic is inclined to the Celestial Equator */
        topo.OB = DegToRad(23.452294 - 0.0130125*t);

        topo.RA = DegToRad(Mod((6.6460656+2400.0513*t+2.58E-5*t*t+topo.tim)*15.0-topo.lon));
    }

    topo.MC = [self mc];       // Calculate Ascendant & Midheaven
    topo.AC = [self ac];
    /* Flip the Ascendant if it falls in the wrong half of the zodiac. */
    if (MinDifference(topo.MC, topo.AC) < 0.0)
        topo.AC = Mod(topo.AC + 180.0);
}

/* return topocentric houses with variable number of houses
 * houseCnt % 4 == 0
 * created: 1999-11-30
 */
- (double*)housesAtDate:(NSCalendarDate*)utc latitude:(float)lat longitude:(float)lon number:(int)houseCnt
{
    [self setDate:utc latitude:lat longitude:lon];
    return [self houses:houseCnt];
}

/* return ecliptic position for topocentric position
 * deg = 0 (AC), ..., 30, 60, 90 (IC), 120, ...
 * created: 2003-08-11
 */
- (double)eclipticDegree:(double)deg
{   double	tl, savedLat = topo.lat, m, m1, beg, lDeg;

    if ( Diff(deg, 90.0) < TOLERANCE )
        return Mod(topo.MC+180.0);
    else if ( Diff(deg, 270.0) < TOLERANCE )
        return topo.MC;

    tl = tan(topo.lat);

    m = (deg-floor(deg/90.0)*90.0) / 90.0;
    if ( (deg >= 0.0 && deg <= 90.0) || (deg >= 180.0 && deg <= 270.0) )
        beg = 90.0, m1 = 1-m;
    else
        beg =  0.0, m1 = m;

    topo.lat = atan(tl * m1);
    lDeg = CuspTopocentric(topo, beg + 90.0*m);	// CuspTopocentric(beg + 90.0 * 1/n)
    if ( deg >= 90.0 && deg <= 270.0 ) 		// 2nd + 3rd quadrant
        lDeg += Pi;

    topo.lat = savedLat;
    lDeg = Mod(RadToDeg(lDeg));
    return lDeg;
}

/* return local position for ecliptic position.
 * AC = 0, 2 = 30, 3 = 60, 4 = 90, ... MC = 270, ...
 * created: 2003-08-11
 * FIXME: the algorithm is just a quick and dirty approximation of what we really want.
 *        It should be constructed directly using trigonometric functions.
 */
- (double)topocentricDegree:(double)eclipticDeg
{   int		i;
    double	eDeg, eDegMin;
    double	tDeg, tDegMin = 0.0, tDegMax = 360.0, aTarget;

    if ( Abs(eclipticDeg - topo.MC) < TOLERANCE )			// MC
        return 270.0;
    else if ( Abs(eclipticDeg - Mod(topo.MC+180.0)) < TOLERANCE )	// IC
        return 90.0;

    eDegMin = [self eclipticDegree:tDegMin];
    for (i=0; i<20; i++)
    {
        tDeg = (tDegMin + tDegMax) / 2.0;
        eDeg = [self eclipticDegree:tDeg];
        if (Diff(eDeg, eclipticDeg) < TOLERANCE)
            break;
        else if (Diff(Mod(eDeg+180.0), eclipticDeg) < TOLERANCE)
            return Mod(tDeg+180);
        aTarget = angle(eclipticDeg, eDegMin);
        if (aTarget > angle(eDeg, eDegMin))	// inside 1st half (min < deg < target)
        {   tDegMin = tDeg;
            eDegMin = eDeg;
        }
        else					// inside 2nd half
            tDegMax = tDeg;
    }

    return tDeg;
}

/* test
 */
- (BOOL)testAtDate:(NSCalendarDate*)utc latitude:(float)lat longitude:(float)lon
{   int		i;
    BOOL	error = NO;

    [self setDate:utc latitude:lat longitude:lon];
    for (i=0; i<360; i++)
    {   double	eDeg, tDeg;

        eDeg = [self eclipticDegree:i];
        tDeg = [self topocentricDegree:eDeg];
        printf("%d -> %.3f -> %.3f\n", i, eDeg, tDeg);
        if (Diff(tDeg, (double)i) > 0.1)
        {   NSLog(@"Error!");
            error = YES;
        }
    }
    return error;
}

@end
