/* Map.m
 * Commander of the map
 * This class is associated to a view containing the map.
 *
 * Copyright (C) 2003-2004 by vhf interservice GmbH
 * Author:   Georg Fleischmann, Ilonka Fleischmann
 *
 * created:  2003-07-14
 * modified: 2004-03-19
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the vhf Public License as
 * published by vhf interservice GmbH. Among other things, the
 * License requires that the copyright notices and this notice
 * be preserved on all copies.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the vhf Public License for more details.
 *
 * You should have received a copy of the vhf Public License along
 * with this program; see the file LICENSE. If not, write to vhf.
 *
 * vhf interservice GmbH, Im Marxle 3, 72119 Altingen, Germany
 * eMail: info@vhf.de
 * http://www.vhf.de
 */

#include <Foundation/Foundation.h>
#include <math.h>
#include <VHFShared/types.h>
#include <VHFShared/vhf2DFunctions.h>
#include <VHFShared/VHFDictionaryAdditions.h>
#include "../Cenon/functions.h"
#include "../Cenon/messages.h"
#include "../Cenon/propertyList.h"
#include "AstroPrincipal.h"	// shared ephemeris object
#include "astroCommon.h"
#include "Astro.bproj/AstroController.h"
#include "Map.h"

#define	EarthRadiusA	6378137.0
#define	EarthRadiusB	6356752.0
#define Eccentricity	(EarthRadiusA - EarthRadiusB)/EarthRadiusA
#define	OneDegInM	(2.0*Pi*EarthRadiusA / 360.0)	// meter -> degree (2*Pi*r/360 -> 1 degree in meter)

#define MeterToPoint(v)	((v)*72000.0/25.4)	// convert meter to point

static CIAMap	*ciaMap = nil;	// one shared cia map for all views

@interface Map(PrivateMethods)
- (NSPoint)mercatorLatLonForPointOnMap:(NSPoint) mapP;
- (NSPoint)mercatorPointOnMapForLat:(float)lat lon:(float)lon;
- (NSPoint)stereoLatLonForPointOnMap:(NSPoint)mapP;
- (NSPoint)stereoPointOnMapForLat:(float)lat lon:(float)lon;
- (float)freeLatitudeLimitForLongitude:(float)lon searchMinLatitude:(BOOL)min;
- (NSPoint)freeLatLonForPointOnMap:(NSPoint)mapP;
- (NSPoint)freePointOnMapForLat:(float)lat lon:(float)lon;
@end

@implementation Map

/* here we load the map dictionary from the Cenon document folder, if we find one we are a map
 */
+ (Map*)mapWithMapDictionary:(NSDictionary*)dict
{
    if (!dict)
        return nil;
    return [[[Map alloc] initWithContentsOfDictionary:dict] autorelease];
}

/* the dictionary has to contain the following:
 *
 * Mercator maps:
 *   "name"       = a short description of the map
 *   "projection" = 'M'
 *   "x", "y"     = the 0 degree lat/lon position [point]
 *   "scale"      = the scale of the map (map unit = earth unit / scale)
 * 
 * Stereographic maps:
 *   "name"       = a short description of the map
 *   "projection" = 'S'
 *   "x", "y"     = position of the pole (north or south) [point]
 *   "scale"      = the scale of the map (map unit = earth unit / scale)
 *   "deg0"       = the angle of the 0 meridian to the horicontal axis [degree]
 * 
 * Free style maps (any projection):
 * Linear Interpolation between given points of a grid.
 * The more points are provided, the more precise is the positions.
 *   "name"       = a short description of the map
 *   "projection" = 'F'
 * A list of points describing the latitude and longitude positions of the map.
 * x, y are the screen coordinates in point. the latitude/longitude entries are in degrees.
 *   "longitude"   = { latitude=(x,y); latitude=(x,y); ... };
 * ...
 */
- (id)initWithContentsOfDictionary:(NSDictionary*)dict
{
    [super init];

    mapDict = [dict mutableCopy];

    /* CIA map */
    if ( ![dict objectForKey:@"projection"] )
    {
        mapset = [[dict objectForKey:@"mapset"] retain];
        center = NSMakePoint([dict floatForKey:@"centerLon"], [dict floatForKey:@"centerLat"]);
        size   = NSMakeSize([dict floatForKey:@"widthLon"], [dict floatForKey:@"heightLat"]);
        /*dict = [NSMutableDictionary dictionary];
        [dict setObject: @"11.0" forKey:@"centerLon"];
        [dict setObject:  @"0.0" forKey:@"centerLat"];
        [dict setObject:@"360.0" forKey:@"widthLon"];
        [dict setObject:@"180.0" forKey:@"heightLat"];
        [NSDictionary dictionaryWithObject:dict forKey:@"World"]*/
    }
    /* map in document */
    else
    {
        name = [[dict objectForKey:@"name"] retain];
        if ([dict objectForKey:@"x"])
            origin = NSMakePoint([dict floatForKey:@"x"], [dict floatForKey:@"y"]);
        if ([dict objectForKey:@"scale"])
            scale = [dict doubleForKey:@"scale"];
        if ([dict objectForKey:@"deg0"])
            deg0 = [dict floatForKey:@"deg0"];
        if ([dict objectForKey:@"grid"])
            freeGrid = [[dict objectForKey:@"grid"] retain];
        if ([dict objectForKey:@"bounds"])
            bounds = rectFromPropertyList([dict objectForKey:@"bounds"]);
        projectionType = [[dict stringForKey:@"projection"] characterAtIndex:0];
    }

    return self;
}

/* set the map bounds */
- (void)setBounds:(NSRect)rect
{
    bounds = rect;
}

NSPoint pointOnLineDistanceFromEnd(float dist, NSPoint p0, NSPoint p1)
{   NSPoint	p, d;
    float	len;

    d.x = p1.x - p0.x;
    d.y = p1.y - p0.y;
    if (d.x == 0.0 && d.y == 0.0)
        return NSZeroPoint;
    len = sqrt(SqrDistPoints(p0, p1));
    p.x = p0.x + d.x * (len - dist) / len;
    p.y = p0.y + d.y * (len - dist) / len;
    return p;
}

/* text convenience method
 */
- (VText*)text:(NSString*)string
        center:(NSPoint)p
         color:(NSColor*)color
          size:(float)textSize
      fontName:(NSString*)fontName
{   id		font = [NSFont fontWithName:fontName size:textSize];
    VText	*text = [VText textGraphic];
    NSRect	r;

    [text setString:string font:font lineHeight:0.0 color:color];
    r = [text bounds];
    p.x -= r.size.width  / 2.0;
    p.y -= r.size.height / 2.0;
    [text moveTo:p];
    return text;
}

/* map AC, DC, MC, IC at given date
 * we draw lines through locations where the axes are on a planet
 */
- (VGroup*)mapChartWithDate:(NSCalendarDate*)utc title:(NSString*)title inBounds:(NSRect)viewBounds
{   VGroup		*group = [VGroup group];
    VGroup		*groupACText = [VGroup group], *groupMCText = [VGroup group];
    VLine		*line;
    VArc		*arc;
    int			i, j;
    Ephemeris		*ephemeris = [Astro_Principal ephemeris];
    NSDictionary	*degreeDict = [ephemeris objectDictAtUTC:utc];
    NSArray		*planets = Prefs_Objects;
    NSColor		*color;
    float		lineWidth = 1.0;
    BOOL		isACLine, wasACLine = NO;
    NSString		*fontName = Prefs_AstroFont;

    [date release];
    date = [utc retain];

    /* calculate all map positions with AC/IC/MC/DC on planets.
     * 1. get planet positions
     * 2. calculate lat/lon for axis at these positions
     */

    /* MC/IC - blue/cyan color
     */

    for (i=0; i<(int)[planets count]; i++)
    {   NSString	*planet = [planets objectAtIndex:i];
        float		deg = [degreeDict floatForKey:planet], lon;
        NSPoint		p, pts[2], pa0, pa1, pb0, pb1, p0, p1;

        lon = longitudeForMC(deg, utc);
        switch (projectionType)
        {
            case 'S':	// stereo:	straight lines from origin to bounds
                p = [self pointOnMapForLat:0.0 lon:lon];
                if (vhfIntersectVectorAndRect(p, NSMakePoint(origin.x-p.x, origin.y-p.y), viewBounds, pts) < 2)
                {   NSLog(@"mapChartWithDate: two intersections expected!");
                    return nil;
                }
                pa0 = pb0 = origin;
                pa1 = ((origin.x-p.x) * (origin.x-pts[0].x) > 0.0) ? pts[0] : pts[1];
                pb1 = ((origin.x-p.x) * (origin.x-pts[0].x) > 0.0) ? pts[1] : pts[0];
                break;
            case 'M':	// mercator:	straight lines from top to bottom
                pa0.x = pa1.x = [self pointOnMapForLat:0.0 lon:lon].x;
                pb0.x = pb1.x = [self pointOnMapForLat:0.0 lon:lon+180.0].x;
                pa0.y = pb0.y = viewBounds.origin.y;
                pa1.y = pb1.y = viewBounds.origin.y + viewBounds.size.height;
                break;
            case 'F':	// free style
            {   float	minLat = [self freeLatitudeLimitForLongitude:lon searchMinLatitude:YES];
                float	maxLat = [self freeLatitudeLimitForLongitude:lon searchMinLatitude:NO];

                /* we have to create the MC/IC lines at points within the grid */
                pa0 = [self pointOnMapForLat:minLat lon:lon];
                pa1 = [self pointOnMapForLat:maxLat lon:lon];
                minLat = [self freeLatitudeLimitForLongitude:lon+180.0 searchMinLatitude:YES];
                maxLat = [self freeLatitudeLimitForLongitude:lon+180.0 searchMinLatitude:NO];
                pb0 = [self pointOnMapForLat:minLat lon:lon+180.0];
                pb1 = [self pointOnMapForLat:maxLat lon:lon+180.0];
                break;
            }
            default:
                NSLog(@"mapChartWithDate: unknown projection '%c' !", projectionType);
                return nil;
        }
        for (j=0; j<2; j++)	// MC, IC
        {
            p0 = (j==0) ? pa0 : pb0;
            p1 = (j==0) ? pa1 : pb1;
            if ( p0.x == MINCOORD || p1.x == MINCOORD )
                continue;	// skip lines outside of map
            color = (j==0) ? [NSColor blueColor]
                           : [NSColor colorWithCalibratedRed:0.0 green:0.5 blue:1.0 alpha: 1.0];

            line = [VLine lineWithPoints:p0 :p1];
            [line setColor:color];
            [line setWidth:lineWidth];
            [group addObject:line];
            p = pointOnLineDistanceFromEnd(20.0, p0, p1); // line end (p*1) minus some space
            [groupMCText addObject:arc = [VArc arcWithCenter:p radius:8.0 filled:YES]];
            [arc setFillColor:[NSColor yellowColor]];
            [groupMCText addObject:[self text:planetSymbol(planet) center:p
                                        color:color size:14.0 fontName:fontName]];
        }
    }

    /* AC/DC - red/orange color
     */
    for (i=0; i<(int)[planets count]; i++)
    {   NSString	*planet = [planets objectAtIndex:i];
        float		deg = [degreeDict floatForKey:planet], lonStart = 0.0, lon, l0 = 0.0, lat;
        float		jump = (projectionType == 'F') ? 0.5 : 5.0;
        NSPoint		textLatLon[2] = {{180.0, 180.0}, {180.0, 180.0}};
        NSPoint		p0 = NSMakePoint(-LARGE_COORD, -LARGE_COORD), p1;
        VPolyLine	*polyLine = nil;
        int		groupCount = [[group list] count];	// we need to know which polyLines were added

//if (![planet isEqual:@"Saturn"])
//    continue;
        if (projectionType == 'M')
            lonStart = [self latLonForPointOnMap:viewBounds.origin].x;
        for ( lon = lonStart; lon <= lonStart+360.0; lon += jump)	// segment every 5 deg longitude
        {
            lat = latitudeForAC(deg, sideralTime(utc, lon));
            p1 = [self pointOnMapForLat:lat lon:lon];

//NSLog(@"1 lat=%f lon=%f -> p1={%.2f %.2f}", lat, lon, p1.x, p1.y);
            if ( p0.x > -LARGE_COORD && p1.x > MINCOORD &&			// p0 ok
                 (NSPointInRect(p0, viewBounds) || NSPointInRect(p1, viewBounds)) )	// one point in bounds
            {   float	l1, lStep, dist = sqrt(SqrDistPoints(p0, p1));

//NSLog(@"1 lat=%f lon=%f -> p1={%.2f %.2f} p0={%.2f %.2f} ", lat, lon, p1.x, p1.y, p0.x, p0.y);
                /* Problem: je nher AC an 0, desto enger der Bereich gltiger Lngengrade.
                 *          wenn AC = 0 -> nur noch ein lngengrad ergibt sinnvolle Werte.
                 * Solution: We split the length between two points on the map down to a maximum length.
                 */
                lStep = Max(Min((lon - l0) / (dist / 10.0), jump), 0.05);	// stepsize between min and max
                for (l1=l0+lStep; l1<=lon; l1 += lStep)
                {   NSCalendarDate	*st = sideralTime(utc, l1);

                    lat = latitudeForAC(deg, st);
                    isACLine = ( isAC(lat, deg, st) ) ? YES : NO;
                    /* find lat close to equator for text */
                    if (projectionType != 'S')
                    {
                        if (lon < lonStart+180.0 && Abs(lat) < Abs(textLatLon[0].y))
                            textLatLon[0] = NSMakePoint(l1, lat);
                        else if (lon >= lonStart+180.0 && Abs(lat) < Abs(textLatLon[1].y))
                            textLatLon[1] = NSMakePoint(l1, lat);
                    }

                    p1 = [self pointOnMapForLat:lat lon:l1];
//NSLog(@"2 lat=%f l1=%f -> p1={%.2f %.2f}", lat, l1, p1.x, p1.y);
                    if ( p0.x > MINCOORD && p1.x > MINCOORD &&
                         (NSPointInRect(p0, viewBounds) || NSPointInRect(p1, viewBounds)) )	// line inside bounds
                    {
//NSLog(@"2 inside");
                        if (!polyLine || isACLine != wasACLine
                            || DiffPoint([polyLine pointWithNum:[polyLine numPoints]-1], p0) > TOLERANCE)
                        {
                            if ([polyLine numPoints] >= 2)
                                [group addObject:polyLine];
                            polyLine = [VPolyLine polyLine];
                            [polyLine setColor:(isACLine) ? [NSColor redColor] : [NSColor orangeColor]];
                            [polyLine setWidth:lineWidth];
                            [polyLine addPoint:p0];
                            wasACLine = isACLine;
                        }
                        [polyLine addPoint:p1];
                    }

                    p0 = p1;
                    l0 = l1;
                }
            }
            p0 = p1;
            l0 = lon;
        }
        if ([polyLine numPoints] >= 2)
            [group addObject:polyLine];

        /* text */
        {   NSPoint	ps[2] = {{MINCOORD, MINCOORD}, {MINCOORD, MINCOORD}}, *pts;

            /* for stereo projection intersect PolyLine(s) and bounds and
             * place text a little away from the intersection points
             */
            if (projectionType == 'S' || projectionType == 'F')	// stereo or free
            {   int	n, cnt;
                NSRect	rect = (bounds.size.width == 0.0) ? NSInsetRect(viewBounds, 2*16, 2*16)
                                                          : bounds;

                for (n=groupCount; n<(int)[[group list] count]; n++)
                {   VPolyLine	*g = [[group list] objectAtIndex:n];

                    if ( (cnt = [g intersections:&pts withRect:rect]) )
                    {
                        if (cnt >= 1)
                            ps[0] = pts[0];
                        if (cnt >= 2)
                            ps[1] = pts[1];
                        free(pts);
                    }
                    /*else if (projectionType == 'F')	// polyline is too short -> place text at ends
                    {
                        ps[0] = [];
                    }*/
                }
            }
            else			// merkator, ...
            {
                if (textLatLon[0].x != 180.0)
                    ps[0] = [self pointOnMapForLat:0.0 lon:textLatLon[0].x];
                if (textLatLon[1].x != 180.0)
                    ps[1] = [self pointOnMapForLat:0.0 lon:textLatLon[1].x];
            }
            for (j=0; j<2; j++)
            {
                if (ps[j].x == MINCOORD)
                    continue;
                [groupACText addObject:arc = [VArc arcWithCenter:ps[j] radius:8.0 filled:YES]];
                [arc setFillColor:[NSColor yellowColor]];
                [groupACText addObject:[self text:planetSymbol(planet) center:ps[j]
                                            color:[NSColor redColor] size:14.0 fontName:fontName]];
            }
        }
    }

/*{   NSPoint	p;
    float	lat, lon;
    lat = 54.171108; lon = 315.500000;
    p = [self pointOnMapForLat:lat lon:lon];
NSLog(@"1: lat=%f l1=%f -> p={%.2f %.2f}", lat, lon, p.x, p.y);
//    lat = 71.745651; lon = 196.662415;
//    p = [self pointOnMapForLat:lat lon:lon];
//NSLog(@"2: lat=%f l1=%f -> p={%.2f %.2f}", lat, lon, p.x, p.y);
}*/

    [group addObject:groupMCText];
    [group addObject:groupACText];

    return group;
}
/* date of last mapping
 */
- (NSCalendarDate*)date
{
    return date;
}


/* return latitude and longitude for point on map
 */
- (NSPoint)latLonForPointOnMap:(NSPoint)mapP
{
    switch (projectionType)
    {
        case 'M': return [self mercatorLatLonForPointOnMap:mapP]; break;	// Mercator
        case 'S': return [self stereoLatLonForPointOnMap:mapP];   break;	// Stereographic
        case 'F': return [self freeLatLonForPointOnMap:mapP];     break;	// Free style
    }

    return NSZeroPoint;
}
/* return point on map for given latitude and longitude
 */
- (NSPoint)pointOnMapForLat:(float)lat lon:(float)lon
{
    switch (projectionType)
    {
        case 'M': return [self mercatorPointOnMapForLat:lat lon:lon]; break;	// Mercator
        case 'S': return [self stereoPointOnMapForLat:lat lon:lon];   break;	// Stereographic
        case 'F': return [self freePointOnMapForLat:lat lon:lon];     break;	// Free style
    }

    return NSZeroPoint;
}
/*
 * Mercator projection
 *
 * map parameter:
 * - origin of the map (0 lat/lon) [point]
 * - scale of the map
 */
/* Mercator projection
 * convert screen position to latitude
 * we need origin, scale
 *
 * formular: yfloat = r * ln ( tan (pi/4 + L/2) * ( (1 - e * sin (L)) / (1 + e * sin (L))) ** (e/2) )
 */
- (NSPoint)mercatorLatLonForPointOnMap:(NSPoint)mapP
{   double	yrad, y, e = 2.718281828;
    NSPoint	p;

    mapP.x -= origin.x;	// relative 0 meridian
    p.x = (mapP.x * scale * 360.0) / MeterToPoint(EarthRadiusA * 2.0 * Pi);
    if ( p.x >  180.0 ) p.x -= 360.0;
    if ( p.x < -180.0 ) p.x += 360.0;

    mapP.y -= origin.y;	// relative 0 meridian
    p.y = (mapP.y * scale * 180.0) / MeterToPoint(EarthRadiusA * Pi);

    y = p.y;
    y *= OneDegInM;
    y /= EarthRadiusA;

    y = pow(e, y); // log -> e pow yfloat = ex log

    yrad = (atan(y) - Pi/4.0) * 2.0;
    p.y = RadToDeg(yrad);
    if ( p.y >  90.0 ) p.y -= 180.0;
    if ( p.y < -90.0 ) p.y += 180.0;

    return p;
}
/* Mercator projection
 * convert latitude to y screen position
 * we need origin, scale
 *
 * formular: yfloat = r * ln ( tan (pi/4 + L/2) )
 */
- (NSPoint)mercatorPointOnMapForLat:(float)lat lon:(float)lon
{   double	yrad, tangens, y, degInPoint;
    NSPoint	p;

    if ( Abs(lat) >= 90.0 )	// limit latitude to range [-89.99, 89.99]
        lat = (lat > 0) ? (89.99) : (-89.99);
    if ( Abs(lon) >= 180.0 )	// limit longitude to range [-179.99, 179.99]
    {
        while (lon >= 180.0)  lon -= 360.0;
        while (lon <= -180.0) lon += 360.0;
        if (Abs(lon) >= 179.99)
            lon = (lon == 180.0) ? 179.99 : -179.99;
    }

    degInPoint = MeterToPoint(EarthRadiusA * Pi) / (scale * 180.0);
    p.x = lon * degInPoint;
    p.x += origin.x;
    /* wrap around, if point is on the wrong side of the map */
    if (p.x < 0.0)	//if (p.x < mapBounds.origin.x)
        p.x += 360.0 * degInPoint;

    yrad = DegToRad(lat);
    tangens = tan(Pi/4.0 + yrad/2.0);
    y = log(tangens);

    y *= EarthRadiusA;
    y /= OneDegInM;

    p.y = y * degInPoint /*MeterToPoint(EarthRadiusA * Pi) / (scale * 180.0)*/;
    p.y += origin.y;

    return p;
}

/*
 * Stereographic projection (Azimut projection)
 *
 * map parameter:
 * - origin of the map (pole) [point]
 * - angle of 0 meridian on the map to horicontal axis [deg]
 * - scale of the map on screen
 */
/* Stereographic projection (Azimut projection)
 * return latitude and longitude for a point on the map
 * we need origin, deg0, scale
 */
- (NSPoint)stereoLatLonForPointOnMap:(NSPoint)mapP
{   NSPoint	p;
    double	d, x, y, m, b, lat, lon;

    /* longitude = angle between line of 0 meriadian and point on map */
    p = vhfPointRotatedAroundCenter(NSMakePoint(origin.x+100.0, origin.y), deg0, origin);
    lon = vhfAngleBetweenPoints(p, origin, mapP);
    if (lon > 180.0)
        lon = lon-360.0;

    /* latitude = stereographic (azimut) projection */
    d = 6356752.0 * 2.0 * 72000.0 / 25.0;	// earth diameter in point
    x = mapP.x - origin.x;
    y = mapP.y - origin.y;

    m = sqrt(x*x + y*y) * scale;		// distance from center on map
    b = Atan(m/d);
    lat = 90.0 - 2.0*b;

    return NSMakePoint(lon, lat);
}
/* Stereographic projection (Azimut projection)
 * return point on map for given latitude and longitude
 * we need origin, deg0, scale
 */
- (NSPoint)stereoPointOnMapForLat:(float)lat lon:(float)lon
{   NSPoint	p;
    double	deg, d, b, rad;

    /* get angle to rotate 0 meridian */
    deg = lon + deg0;

    /* radius of latitude = stereographic (azimut) projection */
    d = 6356752.0 * 2.0 * 72000.0 / 25.0;	// earth diameter in point
    b = (90.0 - lat) / 2.0;
    rad = Tan(b) * d / scale;

    p = vhfPointRotatedAroundCenter(NSMakePoint(origin.x+rad, origin.y), deg, origin);
    return p;
}

/*
 * Free projection
 * convert screen position to latitude
 * we need grid positions (mapDict)
 */
NSPoint vhfProjectedPointInPolygon(NSPoint p, NSPoint *pts, NSPoint *ptsOut)
{   int		i, cnt = 4;
    float	f;
    NSPoint	p0, d0, p1, d1, pEnd, pEndOut, pOut;
    NSRect	lineRect;

    /* line from pts[0] through p with end at intersection with lines in pts */
    p0 = pts[0];
    d0 = NSMakePoint(p.x-p0.x, p.y-p0.y);
    for (i=1; i<cnt-1; i++)
    {
        p1 = pts[i];
        d1 = NSMakePoint(pts[i+1].x-p1.x, pts[i+1].y-p1.y);
        lineRect.origin.x = Min(p1.x, p1.x + d1.x) - TOLERANCE;
        lineRect.origin.y = Min(p1.y, p1.y + d1.y) - TOLERANCE;
        lineRect.size = NSMakeSize(Abs(d1.x)+2.0*TOLERANCE, Abs(d1.y)+2.0*TOLERANCE);
//iCnt = vhfIntersectVectors(p0, d0, p1, d1, &pEnd);
//NSLog(@"p0={%.2f %.2f} d0={%.2f %.2f} p1={%.2f %.2f} d1={%.2f %.2f} iCnt=%d pEnd={%.2f %.2f}", p0.x, p0.y, d0.x, d0.y, p1.x, p1.y, d1.x, d1.y, iCnt, pEnd.x, pEnd.y);
        if ( vhfIntersectVectors(p0, d0, p1, d1, &pEnd) && NSPointInRect(pEnd, lineRect) )
            break;
    }
    if (i >= cnt-1)
    {   NSLog(@"vhfProjectedPointInPolygon(): intersection expected, ignoring!");
        return NSZeroPoint;
    }

    /* get end point in outer coordinates on line LR/UR
     * we have: pEnd, pts[i], pts[i+1], ptsOut[i], ptsOut[i+1]
     */
    f = (Diff(d1.x, 0.0) > TOLERANCE) ? (pEnd.x - pts[i].x) / d1.x : (pEnd.y - pts[i].y) / d1.y;
    pEndOut.x = ptsOut[i].x + (ptsOut[i+1].x-ptsOut[i].x) * f;
    pEndOut.y = ptsOut[i].y + (ptsOut[i+1].y-ptsOut[i].y) * f;
//NSLog(@"pts[i]={%.2f %.2f} ptsOut[i]={%.2f %.2f} pEnd={%.2f %.2f} pEndOut={%.2f %.2f} pts[i+1]={%.2f %.2f} ptsOut[i+1]={%.2f %.2f}", pts[i].x, pts[i].y, ptsOut[i].x, ptsOut[i].y, pEnd.x, pEnd.y, pEndOut.x, pEndOut.y, pts[i+1].x, pts[i+1].y, ptsOut[i+1].x, ptsOut[i+1].y);

    /* convert p to outer coordinates. we have: p, p0, pEnd, ptsOut[0], pEndOut */
    d0 = NSMakePoint(pEnd.x-p0.x, pEnd.y-p0.y);
    f = (Diff(d0.x, 0.0) > TOLERANCE) ? (p.x - p0.x) / d0.x : (p.y - p0.y) / d0.y;
    pOut.x = ptsOut[0].x + (pEndOut.x-ptsOut[0].x) * f;
    pOut.y = ptsOut[0].y + (pEndOut.y-ptsOut[0].y) * f;
//NSLog(@"pts[0]={%.2f %.2f} ptsOut[0]={%.2f %.2f} p={%.2f %.2f} pOut={%.2f %.2f} pEnd={%.2f %.2f} pEndOut={%.2f %.2f}", pts[0].x, pts[0].y, ptsOut[0].x, ptsOut[0].y, p.x, p.y, pOut.x, pOut.y, pEnd.x, pEnd.y, pEndOut.x, pEndOut.y);

    return pOut;
}
- (NSPoint)pointForLongKey:(NSString*)longitude latKey:(NSString*)latitude
{   NSArray	*coords = [[freeGrid objectForKey:longitude] objectForKey:latitude];

    if (!coords)
        return NSZeroPoint;
    return NSMakePoint( [[coords objectAtIndex:0] floatValue], [[coords objectAtIndex:1] floatValue] );
}
- (float)freeLatitudeLimitForLongitude:(float)lon searchMinLatitude:(BOOL)min
{   NSArray	*lons = [freeGrid allKeys], *lats;
    int		i;
    float 	gridLon = 360.0, lat;

    if (lon >  180) lon -= 360.0;
    if (lon < -180) lon += 360.0;

    for (i=0; i<(int)[lons count]; i++)
    {   float	v = [[lons objectAtIndex:i] floatValue];
        if ( (gridLon > lon) || (v > gridLon && v < lon) )
            gridLon = v;
    }
    if (gridLon == 360.0)
    {   NSLog(@"freeLatitudeLimitForLongitude: no longitude found in range of lon = %f", lon);
        return 0.0;
    }
    lats = [[freeGrid objectForKey:vhfStringWithFloat(gridLon)] allKeys];
    lat = [[lats objectAtIndex:0] floatValue];
    for (i=1; i<(int)[lats count]; i++)
    {   float	v = [[lats objectAtIndex:i] floatValue];
        if ( (min && v < lat) || (!min && v > lat) )
            lat = v;
    }
//NSLog(@"gridlon=%f lon=%f lat=%f", gridLon, lon, lat);
    return lat;
}
- (NSPoint)freeLatLonForPointOnMap:(NSPoint)p
{   NSMutableArray	*lons, *lats;
    float		latMin, latMax, lngMin, lngMax;
    NSPoint		pts[4];
    int			i, j, k;

    /* find edge points of rectangle */
    lons = [[[freeGrid allKeys] mutableCopy] autorelease];
    [lons sortUsingFunction:sortAsNumbers context:nil];
    for ( i=0; i<(int)[lons count]-1; i++ )
    {
        lats = [[[[freeGrid objectForKey:[lons objectAtIndex:i]] allKeys] mutableCopy] autorelease];
        [lats sortUsingFunction:sortAsNumbers context:nil];
        for ( j=0; j<(int)[lats count]-1; j++ )
        {
            pts[0] = [self pointForLongKey:[lons objectAtIndex:i]   latKey:[lats objectAtIndex:j]];
            pts[1] = [self pointForLongKey:[lons objectAtIndex:i+1] latKey:[lats objectAtIndex:j]];
            pts[2] = [self pointForLongKey:[lons objectAtIndex:i+1] latKey:[lats objectAtIndex:j+1]];
            pts[3] = [self pointForLongKey:[lons objectAtIndex:i]   latKey:[lats objectAtIndex:j+1]];
            for (k=0; k<4; k++)
                if (pts[k].x == 0.0 && pts[k].y == 0.0)
                    break;
            if (k<4)
                continue;

            /* check, if point (p) is inside the polygon defined by our 4 points */
            if ( vhfIsPointInsidePolygon(p, pts, 4) )
            {   NSPoint	pt, ptsOut[4];

                /* calculate latitude, longitude */
                latMin = [[lats objectAtIndex:j]   floatValue];
                latMax = [[lats objectAtIndex:j+1] floatValue];
                lngMin = [[lons objectAtIndex:i]   floatValue];
                lngMax = [[lons objectAtIndex:i+1] floatValue];
                ptsOut[0] = NSMakePoint(lngMin, latMin);
                ptsOut[1] = NSMakePoint(lngMax, latMin);
                ptsOut[2] = NSMakePoint(lngMax, latMax);
                ptsOut[3] = NSMakePoint(lngMin, latMax);

                pt = vhfProjectedPointInPolygon(p, pts, ptsOut);
                return pt;
            }
        }
    }
    return NSZeroPoint;
}
/* Free style projection (interpolation from grid points)
 * return point on map for given latitude and longitude
 * we require grid
 */
- (NSPoint)freePointOnMapForLat:(float)lat lon:(float)lon
{   NSMutableArray	*lons, *lats;
    float		latMin = 0.0, latMax = 0.0, lngMin = 0.0, lngMax = 0.0;
    NSPoint		p, pt, pts[4], ptsOut[4];
    int			i, j, k;

    if (lon >  180.0) lon -= 360.0;
    if (lon < -180.0) lon += 360.0;

    /* find edge points of area */
    lons = [[[freeGrid allKeys] mutableCopy] autorelease];
    [lons sortUsingFunction:sortAsNumbers context:nil];
    for ( i=0; i<(int)[lons count]-1; i++ )
    {
        lngMin = [[lons objectAtIndex:i]   floatValue];
        lngMax = [[lons objectAtIndex:i+1] floatValue];
        if ( lngMin <= lon && lngMax >= lon )
            break;
    }
    if ( i>=(int)[lons count]-1 )
    {   //NSLog(@"longitude missing in map.dict!");
        return NSMakePoint(MINCOORD, MINCOORD);
    }
    lats = [[[[freeGrid objectForKey:[lons objectAtIndex:i]] allKeys] mutableCopy] autorelease];
    [lats sortUsingFunction:sortAsNumbers context:nil];
    for ( j=0; j<(int)[lats count]-1; j++ )
    {
        latMin = [[lats objectAtIndex:j]   floatValue];
        latMax = [[lats objectAtIndex:j+1] floatValue];
        if ( latMin<=lat && latMax>=lat )
            break;
    }
    if ( j>=(int)[lats count]-1 )
    {   //NSLog(@"latitude missing in map.dict!");
        return NSMakePoint(MINCOORD, MINCOORD);
    }
    ptsOut[0] = [self pointForLongKey:[lons objectAtIndex:i]   latKey:[lats objectAtIndex:j]];
    ptsOut[1] = [self pointForLongKey:[lons objectAtIndex:i+1] latKey:[lats objectAtIndex:j]];
    ptsOut[2] = [self pointForLongKey:[lons objectAtIndex:i+1] latKey:[lats objectAtIndex:j+1]];
    ptsOut[3] = [self pointForLongKey:[lons objectAtIndex:i]   latKey:[lats objectAtIndex:j+1]];
    for (k=0; k<4; k++)
        if (ptsOut[k].x == 0.0 && ptsOut[k].y == 0.0)
            break;
    if (k<4)	// area is not completely defined!
        return NSMakePoint(MINCOORD, MINCOORD);

    /* calculate coordinates for lat, lng */
    pt.x = lon; pt.y = lat;
    pts[0] = NSMakePoint(lngMin, latMin);
    pts[1] = NSMakePoint(lngMax, latMin);
    pts[2] = NSMakePoint(lngMax, latMax);
    pts[3] = NSMakePoint(lngMin, latMax);

//NSLog(@"in:  p0={%.2f %.2f} p1={%.2f %.2f} p2={%.2f %.2f} p3={%.2f %.2f}", pts[0].x, pts[0].y, pts[1].x, pts[1].y, pts[2].x, pts[2].y, pts[3].x, pts[3].y);
//NSLog(@"out: p0={%.2f %.2f} p1={%.2f %.2f} p2={%.2f %.2f} p3={%.2f %.2f}", ptsOut[0].x, ptsOut[0].y, ptsOut[1].x, ptsOut[1].y, ptsOut[2].x, ptsOut[2].y, ptsOut[3].x, ptsOut[3].y);
    p = vhfProjectedPointInPolygon(pt, pts, ptsOut);

//NSLog(@"freePointOnMapForLatLon: lat=%f lon=%f -> p={%.2f %.2f}", lat, lon, p.x, p.y);
    return p;
}



/* calculate CIA map, if necessary
 */
- (BOOL)needsCalculation
{
    return (!projectionType) ? YES : NO;
}
- (NSMutableDictionary*)createMap
{   NSMutableDictionary	*dict;

    if ( projectionType )	// allready calculated
        return nil;
    if (!ciaMap)
        ciaMap = [[CIAMap map] retain];
NSLog(@"%f %f %f %f", center.x, center.y, size.width, size.height);
    dict = [ciaMap generateMapListWithCenter:center size:size grid:10.0 :NO :NO mapset:mapset];

    /* set parameter */
    projectionType = 'M';
    origin = [ciaMap origin];
    scale = [ciaMap scale];
NSLog(@"createMap: scale = %f", scale);

    return dict;
}
- (CIAMap*)ciaMap
{
    return ciaMap;
}


/* load map image and add background to map
 */
- (void)displayMapAtDate:(NSCalendarDate*)newMapDate inView:(DocView*)view
{   NSString	*fileName;
    NSString	*path = [mapDict objectForKey:@"path"];
    int		layerBG  = ([mapDict objectForKey:@"bgImageLayer"]) ? [mapDict intForKey:@"bgImageLayer"] : 0;
    int		layerMap = ([mapDict objectForKey:@"mapLayer"])     ? [mapDict intForKey:@"mapLayer"]     : 1;
    NSArray	*reps;
    VImage	*image;
    LayerObject	*layer = [[view layerList] objectAtIndex:layerMap];
    NSArray	*mapTimes = [mapDict objectForKey:@"mapTimes"];

    if (![mapDict objectForKey:@"fileFormat"])
        return;

    /* find map closest to available maps
     */
    if ([mapTimes count])
    {   int	hourTarget = [newMapDate hourOfDay]*100 + [newMapDate minuteOfHour];	// HHMM
        int	minDiff = 9999, bestI = 0, i, hour;

        for (i=0; i<[mapTimes count]; i++)
        {   hour = [[mapTimes objectAtIndex:i] intValue];
            if (Diff(hour, hourTarget) < minDiff)
            {   minDiff = Diff(hour, hourTarget);
                bestI = i;
            }
        }

        hour = [[mapTimes objectAtIndex:bestI] intValue];
        newMapDate = [NSCalendarDate dateWithYear:[newMapDate yearOfCommonEra]
                                            month:[newMapDate monthOfYear]
                                              day:[newMapDate dayOfMonth]
                                             hour:hour/100
                                           minute:hour-(hour/100*100)
                                           second:0 timeZone:[NSTimeZone timeZoneWithName:@"UTC"]];
    }

    fileName = [newMapDate descriptionWithCalendarFormat:[mapDict objectForKey:@"fileFormat"]];
    if ( [path rangeOfString:@"%@"].length > 0 )	// substitute year
        path = [NSString stringWithFormat:path, [newMapDate descriptionWithCalendarFormat:@"%Y"]];
    if (![path hasPrefix:@"/"])	// relative path -> add user library
        path = [userLibrary() stringByAppendingPathComponent:path];
    path = [path stringByAppendingPathComponent:fileName];

    if ( ![[NSFileManager defaultManager] fileExistsAtPath:path] )
    {   NSLog(@"Map -displayMapAtDate: Can't load image from file %@", path);
        return;
    }

    if (mapDate && [mapDate isEqual: newMapDate])
        return;
    [mapDate release];
    mapDate = [newMapDate retain];

    /* load image */
    reps = [NSImageRep imageRepsWithContentsOfFile:path];
    image = [[[VImage alloc] initWithRepresentations:reps] autorelease];

    /* if background layer is off
     * we copy gray pixels from background over white pixels of map
     */
    if (![[[view layerList] objectAtIndex:layerBG] state])
    {   NSBitmapImageRep	*repMap = [reps objectAtIndex:0], *repBG;
        unsigned char		*planesMap[5], *planesBG[5];
        NSArray			*list = [[[view layerList] objectAtIndex:layerBG] list];

        if ( (repBG = [[[[list objectAtIndex:0] image] representations] objectAtIndex:0]) )
        {
            /* we draw the background only if planes are 8bit or 24bit (RGB)
             */
            if ([repBG numberOfPlanes] <= 3 || [repMap numberOfPlanes] <= 3)
            {   int	numPlanes = [repMap numberOfPlanes];
                int	wMap = [repMap pixelsWide], wBG = [repBG pixelsWide];
                int	h = Min([repMap pixelsHigh], [repBG pixelsHigh]), x, y;

                [repBG  getBitmapDataPlanes:planesBG];
                [repMap getBitmapDataPlanes:planesMap];

                /* background != white and map == white -> map gray */
                for (x=0; x<wMap; x++)
                {
                    for (y=0; y<h; y++)
                    {   int	bMap, bBG;	// byte index in plane

                        if (x >= wBG)		// width of map != width of background
                            continue;
                        bMap = y*wMap + x;
                        bBG  = y*wBG  + x;	// adopt width of background to map
                        if (planesBG[0][bBG] != 255 && planesMap[0][bMap] == 255)
                        {
                            if (numPlanes == 1)
                                planesMap[0][bMap] = planesBG[0][bBG];
                            else if (numPlanes == 3)
                                planesMap[0][bMap] = planesMap[1][bMap] = planesMap[2][bMap] = planesBG[0][bBG];
                        }
                    }
                }
            }
            else
            {
                if ([repBG numberOfPlanes] > 3)
                    NSLog(@"Map -displayMapAtDate: Background must be 8 bit or RGB !");
                if ([repMap numberOfPlanes] > 3)
                    NSLog(@"Map -displayMapAtDate: Map Image must be 8 bit or RGB !");
            }
        }
    }

    [image setOrigin:NSZeroPoint];
    [image scale:0.5 :0.5 withCenter:[image origin]];
    if ([view scaleFactor] == 2.0)
        [[image image] recache];	// needed for 1:1 scale to make background visible
    [layer removeAllObjects];
    [layer addObject:image];

    [view drawAndDisplay];
}


- (char)projectionType
{
    return projectionType;
}
- (void)setFreeGrid:(NSDictionary*)grid
{
    [freeGrid release];
    freeGrid = [grid retain];
}
/* here we write the Dictionary for the loaded or calculated map
 * we write the dictionary into the Cenon document
 */
- (void)writeInfoToFile:(NSString*)file
{
    if (name)
        [mapDict setObject:name forKey:@"name"];
    if (bounds.size.width > 0.0 && bounds.size.height > 0.0)
        [mapDict setObject:propertyListFromNSRect(bounds) forKey:@"bounds"];
    if (projectionType > 0)
        [mapDict setObject:[NSString stringWithFormat:@"%c", projectionType] forKey:@"projection"];
    switch (projectionType)
    {
        case 'M':	// Mercator
            [mapDict setObject:[NSString stringWithFormat:@"%f", origin.x] forKey:@"x"];
            [mapDict setObject:[NSString stringWithFormat:@"%f", origin.y] forKey:@"y"];
            [mapDict setObject:[NSString stringWithFormat:@"%f", scale] forKey:@"scale"];
            if (mapset)
            {   [mapDict setObject:mapset forKey:@"mapset"];
                //[mapDict setObject:centerLon forKey:@"centerLon"];
                //[mapDict setObject:centerLat forKey:@"centerLat"];
                //[mapDict setObject:widthLon forKey:@"widthLon"];
                //[mapDict setObject:widthLat forKey:@"widthLat"];
            }
            break;
        case 'S':	// Stereographic
            [mapDict setObject:[NSString stringWithFormat:@"%f", origin.x] forKey:@"x"];
            [mapDict setObject:[NSString stringWithFormat:@"%f", origin.y] forKey:@"y"];
            [mapDict setObject:[NSString stringWithFormat:@"%f", scale] forKey:@"scale"];
            [mapDict setObject:[NSString stringWithFormat:@"%f", deg0] forKey:@"deg0"];
            break;
        case 'F':	// Free style
            if (freeGrid)
                [mapDict setObject:freeGrid forKey:@"grid"];
            break;
        default:	// CIA Map
            if (mapset)
            {   [mapDict setObject:mapset forKey:@"mapset"];
                [mapDict setObject:vhfStringWithFloat(center.x) forKey:@"centerLon"];
                [mapDict setObject:vhfStringWithFloat(center.y) forKey:@"centerLat"];
                [mapDict setObject:vhfStringWithFloat(size.width)  forKey:@"widthLon"];
                [mapDict setObject:vhfStringWithFloat(size.height) forKey:@"widthLat"];
            }
    }

    [mapDict writeToFile:file atomically:YES];
}

- (void)dealloc
{
    [mapDict release];
    [name release];
    [mapset release];
    [freeGrid release];

    [date release];
    [mapDate release];

    [super dealloc];
}

@end
