/* DocViewMap.m
 * Map additions for DocView.m
 *
 * Copyright (C) 2001-2006 by vhf interservice GmbH
 * Author:   Ilonka Fleischmann
 *
 * created:  2001-08-??
 * modified: 2003-07-19
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the vhf Public License as
 * published by vhf interservice GmbH. Among other things, the
 * License requires that the copyright notices and this notice
 * be preserved on all copies.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the vhf Public License for more details.
 *
 * You should have received a copy of the vhf Public License along
 * with this program; see the file LICENSE. If not, write to vhf.
 *
 * vhf interservice GmbH, Im Marxle 3, 72119 Altingen, Germany
 * eMail: info@vhf.de
 * http://www.vhf.de
 *
 *
 * MapDocView -> Map
 */

#include <AppKit/AppKit.h>
#include <VHFShared/VHFDictionaryAdditions.h>
#include <VHFShared/vhfCommonFunctions.h>
#include "../Cenon/App.h"
#include "../Cenon/DocWindow.h"
#include "AstroPrincipal.h"
#include "DocViewMap.h"
#include "AstroPanels.h"
#include "locations.h"
#include "astroLocations.h"
#include "astroCommon.h"
#include "AstroChart.h"
#include "Map.h"

static NSArray	*colors = nil;

@implementation DocView(Map)

/* a map document contains a map info file
 * here we load this file, create a map object and store it
 */
- (void)initMap
{   NSString			*file = [[document filename] stringByAppendingPathComponent:MAP_INFOFILE];
    NSDictionary		*dict;
    Map				*map;
    NSNotificationCenter	*notificationCenter = [NSNotificationCenter defaultCenter];

    if ( !(dict = [NSDictionary dictionaryWithContentsOfFile:file]) )
	return;	// we are not a map
    map = [Map mapWithMapDictionary:dict];
    [statusDict setObject:map forKey:@"map"];

    /* add observer for mouse down */
    [notificationCenter removeObserver:self name:DocViewMouseDown object:self];
    [notificationCenter addObserver:self
                           selector:@selector(mouseWentDown:)
                               name:DocViewMouseDown
                             object:nil];	// 'self' doesn't work here!

    if ( !colors )
        colors = [[NSArray arrayWithObjects:[NSColor blackColor],   [NSColor redColor],
                                            [NSColor blueColor],    [NSColor greenColor],
                                            [NSColor brownColor],   [NSColor cyanColor],
                                            [NSColor magentaColor], [NSColor orangeColor],
                                            [NSColor grayColor],    [NSColor yellowColor],
                                            [NSColor purpleColor],  [NSColor lightGrayColor],
                                            [NSColor darkGrayColor], nil] retain];
}

- (BOOL)isMap
{
    return ([statusDict objectForKey:@"map"]) ? YES : NO;
}

- (CIAMap*)ciaMap
{
    return [[statusDict objectForKey:@"map"] ciaMap];
}

- (void)mouseWentDown:(NSNotification*)notification
{   DocView	*view = [notification object];
    NSEvent	*event = [[notification userInfo] objectForKey:@"event"];

    if (view != self || ![self isMap])
        return;
    if ([event isMemberOfClass:[NSEvent class]])
        [self setMapRadix:event];
    else
        NSLog(@"DocView(Map), notification send with userInfo = '%@', should be NSEvent!", event);
}
- (void)setMapRadix:(NSEvent*)event
{   NSPoint		p, pt;
    //int		i;
    //Document		*doc, *docCircle = nil;
    AstroPanel		*astroPanel = [Astro_Principal astroPanel];
    NSCalendarDate	*date = [(APChart*)[astroPanel windowAt:AP_CHART] date];
    Map			*map;
    NSMutableDictionary	*dict;
    NSDictionary	*composite;

    if (![self isMap])
    {   NSLog(@"No map loaded!");
        return;
    }

/*
    for ( i=[[NSApp windows] count]-1; i>=0; i-- )
    {   doc = [NSApp documentInWindow:[[(App*)NSApp windows] objectAtIndex:i]];
        if ( [[doc name] hasPrefix:CHART_PREFIX] )
            docCircle = doc;
    }
    if ( !docCircle )
        return;
*/

    map = [statusDict objectForKey:@"map"];
    if ([map date])	// use date of last mapping, instead of chart date
        date = [map date];
    if (!date)
        return;

    /* get p */
    p = [self convertPoint:[event locationInWindow] fromView:nil]; // p in points

    /* calc lat/lon from point */
    pt = [map latLonForPointOnMap:p];
printf("%s  p = {%f %f} -> lon = %f, lat = %f\n", [[date description] cString], p.x, p.y, pt.x, pt.y);

    dict = [NSMutableDictionary dictionary];
    [dict setObject:@"Map Chart" forKey:@"name"];
    [dict setObject:date forKey:@"date"];
    [dict setObject:[NSNumber numberWithFloat:pt.y] forKey:@"lat"];
    [dict setObject:[NSNumber numberWithFloat:pt.x] forKey:@"lon"];
    [dict setObject:@" " forKey:@"city"];
    if ( (composite = [[astroPanel windowAt:AP_CHART] compositeData]) )
        [dict setObject:composite forKey:@"composite"];
    [[NSNotificationCenter defaultCenter] postNotificationName:CalculateChartNotification object:dict];

#if 0
    //if ( [[self ciaMap] convertScreenPointToLng:&lon :&lat :p] )
    {   //NSCalendarDate	*sideralTime;
        //float		ac;
        AstroChart	*astroChart = [[astroPanel windowAt:AP_CHART] astroChart];

        /* calculate astro chart at click location */
        if (!astroChart)
            astroChart = [AstroChart astroChartWithView:[docCircle documentView]];
        [astroChart radix:date longitude:pt.x latitude:pt.y city:@" " title:@"Map Chart"
                composite:[[astroPanel windowAt:AP_CHART] compositeData]];

        /* calculate values to radix date at this point */
        //sideralTime = [self sideralTime:date longitude:rLon];
        //ac = [self ac:sideralTime latitude:rLat];
        //[[astroPanel windowAt:AP_MAP] valuesToRadixAC:ac atLat:pt.y lon:pt.x];
    }
#endif
}

/* create tables of planets relating chart
 * positions, angles, mirrors, harmonies
 * modified: 2006-02-19
 */
- (void)createMapWithCenter:(NSPoint)center size:(NSSize)size grid:(float)gridSize
                   features:(NSDictionary*)features
{   int			i, l;
    LayerObject		*layer = nil;
    NSMutableDictionary	*dict;
    NSAutoreleasePool	*pool = [NSAutoreleasePool new];
    NSArray		*allKeys;
    Map			*map = [statusDict objectForKey:@"map"];

    /* if we are not a map view, we find one and pass the job over or return */
    if ( ![self isMap] )
    {
        for ( i=[[(App*)NSApp windows] count]-1; i>=0; i-- )
        {   Document *doc = [(App*)NSApp documentInWindow:[[NSApp windows] objectAtIndex:i]];
            if ( [self isMap] )
                [[doc documentView] createMapWithCenter:center size:size grid:gridSize features:features];
        }
        return;
    }

    /* empty layers */
    if ([map needsCalculation])	// cia maps, only
        for ( l=0; l<(int)[layerList count]; l++ )
        {
            layer = [layerList objectAtIndex:l];
            [layer setList:[NSMutableArray array]];	// free memory as soon as possible
        }

    /* get map */
    if ( (dict = [map createMap]) )
    {
        allKeys = [dict allKeys];

        /* create layers */
        for (i=0; i<(int)[allKeys count]; i++)
        {
            for ( l=0; l<(int)[layerList count]; l++ )
            {
                if ( [[[layerList objectAtIndex:l] string] isEqual:[allKeys objectAtIndex:i]] )
                {   layer = [layerList objectAtIndex:l];
                    break;
                }
            }
            if ( !layer )
            {   layer = [LayerObject layerObjectWithFrame:[self bounds]];
                [layerList addObject:layer];
            }

            [layer setState:1];
            [layer setString:[NSString stringWithFormat:[allKeys objectAtIndex:i]]];
            [layer setList:[dict objectForKey:[allKeys objectAtIndex:i]]];
            [layer setEditable:NO];
        }
    }

    layer = [layerList objectAtIndex:[layerList count]-1];
    [layer removeAllObjects];

    /* map AC/MC (Axes) lines on planets, FIXME: map aspects to AC/MC */
    if ([features objectForKey:@"featureACMC"])
    {   NSCalendarDate	*utc = [(APChart*)[[Astro_Principal astroPanel] windowAt:AP_CHART] date];
        VGroup		*group;

        if ( (group = [map mapChartWithDate:utc title:nil inBounds:[self bounds]]) )
        {
            //layer = [layerList objectAtIndex:[layerList count]-1];
            //[layer removeAllObjects];
            [layer addObject:group];
        }
    }

    /* map house tips (keep house tip on Zodiac degree for every location) */
    if ([features objectForKey:@"featureHouses"])
    {   APChart		*chartPanel = [[Astro_Principal astroPanel] windowAt:AP_CHART];
        NSCalendarDate	*utc = [chartPanel date];
        float		lat = [chartPanel latitude], lon = [chartPanel longitude];
        VGroup		*group;

        if ( (group = [map mapHousesAtDate:utc lat:lat lon:lon title:nil inBounds:[self bounds]]) )
        {
            //layer = [layerList objectAtIndex:[layerList count]-1];
            //[layer removeAllObjects];
            [layer addObject:group];
        }
    }

    /* FIXME: map planet degrees */

    /* FIXME: map free Zodiac degree(s) */

    /* map Time Tracks */
    if ([features objectForKey:@"featureTimeTracks"])
    {   NSCalendarDate	*dateFr = [features objectForKey:@"dateFr"];
        NSCalendarDate	*dateTo = [features objectForKey:@"dateTo"];
        VGroup		*group;

        if ( (group = [map mapTimeTrackFromDate:dateFr toDate:dateTo
                                     forObjects:[features objectForKey:@"objects"]
                                earthRevolution:([features intForKey:@"earthRev"]) ? YES : NO
                                          title:nil inBounds:[self bounds]]) )
        {
            //layer = [layerList objectAtIndex:[layerList count]-1];
            //[layer removeAllObjects];
            [layer addObject:group];
        }
    }

    [[Astro_Principal astroPanel] update:document];

    [pool release];
    [self drawAndDisplay];
}

/* create dictionary from Markers in the format:
 * longitude = { latitude=(x,y); latitude=(x,y); ... };
 */
- (NSDictionary*)freeGridFromList:(NSArray*)list
{   int			i, cnt;
    NSMutableDictionary	*freeGrid = [NSMutableDictionary dictionary];

    for (i=0, cnt = [list count]; i<cnt; i++)
    {   VMark	*g = [list objectAtIndex:i];

        if ([g isKindOfClass:[VMark class]])
        {   NSArray		*components = [[g name] componentsSeparatedByString:@" "];
            NSPoint		p;
            float		lat, lon;
            NSString		*lonKey, *latKey;
            NSMutableDictionary	*lons;

            if ([components count] != 2)
            {   NSLog(@"Map, Free Grid: Marker expected with contents 'lat lon', got %@", [g name]);
                break;
            }
            lat = [[components objectAtIndex:0] floatValue];
            lon = [[components objectAtIndex:1] floatValue];
            p = [g pointWithNum:0];

            lonKey = vhfStringWithFloat(lon);
            if ( !(lons = [freeGrid objectForKey:lonKey]) )
            {   lons = [NSMutableDictionary dictionary];
                [freeGrid setObject:lons forKey:lonKey];
            }
            latKey = vhfStringWithFloat(lat);
            if ( [lons objectForKey:latKey] )
            {   NSLog(@"Map, Free Grid: coordinates '%@' p={%.4f %.4f} already in grid list, ignoring.",
                      [g name], p.x , p.y);
                continue;
            }
            [lons setObject:[NSArray arrayWithObjects:vhfStringWithFloat(p.x), vhfStringWithFloat(p.y), nil]
                     forKey:latKey];
        }
    }
    return freeGrid;
}

/* notification: write map.dict into document file
 */
- (void)mapDocumentHasBeenSaved:(NSNotification*)notification
{   DocView	*view = [(Document*)[notification object] documentView];
    NSString	*cenonFile = [[notification userInfo] objectForKey:@"file"];
    NSString	*file = [cenonFile stringByAppendingPathComponent:MAP_INFOFILE];

    if (view == self && [self isMap])
    {   Map	*map = [statusDict objectForKey:@"map"];

        if ([map projectionType] == 'F')
        {   NSArray	*list = nil;
            int		l, i;

            for (l=0; l<(int)[layerList count]; l++)
                if ( [[[layerList objectAtIndex:l] string] isEqual:@"Grid"] )
                    list = [[layerList objectAtIndex:l] list];
            if (list)
            {
                [map setFreeGrid: [self freeGridFromList:list]];
                for (i=0; i<(int)[list count]; i++)
                {   VRectangle	*g = [list objectAtIndex:i];

                    if ([g isKindOfClass:[VRectangle class]])
                        [map setBounds:[g coordBounds]];
                }
            }
        }
        [map writeInfoToFile:file];
    }
}

@end
