/* AstroController.m
 * Astrology Preferences controller
 *
 * Copyright (C) 1999-2006 by vhf interservice GmbH
 * Author:   Georg Fleischmann
 *
 * Created:  2003-03-23
 * Modified: 2006-06-04
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the vhf Public License as
 * published by vhf interservice GmbH. Among other things, the
 * License requires that the copyright notices and this notice
 * be preserved on all copies.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the vhf Public License for more details.
 *
 * You should have received a copy of the vhf Public License along
 * with this program; see the file LICENSE. If not, write to vhf.
 *
 * vhf interservice GmbH, Im Marxle 3, 72119 Ammerbuch, Germany
 * eMail: info@vhf.de
 * http://www.vhf.de
 */

#include <AppKit/AppKit.h>
#include "AstroController.h"
#include "../../Cenon/functions.h"
#include "../../Cenon/PreferencesPanel.subproj/NotificationNames.h"
#include "../AstroPrincipal.h"
#include "../AstroDate.h"	// date formats
#include "../astroMessages.h"	// localized strings

@interface AstroController(PrivateMethods)
@end

@implementation AstroController

/*
 * Defaults are registered in [App +initialize]
 * we are loaded too late here
 */


// protocol methods

/* create new instance of GeneralController
 */
+ (id)controller
{   static AstroController *controller = nil;

    if (!controller)
        controller = [[AstroController alloc] init];
    return controller;
}

- (id)init
{   NSNotificationCenter	*notificationCenter = [NSNotificationCenter defaultCenter];

    [super init];

    /* load main interface file */
    if ( ![NSBundle loadNibNamed:@"Astro" owner:self] )
    {   NSLog(@"Cannot load 'Astro' interface file");
        return nil;
    }

    /* the preferences panes */
    prefViews = [[NSMutableArray alloc] init];

    /* add observer to add preferences pane */
    [notificationCenter addObserver:self
                           selector:@selector(addItemNotification:)
                               name:AstroPrefsAddItemNotification
                             object:nil];

    //[self update:self];	// loads sub-panes

    return self;
}

/* add the standard preferences panes
 * created: 2006-04-27
 */
- (void)initSubPanes
{   NSNotificationCenter	*notificationCenter = [NSNotificationCenter defaultCenter];
    NSDictionary		*infoDict;

    /* General */
    [menuMatrix renewRows:0 columns:0];
    if ( ![NSBundle loadNibNamed:@"AstroGeneral" owner:self] )
        NSLog(@"Cannot load 'AstroGeneral' interface file");
    infoDict = [NSDictionary dictionaryWithObjectsAndKeys:PREF_GENERAL_TITLE, @"title", nil];
    [notificationCenter postNotificationName:AstroPrefsAddItemNotification
                                      object:boxGeneral userInfo:infoDict];
    [(NSBox*)subBox setContentView:boxGeneral];
    /* set target and action for GNUstep (missing in gmodels) */
    [datePopup setTarget:self];			// set the target
    [datePopup setAction:@selector(set:)];	// set the action
    [fontPopup setTarget:self];			// set the target
    [fontPopup setAction:@selector(set:)];	// set the action

    /* Objects */
    if ( ![NSBundle loadNibNamed:@"AstroObjects" owner:self] )
        NSLog(@"Cannot load 'AstroObjects' interface file");
    [notificationCenter postNotificationName:AstroPrefsAddItemNotification object:boxObjects];

    /* Chart */
    if ( ![NSBundle loadNibNamed:@"AstroChart" owner:self] )
        NSLog(@"Cannot load 'AstroChart' interface file");
    [notificationCenter postNotificationName:AstroPrefsAddItemNotification object:boxChart];

    /* Map */
    //if ( ![NSBundle loadNibNamed:@"AstroMap" owner:self] )
    //    NSLog(@"Cannot load 'AstroMap' interface file");
    //[notificationCenter postNotificationName:AstroPrefsAddItemNotification object:boxMap];

    /* call other modules to add their sub-panes */
    [notificationCenter postNotificationName:AstroPrefsCallToAddItemsNotification object:self];

    [menuMatrix selectCellAtRow:0 column:0];
}

- (NSImage*)icon
{   NSImage	*icon = nil;

    if (!icon)
    {   NSBundle	*bundle = [NSBundle bundleForClass:[self class]];
        NSString	*file = [bundle pathForResource:@"prefsAstro" ofType:@"tiff"];

        icon = [[NSImage alloc] initWithContentsOfFile:file];
    }
    return icon;
}

- (NSString*)name
{
    return @"Astro";
}

- (NSView*)view
{
    return box;
}

// end methods from protocol


/* select level
 * created: 2006-04-27
 */
- (void)setLevel:(id)sender
{   NSBox	*theView = [prefViews objectAtIndex:[menuMatrix selectedRow]];

    [(NSBox*)subBox setContentView:[theView retain]];
    //[theView setAutoresizingMask:NSViewWidthSizable|NSViewHeightSizable];

    [box display];
    //[[box window] flushWindow]; 
}


/* update panel from defaults
 */
- (void)update:sender
{   id		defaults = [NSUserDefaults standardUserDefaults];
    int		i;
    NSArray	*objects, *nodes, *objects1 = OBJECTS_1, *objects2 = OBJECTS_2;
    NSArray	*dateFormats = [[AstroDate sharedInstance] dateFormats];
    NSString	*string;

    if ( !boxGeneral )	// not loaded yet
        [self initSubPanes];

    /* General */
    for (i=0; i<[dateFormats count]; i++)
    {
        if ([[defaults objectForKey:@"astroDateFormat"] isEqual:[dateFormats objectAtIndex:i]])
            [datePopup selectItemAtIndex:i];
    }

    [fontPopup selectItemWithTitle:[defaults objectForKey:@"astroFont"]];
    /* GNUstep: workaround for uninitialized popup menus (they start with selected item = -1) */
#ifdef GNUSTEP_BASE_VERSION
    if ([fontPopup indexOfSelectedItem] == -1)
        [fontPopup selectItemAtIndex:0];
#endif

    [[generalSwitches cellAtRow:SWITCH_SHOWASTROPANEL column:0]
      setState:([[defaults objectForKey:@"astroShowPanel"] isEqual:@"YES"]) ? 1 : 0];
    [[generalSwitches cellAtRow:SWITCH_COLORFUL column:0]
      setState:([[defaults objectForKey:@"astroColorful"] isEqual:@"YES"]) ? 1 : 0];
    [[generalSwitches cellAtRow:SWITCH_DEGREESMINUTES column:0]
      setState:([[defaults objectForKey:@"astroDegreesMinutes"] isEqual:@"YES"]) ? 1 : 0];

    /* Objects */
    [[objectSwitches cellAtRow:SWITCH_TOPOCENTRIC column:0]
      setState:([[defaults objectForKey:@"astroTopocentricPosition"] isEqual:@"YES"]) ? 1 : 0];
    [[objectSwitches cellAtRow:SWITCH_MEANNODES column:0]
      setState:([[defaults objectForKey:@"astroMeanNodes"] isEqual:@"YES"]) ? 1 : 0];
    objects = [defaults objectForKey:@"astroObjects"];
    for (i=0; i<(int)[objects1 count]; i++)	// set Planet switches
    {
        [[object1Switches cellAtRow:i column:0]
             setState:[objects containsObject:[objects1 objectAtIndex:i]]];
    }
    nodes = [defaults objectForKey:@"astroNodes"];
    for (i=0; i<(int)[objects1 count]; i++)	// set Node switches
    {
        [[nodeSwitches cellAtRow:i column:0]
             setState:[nodes containsObject:[objects1 objectAtIndex:i]]];
    }
    for (i=0; i<(int)[objects2 count]; i++)	// set other objects switches
    {
        [[object2Switches cellAtRow:i column:0]
             setState:[objects containsObject:[objects2 objectAtIndex:i]]];
    }

    /* Chart */
    [[chartSwitches cellAtRow:SWITCH_SHOWRETROGRADE column:0]
      setState:([[defaults objectForKey:@"astroShowRetrograde"] isEqual:@"YES"]) ? 1 : 0];
    [[chartSwitches cellAtRow:SWITCH_LOCALSCALE column:0]
      setState:([[defaults objectForKey:@"astroLocalScale"] isEqual:@"YES"]) ? 1 : 0];
    [[chartSwitches cellAtRow:SWITCH_SHOWDECLINATION column:0]
      setState:([[defaults objectForKey:@"astroShowDeclination"] isEqual:@"YES"]) ? 1 : 0];
    [[chartSwitches cellAtRow:SWITCH_SHOWASPECTGEO column:0]
      setState:([[defaults objectForKey:@"Prefs_ShowAspectGeo"] isEqual:@"YES"]) ? 1 : 0];
    [[chartSwitches cellAtRow:SWITCH_ACFIX column:0]
      setState:([[defaults objectForKey:@"astroACFix"] isEqual:@"YES"]) ? 1 : 0];
    if ( (string = [defaults objectForKey:@"astroInterval"]) )
        [chartIntervalField setStringValue:string];

    // FIXME: setting the color sends a set action, so this must come at last
    [bgColorWell setColor:vhfRGBColorFromString([defaults objectForKey:@"astroBGColor"])];
}


/* set General defaults
 */
- (void)setGeneral:sender
{   NSString	*string;
    id		defaults = [NSUserDefaults standardUserDefaults];
    int		i;
    NSArray	*dateFormats = [[AstroDate sharedInstance] dateFormats];

    /* date format */
    i = [datePopup indexOfSelectedItem];
    if ( ![[defaults objectForKey:@"dateFormat"] isEqual:[dateFormats objectAtIndex:i]] )
        [defaults setObject:[dateFormats objectAtIndex:i] forKey:@"astroDateFormat"];

    /* font */
    string = [fontPopup titleOfSelectedItem];
    if ( ![string isEqual:[defaults objectForKey:@"astroFont"]] )
        [defaults setObject:string forKey:@"astroFont"];

    /* show Astro Panel */
    string = ([[generalSwitches cellAtRow:SWITCH_SHOWASTROPANEL column:0] state]) ? @"YES" : @"NO";
    if ( ![string isEqual:[defaults objectForKey:@"astroShowPanel"]] )
        [defaults setObject:string forKey:@"astroShowPanel"];
    /* show degrees with minutes */
    string = ([[generalSwitches cellAtRow:SWITCH_DEGREESMINUTES column:0] state]) ? @"YES" : @"NO";
    if ( ![string isEqual:[defaults objectForKey:@"astroDegreesMinutes"]] )
        [defaults setObject:string forKey:@"astroDegreesMinutes"];
    /* make colorful */
    string = ([[generalSwitches cellAtRow:SWITCH_COLORFUL column:0] state]) ? @"YES" : @"NO";
    if ( ![string isEqual:[defaults objectForKey:@"astroColorful"]] )
        [defaults setObject:string forKey:@"astroColorful"];
}

/* set Objects defaults
 */
- (void)setObjects:sender
{   id		defaults = [NSUserDefaults standardUserDefaults];
    NSString	*string;
    int		i;

    /* topocentric planet position */
    string = ([[objectSwitches cellAtRow:SWITCH_TOPOCENTRIC column:0] state]) ? @"YES" : @"NO";
    if ( ![string isEqual:[defaults objectForKey:@"astroTopocentricPosition"]] )
        [defaults setObject:string forKey:@"astroTopocentricPosition"];
    /* display mean nodes */
    string = ([[objectSwitches cellAtRow:SWITCH_MEANNODES column:0] state]) ? @"YES" : @"NO";
    if ( ![string isEqual:[defaults objectForKey:@"astroMeanNodes"]] )
        [defaults setObject:string forKey:@"astroMeanNodes"];

    /* write names of active objects */
    {   NSArray		*objects1, *objects2;
        NSMutableArray	*objects = [NSMutableArray array], *nodes = [NSMutableArray array];

        objects1 = OBJECTS_1;
        for (i=0; i<(int)[objects1 count]; i++)	// planets
            if ([[object1Switches cellAtRow:i column:0] state])
                [objects addObject:[objects1 objectAtIndex:i]];

        for (i=0; i<(int)[objects1 count]; i++)	// nodes
            if ([[nodeSwitches cellAtRow:i column:0] state])
                [nodes addObject:[objects1 objectAtIndex:i]];
        [defaults setObject:nodes forKey:@"astroNodes"];

        objects2 = OBJECTS_2;
        for (i=0; i<(int)[objects2 count]; i++)	// moon nodes, asteroids
            if ([[object2Switches cellAtRow:i column:0] state])
                [objects addObject:[objects2 objectAtIndex:i]];
        [defaults setObject:objects forKey:@"astroObjects"];
    }
}

/* set Chart defaults
 */
- (void)setChart:sender
{   NSString	*string;
    NSColor	*color;
    id		defaults = [NSUserDefaults standardUserDefaults];

    /* show retrograde directions */
    string = ([[chartSwitches cellAtRow:SWITCH_SHOWRETROGRADE column:0] state]) ? @"YES" : @"NO";
    if ( ![string isEqual:[defaults objectForKey:@"astroShowRetrograde"]] )
        [defaults setObject:string forKey:@"astroShowRetrograde"];
    /* locale scale */
    string = ([[chartSwitches cellAtRow:SWITCH_LOCALSCALE column:0] state]) ? @"YES" : @"NO";
    if ( ![string isEqual:[defaults objectForKey:@"astroLocalScale"]] )
        [defaults setObject:string forKey:@"astroLocalScale"];
    /* Declination */
    string = ([[chartSwitches cellAtRow:SWITCH_SHOWDECLINATION column:0] state]) ? @"YES" : @"NO";
    if ( ![string isEqual:[defaults objectForKey:@"astroShowDeclination"]] )
        [defaults setObject:string forKey:@"astroShowDeclination"];
    /* Aspect Geometry */
    string = ([[chartSwitches cellAtRow:SWITCH_SHOWASPECTGEO column:0] state]) ? @"YES" : @"NO";
    if ( ![string isEqual:[defaults objectForKey:@"astroShowAspectGeo"]] )
        [defaults setObject:string forKey:@"astroShowAspectGeo"];
    /* AC fix */
    string = ([[chartSwitches cellAtRow:SWITCH_ACFIX column:0] state]) ? @"YES" : @"NO";
    if ( ![string isEqual:[defaults objectForKey:@"astroACFix"]] )
        [defaults setObject:string forKey:@"astroACFix"];

    /* Interval */
    string = [chartIntervalField stringValue];
    if ( ![string isEqual:[defaults objectForKey:@"astroInterval"]] )
    	[defaults setObject:string forKey:@"astroInterval"];

    /* background color */
    color = [bgColorWell color];
    if ( ![color isEqual:vhfRGBColorFromString([defaults objectForKey:@"astroBGColor"])] )
        [defaults setObject:vhfStringFromRGBColor(color) forKey:@"astroBGColor"];
}

/* set everything (for backward compatibility)
 */
- (void)set:sender
{
    [self setGeneral:sender];
    [self setObjects:sender];
    [self setChart:sender];
}


/*
 * Notifications
 */

- (void)addItemNotification:(NSNotification*)notification
{   NSBox		*view  = [notification object];
    NSString		*title = [[notification userInfo] objectForKey:@"title"];
    NSButtonCell	*cell;
    int			row;
    NSPoint		origin = [menuMatrix frame].origin;

    origin.y += [menuMatrix frame].size.height;	// shift to upper/left corner

    if (!view)
        return;
    [prefViews addObject:view];
    //[view retain];	// GNUstep

    [menuMatrix addRow];
    row = [menuMatrix numberOfRows] - 1;
    cell = [menuMatrix cellAtRow:row column:0];
    [cell setTitle:(title) ? title : [view title]];
    [cell setState:0];
    [menuMatrix sizeToCells];
    origin.y -= [menuMatrix frame].size.height;	// back to lower/left corner
    [menuMatrix setFrameOrigin:origin];

    [box display];
    //[[box window] flushWindow];
}

@end
