/* APSearch.m
 * Controls the chart generation
 *
 * Copyright (C) 2004-2006 by vhf interservice GmbH
 * Author:   Georg Fleischmann
 *
 * Created:  2004-05-20
 * Modified: 2006-01-15
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the vhf Public License as
 * published by vhf interservice GmbH. Among other things, the
 * License requires that the copyright notices and this notice
 * be preserved on all copies.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the vhf Public License for more details.
 *
 * You should have received a copy of the vhf Public License along
 * with this program; see the file LICENSE. If not, write to vhf.
 *
 * vhf interservice GmbH, Im Marxle 3, 72119 Altingen, Germany
 * eMail: info@vhf.de
 * http://www.vhf.de
 */

#include <VHFShared/vhfCommonFunctions.h>
#include <VHFShared/VHFArrayAdditions.h>
#include "../Cenon/App.h"
#include "../Cenon/messages.h"
#include "../Cenon/functions.h"
#include "AstroPanel.h"
#include "APSearch.h"
#include "APChart.h"
#include "../astroMessages.h"
#include "../astroCommon.h"
#include "../AstroPrincipal.h"
#include "../AstroDate.h"			// date formats
#include "../Astro.bproj/AstroController.h"	// Preferences

@interface APSearch(PrivateMethods)
@end

@implementation APSearch

- init
{   Ephemeris	*ephemeris = [[AstroPrincipal instance] ephemeris];

    [self setDelegate:self];
    [textView setDelegate:self];
    ephemerisSearch = [EphemerisSearch newWithEphemeris:ephemeris];

    planetList = [[NSArray alloc] initWithObjects:@"Sun", @"Moon", @"Mercury", @"Venus", @"Mars",
                                                  @"Jupiter", @"Saturn", @"Uranus", @"Neptune", @"Pluto",
                                                  @"True Node", @"True Apogee", @"Chiron", @"Ecliptic", nil];

    /* GNUstep: workaround for uninitialized popup menus (they start with selected item = -1) */
#ifdef GNUSTEP_BASE_VERSION
    [planet1Popup selectItemAtIndex:1];
    [planet2Popup selectItemAtIndex:0];
    [searchTypePopup selectItemAtIndex:0];
#endif

    [self update];

    return [super init];
}

- (void)update:sender
{   NSString	*string = [fromField stringValue];

    /* wrong date format -> update */
    if ( [string characterAtIndex:2] != [Prefs_DateFormat characterAtIndex:2] )
    {
        [fromField setStringValue:[[NSCalendarDate date] descriptionWithCalendarFormat:Prefs_DateFormat]];
        [toField   setStringValue:[[NSCalendarDate date] descriptionWithCalendarFormat:Prefs_DateFormat]];
    }
}

/* parse dividers and return array of single dividers
 */
- (NSArray*)dividers
{   NSScanner		*scanner = [NSScanner scannerWithString:[dividerField stringValue]];
    NSMutableArray	*dividers = [NSMutableArray array];
    NSString		*string;
    NSCharacterSet	*whitespaceCharacterSet = [NSCharacterSet whitespaceCharacterSet];
    NSRange		range;

    while (![scanner isAtEnd])
    {
        if (![scanner scanUpToCharactersFromSet:whitespaceCharacterSet intoString:&string])
            break;

        /* '-' -> range of dividers */
        range = [string rangeOfString:@"-"];
        if (range.length)
        {   int	d, dMin, dMax;

            dMin = [[string substringToIndex:range.location] intValue];
            dMax = Min([[string substringFromIndex:range.location+1] intValue], 360);
            for (d=dMin; d<=dMax; d++)
                [dividers addObject:[NSNumber numberWithInt:d]];
        }
        /* '#' -> divider */
        else
            [dividers addObject:[NSNumber numberWithInt:[string intValue]]];
    }
    return dividers;
}

- (NSCalendarDate*)dateWithString:(NSString*)string
{   NSString		*dateFormat = astroDateFormatForString(string);
    NSCalendarDate	*date = [NSCalendarDate dateWithString:string calendarFormat:dateFormat];

    if (!date)
        NSRunAlertPanel(@"", WRONGDATEFORMAT_STRING, OK_STRING, nil, nil, astroErrorDate());
    return date;
}

/* create aspect list
 */
- (void)set:(id)sender
{
    [aspectList release]; aspectList = nil;
    [self add:sender];
}

/* add to aspect list
 */
- (void)add:(id)sender
{   NSCalendarDate	*dateFrom = [self dateWithString:[fromField stringValue]];
    NSCalendarDate	*dateTo   = [self dateWithString:[toField   stringValue]];
    NSDictionary	*searchDict = nil;
    NSArray		*dividers = [self dividers];
    NSString		*planet[2], *aspectStr;
    BOOL		searchAngles  = [[aspectSwitches cellAtRow:SHOW_ASPECTS column:0] state];
    BOOL		searchMirrors = [[aspectSwitches cellAtRow:SHOW_MIRRORS column:0] state];
    int			i;
    NSString		*dateFormat = [NSString stringWithFormat:@"%@ %%H:%%M", Prefs_DateFormat];
    NSMutableString	*mString = [NSMutableString string];
    BOOL		displaySymbols = YES, displayAngles = NO;
    NSMutableAttributedString	*attrString = [[NSMutableAttributedString new] autorelease];
    NSMutableAttributedString	*aString    = [[NSMutableAttributedString new] autorelease];
    NSFont			*systemFont = [NSFont systemFontOfSize:12.0];
    NSFont			*symbolFont = [NSFont fontWithName:Prefs_AstroFont size:12.0];

    if (!dateFrom || !dateTo)
        return;

    planet[0] = [planetList objectAtIndex:[planet1Popup indexOfSelectedItem]];
    planet[1] = [planetList objectAtIndex:[planet2Popup indexOfSelectedItem]];

    if (!symbolFont)
        [NSFont fontWithName:@"vhfAstro" size:12.0];
    if (!aspectList)
        aspectList = [NSMutableDictionary new];

    if ([searchTypePopup indexOfSelectedItem] == SEARCH_ASPECTS)
    {   AstroPanel	*astroPanel = [Astro_Principal astroPanel];
        APChart		*astroChart = (APChart*)[astroPanel windowAt:AP_CHART];

        /* if planet1 or planet2 == Node or Apogee, we have to check, if true or mean */
        for (i=0; i<2; i++)
            if ( ! [Prefs_Objects containsObject:planet[i]] )
            {
                if ([planet[i] isEqual:@"True Node"])
                    planet[i] = @"Node";	// Mean Node
                else if ([planet[i] isEqual:@"True Apogee"])
                    planet[i] = @"Apogee";	// Mean Apogee
            }
        [ephemerisSearch setTopocentricSearch:Prefs_Topocentric
                                          lat:[astroChart latitude] lon:[astroChart longitude] elev:0];
        searchDict = [ephemerisSearch searchAspect:planet[0] :planet[1]
                                          dividers:dividers
                                      searchAngles:searchAngles searchMirrors:searchMirrors
                                          fromDate:dateFrom toDate:dateTo
                                    searchInterval:30];
    }
    else	// search for transits
        ; // FIXME

    if (!searchDict)
        return;
    [aspectList addEntriesFromDictionary:searchDict];

    [aspectKeys release];
    aspectKeys = [[[aspectList allKeys] sortedArrayUsingSelector:@selector(compare:)] retain];
    for (i=0; i<[aspectKeys count]; i++)
    {   NSCalendarDate	*key = [aspectKeys objectAtIndex:i];
        NSArray		*entry = [aspectList objectForKey:key];

        if (displaySymbols)
        {   NSString	*string;

            string = [NSString stringWithFormat:@"%@  ", [key descriptionWithCalendarFormat:dateFormat]];
            [aString replaceCharactersInRange:NSMakeRange(0, [aString length]) withString:string];
            [aString addAttribute:NSFontAttributeName value:systemFont range:NSMakeRange(0, [aString length])];
            [attrString appendAttributedString:aString];

            aspectStr = aspectSymbol([[entry objectAtIndex:3] intValue], 1.0);
            if (!aspectStr)
            {   //double	v = [entry doubleAtIndex:3];
                //aspectStr = [NSString stringWithFormat:(v == floor(v)) ? @"%.0f" : @"%.1f", v];
                aspectStr = [entry objectAtIndex:2];	// divider
            }
            string = [NSString stringWithFormat:@"%@ %@ %@", planetSymbol([entry objectAtIndex:0]),
                                                             aspectStr,
                                                             planetSymbol([entry objectAtIndex:1])];
            [aString replaceCharactersInRange:NSMakeRange(0, [aString length]) withString:string];
            [aString addAttribute:NSFontAttributeName value:symbolFont range:NSMakeRange(0, [aString length])];
            [attrString appendAttributedString:aString];

            if ([entry count] > 4)	// mirror
            {
                [aString replaceCharactersInRange:NSMakeRange(0, [aString length]) withString:@" (M)\n"];
                [aString addAttribute:NSFontAttributeName value:systemFont range:NSMakeRange(0, [aString length])];
                [attrString appendAttributedString:aString];
            }
            else			// new line
            {
                [aString replaceCharactersInRange:NSMakeRange(0, [aString length]) withString:@"\n"];
                [attrString appendAttributedString:aString];
            }
        }
        else	// FIXME: localize planets + format planets to equal lenth
        {
            if (displayAngles)
            {   double	v = [entry doubleAtIndex:3];
                aspectStr = [NSString stringWithFormat:(v == floor(v)) ? @"%.0f" : @"%.1f", v];
            }
            else
                aspectStr = [entry objectAtIndex:2];	// divider
            [mString appendFormat:@"%@  %@ %@ %@%@\n", [key descriptionWithCalendarFormat:dateFormat],
                                                       [entry objectAtIndex:0],
                                                       aspectStr,
                                                       [entry objectAtIndex:1],
                                                       ([entry count] > 4) ? @" (M)" : @""];
        }
    }

    if (displaySymbols)
        [[textView textStorage] setAttributedString:attrString];
    else
        [textView setString:mString];
}



/*
 * notifications and delegate methods
 */

/* calculate chart for selected aspect
 */
- (void)textViewDidChangeSelection:(NSNotification *)aNotification
{   int			loc = [textView selectedRange].location;
    NSString		*string = [textView string];
    int			line = [string countOfCharacter:'\n' inRange:NSMakeRange(0, loc)];
    NSCalendarDate	*date;
    APChart		*apChart = [[Astro_Principal astroPanel] windowAt:AP_CHART];
    NSMutableDictionary	*dict;
    NSDictionary	*composite;

    if (![aspectKeys count])
        return;
    date = [aspectKeys objectAtIndex:(line < [aspectKeys count]) ? line : [aspectKeys count]-1];

    dict = [apChart data];
    [dict setObject:date forKey:@"date"];
    if ( (composite = [apChart compositeData]) )
        [dict setObject:composite forKey:@"composite"];
    [[NSNotificationCenter defaultCenter] postNotificationName:CalculateChartNotification object:dict];
}


/* created:  1993-07-22
 * modified: 2004-05-20
 */
- (void)dealloc
{
    [planetList release];
    [aspectList release];
    [super dealloc];
}

@end
