/* AAFImport.m
 * import of AAF astrological database format
 *
 * Copyrigth (C) 2003-2008 by Georg Fleischmann
 * Author:   Georg Fleischmann
 *
 * created:  2003-07-02
 * modified: 2008-12-20
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the vhf Public License as
 * published by vhf interservice GmbH. Among other things, the
 * License requires that the copyright notices and this notice
 * be preserved on all copies.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the vhf Public License for more details.
 *
 * You should have received a copy of the vhf Public License along
 * with this program; see the file LICENSE. If not, write to vhf.
 *
 * vhf interservice GmbH, Im Marxle 3, 72119 Altingen, Germany
 * eMail: info@vhf.de
 * http://www.vhf.de
 *
 *
 * Example Entry:
 *
 * #A93:Lee,Bruce,*,27.11.1940g,07:12,San Francisco -CA-,USA
 * #B93:*,37n47,122w26,08hw00:00,m
 * #CWORD:FS,(Film-) Schauspieler
 * #VIA:RO,Datenbanken von L. M. Rodden
 * #SUB1_EV:TD,Todesdatum,e,20.7.1973g,*,*,*
 */

#include <Foundation/Foundation.h>
#include <math.h>
#include <VHFShared/types.h>
#include <VHFShared/vhfCommonFunctions.h>
#include <VHFShared/vhfStringAdditions.h>
#include "AAFImport.h"

/* Private methods
 */
@interface AAFImport(PrivateMethods)
@end

@implementation AAFImport

/* HHnMM, HHsMM, HHwMM/HHHwMM, HHoMM:SS -> HH. */
static NSString *geoToDegree(NSString *geoString)
{   NSString	*sep;
    int		i;
    float	h, len = [geoString length];

    if (len <= 1)	// '*'
        return nil;
    if (len < 5)
    {   NSLog(@"lat/lon string '%@' uncomplete, ignoring", geoString);
        return nil;
    }
    for (i=0; i<3; i++)
        if (!isdigit([geoString characterAtIndex:i]))
            break;
    h = (float)[[geoString substringToIndex:i] intValue];
    sep = [geoString substringWithRange:NSMakeRange(i,1)];
    h += (float)[[geoString substringWithRange:NSMakeRange(i+1,2)] intValue] / 60.0;
    if (len >= 8)
        h += (float)[[geoString substringWithRange:NSMakeRange(i+4,2)] intValue] / 3600.0;

    switch ([sep characterAtIndex:0])
    {
        case 's':
        case 'S':
        case 'w':
        case 'W':
            h = -h;
    }

    return vhfStringWithFloat(h);
}

/* "08hw00:00" or "8hw00:00"
 */
static NSString *timeZoneFromString(NSString *zoneString)
{   NSString    *sep, *h, *m, *string = nil;
    int         len = [zoneString length];
    NSRange     range;

    if (len < 6)
    {   NSLog(@"zone string '%@' uncomplete, ignoring", zoneString);
        return nil;
    }

    range = [zoneString rangeOfString:@"h"];
    if ( ! range.length )
    {   NSLog(@"AAF-Import: 'h' expected in %@, using UTC !", zoneString);
        return @"+0000";
    }
    h   = [zoneString substringToIndex:range.location];
    sep = [zoneString substringWithRange:NSMakeRange(range.location+1, 1)];
    m   = [zoneString substringWithRange:NSMakeRange(range.location+2, 2)];

    switch ([sep characterAtIndex:0])
    {
        case 'e':
        case 'E':
            string = [NSString stringWithFormat:@"+%@%@", h, m];
            break;
        default:
            string = [NSString stringWithFormat:@"-%@%@", h, m];
    }
    return string;
}
/*static NSCalendarDate *addZoneToDate(NSString *zoneString, NSCalendarDate *date)
{   NSString		*sep;
    int			s, len = [zoneString length];

    if (len < 6)
    {   NSLog(@"zone string '%@' uncomplete, ignoring", zoneString);
        return nil;
    }
    s = [[zoneString substringToIndex:2] intValue] * 3600;
    sep = [zoneString substringWithRange:NSMakeRange(3,1)];
    s += [[zoneString substringWithRange:NSMakeRange(4,2)] intValue] * 60;
    if (len == 9)
        s += [[zoneString substringWithRange:NSMakeRange(7,2)] intValue];

    switch ([sep characterAtIndex:0])
    {
        case 'e':
        case 'E':
            s = -s;
    }

    return [date dateByAddingYears:0 months:0 days:0 hours:0 minutes:0 seconds:s];
}*/



+ (NSMutableArray*)dictionaryWithAAFFromFile:(NSString*)fileName
{
    return [[[self new] autorelease] dictionaryWithAAFFromFile:fileName];
}


- (NSMutableArray*)dictionaryWithAAFFromFile:(NSString*)fileName
{
    if ( !(aafData = [NSString stringWithContentsOfFile:fileName]) )
    {	NSLog(@"Couldn't load AAF file '%@'", fileName);
        return nil;
    }
    return [self dictionaryWithAAF:aafData];
}

- (NSMutableArray*)dictionaryWithAAF:(NSString*)aafString
{   NSScanner           *scanner = [NSScanner scannerWithString:aafString];
    NSString            *string, *string1, *dateStr = nil;
    NSMutableDictionary *dict = nil;
    NSCalendarDate      *date = nil, *date1;
    NSCharacterSet      *lineEndCharacterSet = [NSCharacterSet characterSetWithCharactersInString:@"\r\n#"];
    NSAutoreleasePool   *pool;
    int                 n = 0;

    data = [NSMutableArray array];
    pool = [NSAutoreleasePool new];
    while (![scanner isAtEnd])
    {
        [scanner scanUpToString:@"#" intoString:NULL];

        /* extract data set */
        if (![scanner scanUpToString:@":" intoString:&string])
            break;
        [scanner scanString:@":" intoString:NULL];

        /* last name, name, type, date, time, city, country id */
        if ([string isEqual:@"#A93"])
        {
            if ([[dict objectForKey:@"name"] length])
                [data addObject:dict];
            if (!(n++ % 500))
            {   [pool release];
                pool = [NSAutoreleasePool new];
                date = nil;
                dateStr = nil;
            }
            dict = [NSMutableDictionary dictionary];

            /* "last name, name" */
            [scanner scanUpToString:@"," intoString:&string];
            [scanner setScanLocation:[scanner scanLocation]+1];
            if ([string length] <= 1)	// '*'
                string = nil;
            [scanner scanUpToString:@"," intoString:&string1];
            [scanner setScanLocation:[scanner scanLocation]+1];
            if ([string1 length] <= 1)	// '*'
                string1 = nil;
            else if (!string)
                string = string1;
            else if (string && string1)
                string = [NSString stringWithFormat:@"%@, %@", string, string1];
            [dict setObject:string forKey:@"name"];

            /* type (f/w=female, m=male, e=event, l=land, o=organisation) */
            [scanner scanUpToString:@"," intoString:&string];
            [scanner setScanLocation:[scanner scanLocation]+1];
            switch ([string characterAtIndex:0])
            {
                case '*':
                    break;
                case 'w':   // Weiblich
                case 'W':
                case 'f':   // Female
                case 'F':
                    [dict setObject:@"f" forKey:@"type"]; break;
                case 'm':   // Male
                case 'M':
                    [dict setObject:@"m" forKey:@"type"]; break;
                case 'E':   // Event
                    [dict setObject:@"e" forKey:@"type"]; break;
                case 'O':   // 
                    [dict setObject:@"o" forKey:@"type"]; break;
                default:
                    [dict setObject:string forKey:@"type"];
            }

            /* date, time */
            [scanner scanUpToString:@"," intoString:&string];
            [scanner setScanLocation:[scanner scanLocation]+1];
            if ( ! isdigit([string characterAtIndex:[string length]-1]) )
                string = [string substringToIndex:[string length]-1];   // remove g, j from date
            [scanner scanUpToString:@"," intoString:&string1];
            [scanner setScanLocation:[scanner scanLocation]+1];
            dateStr = [string stringByAppendingFormat:@" %@", string1];
            // FIXME: handle years BC (-106)
            date = [NSCalendarDate dateWithString:dateStr calendarFormat:@"%d.%m.%Y %H:%M"];
            if (!date)
                NSLog(@"AAF-Import: Problems with date '%@'", dateStr);

            /* city */
            if ([scanner scanUpToString:@"," intoString:&string])
                [dict setObject:string forKey:@"city"];
            [scanner setScanLocation:[scanner scanLocation]+1];

            /* state id */
            if ([scanner scanUpToCharactersFromSet:lineEndCharacterSet intoString:&string])
                [dict setObject:string forKey:@"country"];
        }
        /* julian date incl. time, lat, lon, time zone diff, sommer time */
        else if ([string isEqual:@"#B93"])
        {
            /* julian date */
            [scanner scanUpToString:@"," intoString:NULL];
            [scanner setScanLocation:[scanner scanLocation]+1];

            /* lat */
            if ([scanner scanUpToString:@"," intoString:&string])
                if (([string = geoToDegree(string) length]))
                    [dict setObject:string forKey:@"lat"];
            [scanner setScanLocation:[scanner scanLocation]+1];

            /* lon */
            if ([scanner scanUpToString:@"," intoString:&string])
                if (([string = geoToDegree(string) length]))
                    [dict setObject:string forKey:@"lon"];
            [scanner setScanLocation:[scanner scanLocation]+1];

            /* time zone '00hw00:19', here we add the date with or without time zone */
            if ( [scanner scanUpToString:@"," intoString:&string]
                 && date && (string1 = timeZoneFromString(string)) )
            {
                dateStr = [NSString stringWithFormat:@"%@ %@", dateStr, string1];
                if ( [dateStr appearanceCountOfCharacter:':'] > 1 ) // seconds
                    date1 = [NSCalendarDate dateWithString:dateStr calendarFormat:@"%d.%m.%Y %H:%M:%S %z"];
                else
                    date1 = [NSCalendarDate dateWithString:dateStr calendarFormat:@"%d.%m.%Y %H:%M %z"];
                if (date1)
                    [dict setObject:[date1 descriptionWithCalendarFormat:@"%Y-%m-%d %H:%M %z"] forKey:@"date"];
                else
                    NSLog(@"AAF-Import: Problems with date '%@'", dateStr);
            }
            else if (date)
                [dict setObject:[date descriptionWithCalendarFormat:@"%Y-%m-%d %H:%M +0000"] forKey:@"date"];
        }
        else if ([string isEqual:@"#CWORD"] || [string isEqual:@"#COM"] )
        {
            if ([scanner scanUpToCharactersFromSet:lineEndCharacterSet intoString:&string])
            {
                if ( [[dict objectForKey:@"text"] length] > 1 )
                    string = [string stringByAppendingFormat:@"\n%@", string];
                [dict setObject:string forKey:@"text"];
            }
        }
        /*else if ([string isEqual:@"#SUB1_EV"])
        {
        }*/
    }
    if ([[dict objectForKey:@"name"] length])
        [data addObject:dict];
    [pool release];
    return data;
}

@end
