/* APChart.m
 * Controls the chart generation
 *
 * Copyright (C) 2001-2008 by vhf interservice GmbH
 * Author:   Georg Fleischmann
 *
 * Created:  2001-05-18
 * Modified: 2008-02-02 (save settings from last run)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the vhf Public License as
 * published by vhf interservice GmbH. Among other things, the
 * License requires that the copyright notices and this notice
 * be preserved on all copies.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the vhf Public License for more details.
 *
 * You should have received a copy of the vhf Public License along
 * with this program; see the file LICENSE. If not, write to vhf.
 *
 * vhf interservice GmbH, Im Marxle 3, 72119 Altingen, Germany
 * eMail: info@vhf.de
 * http://www.vhf.de
 */

#include <VHFShared/vhfCommonFunctions.h>
#include "../Cenon/App.h"
#include "../Cenon/locations.h"
#include "../Cenon/messages.h"
#include "../Cenon/functions.h"
#include "AstroPanel.h"
#include "APChart.h"
#include "../AstroDate.h"
#include "../AstroChart.h"
#include "../astroLocations.h"
#include "../EventManager.h"
#include "../AstroPrincipal.h"

#define EVENTFILE	@"event.data"	// list of events

@interface APChart(PrivateMethods)
@end

@implementation APChart

- init
{   NSNotificationCenter	*notificationCenter = [NSNotificationCenter defaultCenter];

    [self setDelegate:self];

    /* add notification to set event */
    [notificationCenter addObserver:self
                           selector:@selector(updateEvent:)
                               name:AstroUpdateEvent
                             object:nil];

    /* reload settings from last time */
    {   NSUserDefaults  *defaults = [NSUserDefaults standardUserDefaults];
        NSDateFormatter *dateFormatter = [[dateField cell] formatter];
        NSString        *city, *name, *str;
        float           lat, lon;

        if ([defaults objectForKey:@"astroPanelChartLat"])
        {
            if ( (name = [defaults objectForKey:@"astroPanelChartName"]) )
                [nameField setStringValue:name];
            if ( (str = [defaults objectForKey:@"astroPanelChartDate"]) )
            {   [[dateField cell] setFormatter:dateFormatter];
                [dateField setStringValue:str];
            }
            if ( (str = [defaults objectForKey:@"astroPanelChartZone"]) )
                [zoneField setStringValue:str];
            if ( (city = [defaults objectForKey:@"astroPanelChartCity"]) )
                [cityField setStringValue:city];
            if ( (lat  = [defaults floatForKey: @"astroPanelChartLat"]) )
                [latField setStringValue:vhfStringWithFloat(lat)];
            if ( (lon  = [defaults floatForKey: @"astroPanelChartLon"]) )
                [lonField setStringValue:vhfStringWithFloat(lon)];
        }
        if ([defaults objectForKey:@"astroPanelChartCLat"])
        {
            if ( (name = [defaults objectForKey:@"astroPanelChartCName"]) )
                [cmpNameField setStringValue:name];
            if ( (str = [defaults objectForKey:@"astroPanelChartCDate"]) )
            {   [[cmpDateField cell] setFormatter:dateFormatter];
                [cmpDateField setStringValue:str];
            }
            if ( (str = [defaults objectForKey:@"astroPanelChartCZone"]) )
                [cmpZoneField setStringValue:str];
            if ( (city = [defaults objectForKey:@"astroPanelChartCCity"]) )
                [cmpCityField setStringValue:city];
            if ( (lat  = [defaults floatForKey: @"astroPanelChartCLat"]) )
                [cmpLatField setStringValue:vhfStringWithFloat(lat)];
            if ( (lon  = [defaults floatForKey: @"astroPanelChartCLon"]) )
                [cmpLonField setStringValue:vhfStringWithFloat(lon)];
        }
    }

    [self setComposite:self];
    [self update:self];
    return [super init];
}

- (void)update:sender
{   NSString            *dateFormat = [[[AstroDate sharedInstance] dateFormat] stringByAppendingString:@" %H:%M"];
    NSDateFormatter     *dateFormatter = [[dateField cell] formatter];
    //NSNumberFormatter   *zoneFormatter = [[zoneField cell] formatter];

    /* set date formatter */
    if ( !dateFormat || ![dateFormat isEqual:[dateFormatter dateFormat]] )
    {   NSCalendarDate	*date = [self date];

//printf("Update: '%s' -> '%s'\n", [[formatter dateFormat] cString], [dateFormat cString]);
        dateFormatter = [[[NSDateFormatter  alloc] initWithDateFormat:dateFormat allowNaturalLanguage:NO] autorelease];
/*        zoneFormatter = [[NSNumberFormatter alloc] autorelease];
#ifdef __APPLE__
        [zoneFormatter setFormatterBehavior:NSNumberFormatterBehavior10_0];
#endif
        [zoneFormatter setFormat:@"+##;+00;-##"];    // doesn't work!
        [zoneFormatter setAllowsFloats:NO];*/

        date = [self date];
        [[dateField cell] setFormatter:dateFormatter];
        [dateField setStringValue:[date descriptionWithCalendarFormat:dateFormat]];
        //[[zoneField cell] setFormatter:zoneFormatter];
        [zoneField setStringValue:[[date descriptionWithCalendarFormat:@"%z"] substringToIndex:3]];

        date = [self compositeDate];
        [[cmpDateField cell] setFormatter:dateFormatter];
        [cmpDateField setStringValue:[date descriptionWithCalendarFormat:dateFormat]];
        //[[cmpZoneField cell] setFormatter:zoneFormatter];
        [cmpZoneField setStringValue:[[date descriptionWithCalendarFormat:@"%z"] substringToIndex:3]];
    }
}

/* Astro Clock
 * created: 2006-06-02
 */
- (void)setClock:(id)sender
{
    if ([sender state])	// start clock timer
    {
        if (!timer)
            timer = [NSTimer scheduledTimerWithTimeInterval:60.0
                                                     target:self selector:@selector(set:)
                                                   userInfo:nil repeats:YES];
    }
    else		// stop clock timer
    {
        [timer invalidate];
        timer = nil;
    }
}

/* start calculation of chart
 * modified: 2008-02-02 (save settings)
 */
- (void)set:(id)sender
{   NSCalendarDate  *date = [self date];
    float           lat = [latField floatValue], lon = [lonField floatValue];
    Document        *doc = [(App*)NSApp currentDocument];
    DocView         *view;

    /* Astro Clock: use current time */
    if ( [sender isKindOfClass:[NSTimer class]] )
    {   int zoneOffset = [zoneField intValue] * 3600;

        date = [NSCalendarDate date];
        [date setTimeZone:[NSTimeZone timeZoneForSecondsFromGMT:zoneOffset]];
    }
    /* save settings */
    else
    {   NSUserDefaults  *defaults = [NSUserDefaults standardUserDefaults];

        [defaults setObject:[nameField stringValue]         forKey:@"astroPanelChartName"];
        [defaults setObject:[dateField stringValue]         forKey:@"astroPanelChartDate"];
        [defaults setObject:[zoneField stringValue]         forKey:@"astroPanelChartZone"];
        [defaults setObject:[self city]                     forKey:@"astroPanelChartCity"];
        [defaults setObject:[NSNumber numberWithFloat:lat]  forKey:@"astroPanelChartLat"];
        [defaults setObject:[NSNumber numberWithFloat:lon]  forKey:@"astroPanelChartLon"];
        if ([self composite])
        {
            [defaults setObject:[cmpNameField stringValue]  forKey:@"astroPanelChartCName"];
            [defaults setObject:[cmpDateField stringValue]  forKey:@"astroPanelChartCDate"];
            [defaults setObject:[cmpZoneField stringValue]  forKey:@"astroPanelChartCZone"];
            [defaults setObject:[cmpCityField stringValue]  forKey:@"astroPanelChartCCity"];
            [defaults setObject:[cmpLatField  stringValue]  forKey:@"astroPanelChartCLat"];
            [defaults setObject:[cmpLonField  stringValue]  forKey:@"astroPanelChartCLon"];
        }
    }

    /* search a chart */
    if ( ![[doc name] hasPrefix:CHART_PREFIX] )
    {   int	i;

        for ( i=[[NSApp windows] count]-1; i>=0; i-- )
        {   doc = [(App*)NSApp documentInWindow:[[(App*)NSApp windows] objectAtIndex:i]];
            if ( [[doc window] isVisible] && [[doc name] hasPrefix:CHART_PREFIX] )
                break;
        }
        if ( i < 0 )
        {   NSFileManager	*fileManager = [NSFileManager defaultManager];
            NSString		*path = nil, *relPath;
            NSString		*file = [CHART_FILE stringByAppendingPathExtension: DOCUMENT_EXT];

            relPath = vhfPathWithPathComponents(@"Astro", @"Charts", file, nil);
            for (i=0; i<3; i++)
            {
                switch (i)
                {
                    case 0:	// Home Library
                        path = vhfPathWithPathComponents(userLibrary(),  relPath, nil);
                        break;
                    case 1:	// Local Library
                        path = vhfPathWithPathComponents(localLibrary(), relPath, nil);
                        break;
                    default:
                        NSLog(@"APChart: No document found with prefix '%@'!", CHART_PREFIX);
                        return;
                }
                if ( [fileManager fileExistsAtPath:path] )
                    break;
            }
            [(App*)NSApp openFile:path];	// open default Flow Chart
            doc = [(App*)NSApp openedDocument];
        }
    }
    view = [doc documentView];


    if (!astroChart && view && [[doc name] hasPrefix:CHART_PREFIX] )
        astroChart = [[AstroChart astroChartWithView:view] retain];
    else if (astroChart && view && [[doc name] hasPrefix:CHART_PREFIX] )
        [astroChart setView:view];
    [astroChart radix:date longitude:lon latitude:lat city:[self city]
                title:[nameField stringValue] composite:[self compositeData]];

    /* calculate chart for date */
    [[NSNotificationCenter defaultCenter] postNotificationName:AstroDateSetNotification object:date];
}

- (id)astroChart
{
    return astroChart;
}

/*
 * chart methods
 */

/* open the database panel
 */
- (void)openDatabase:sender
{
    [[Astro_Principal eventManager] showPanel:self];
}

/* set current time and date
 */
- (void)setNow:sender
{   NSCalendarDate	*date = [NSCalendarDate date];
    NSString		*dateFormat = [[AstroDate sharedInstance] dateFormat];
    NSString		*calendarFormat = [dateFormat stringByAppendingString:@" %H:%M"];

    if ([compositeSwitch state])
    {   int zoneOffset = [cmpZoneField intValue] * 3600;

        [date setTimeZone:[NSTimeZone timeZoneForSecondsFromGMT:zoneOffset]];
        [cmpDateField setStringValue:[date descriptionWithCalendarFormat:calendarFormat]];
        [cmpZoneField setStringValue:[[date descriptionWithCalendarFormat:@"%z"] substringToIndex:3]];
        [cmpDateField display];
    }
    else
    {   int zoneOffset = [zoneField intValue] * 3600;

        [date setTimeZone:[NSTimeZone timeZoneForSecondsFromGMT:zoneOffset]];
        [dateField setStringValue:[date descriptionWithCalendarFormat:calendarFormat]];
        [zoneField setStringValue:[[date descriptionWithCalendarFormat:@"%z"] substringToIndex:3]];
        [dateField display];
    }
    [self set:self];
}

- (void)setDate:sender
{   BOOL            control = [(App*)NSApp control];
    int             m = 0;
    NSCalendarDate  *date = [self date];

    if ([sender isKindOfClass:[NSButton class]])	// buttons
    {   NSString	*dateFormat = [[AstroDate sharedInstance] dateFormat];
        NSString	*calendarFormat = [dateFormat stringByAppendingString:@" %H:%M"];

        switch ([sender tag])
        {
            case BUTTONLEFT:	m = ((control) ? -60*24 : -60); break;
            case BUTTONRIGHT:	m = ((control) ?  60*24 :  60);
        }
        date = [date dateByAddingYears:0 months:0 days:0 hours:0 minutes:m seconds:0];
        [dateField setStringValue:[date descriptionWithCalendarFormat:calendarFormat]];
        [zoneField setStringValue:[[date descriptionWithCalendarFormat:@"%z"] substringToIndex:3]];
        [dateField display];
    }
    [self set:self];
}

- (void)setEvent:(NSDictionary*)event
{   id              dateValue;
    NSString        *string;
    NSCalendarDate  *date = nil;
    NSString        *dateFormat = [[AstroDate sharedInstance] dateFormat];
    NSString        *calendarFormat = [dateFormat stringByAppendingString:@" %H:%M"];

    [nameField setStringValue:[event stringForKey:@"name"]];
    [cityField setStringValue:[event stringForKey:@"city"]];
    [latField setStringValue:vhfStringWithFloat([event floatForKey:@"lat"])];
    [lonField setStringValue:vhfStringWithFloat([event floatForKey:@"lon"])];
    if ( (dateValue = [event objectForKey:@"date"]) )
    {
        if ([dateValue isKindOfClass:[NSCalendarDate class]])
            date = [event objectForKey:@"date"];
        else
        {
            string = dateValue;
            if ( isdigit([string characterAtIndex:[string length]-1]) )
                date = [NSCalendarDate dateWithString:string calendarFormat:@"%Y-%m-%d %H:%M %z"];
            else
                date = [NSCalendarDate dateWithString:string calendarFormat:@"%Y-%m-%d %H:%M %Z"];
        }
    }
    if (!date)
        date = [NSCalendarDate date];
    [dateField setStringValue:[date descriptionWithCalendarFormat:calendarFormat]];
    [zoneField setStringValue:[[date descriptionWithCalendarFormat:@"%z"] substringToIndex:3]];
    [self set:self];
}


- (NSString*)title
{
    return [nameField stringValue];
}
- (NSString*)city
{
    return [cityField stringValue];
}
- (float)latitude
{
    return [latField floatValue];
}
- (float)longitude
{
    return [lonField floatValue];
}
- (NSCalendarDate*)date
{   NSString        *dateStr = [dateField stringValue], *tzStr = [zoneField stringValue];
    NSString		*dateFormat = astroDateFormatForString([dateField stringValue]);
    NSString		*calendarFormat = [dateFormat stringByAppendingString:@" %H:%M %z"];
    NSCalendarDate	*date;

    date = [NSCalendarDate dateWithString:[NSString stringWithFormat:@"%@ %@00", dateStr, tzStr]
                           calendarFormat:calendarFormat];
    if (!date)
        date = [NSCalendarDate date];
    return date;
}

- (NSMutableDictionary*)data
{   NSMutableDictionary	*dict = [NSMutableDictionary dictionary];

    [dict setObject:[cityField stringValue] forKey:@"city"];
    [dict setObject:[latField  stringValue] forKey:@"lat"];
    [dict setObject:[lonField  stringValue] forKey:@"lon"];
    [dict setObject:[nameField stringValue] forKey:@"name"];
    [dict setObject:[self date]             forKey:@"date"];
    return dict;
}



/*
 * composite methods
 */

- (void)setComposite:sender
{
    [cmpNameField setEnabled:[self composite]];
    [cmpCityField setEnabled:[self composite]];
    [cmpLatField  setEnabled:[self composite]];
    [cmpLonField  setEnabled:[self composite]];
    [cmpDateField setEnabled:[self composite]];
    [cmpZoneField setEnabled:[self composite]];
    [cmpBCButton  setEnabled:[self composite]];
    [[cmpBCButton superview] display];	// redraw composite switch ontop of box

    /* enable/disable composite layer */
    [astroChart setCompositeLayerEnabled:[self composite]];
}

- (BOOL)composite
{
    return [compositeSwitch state];
}

- (void)setCompositeDate:sender
{   BOOL		control = [(App*)NSApp control];
    int			m = 0;
    NSCalendarDate	*date = [self compositeDate];

    if ([sender isKindOfClass:[NSButton class]])	// buttons
    {   NSString	*dateFormat = [[AstroDate sharedInstance] dateFormat];
        NSString	*calendarFormat = [dateFormat stringByAppendingString:@" %H:%M"];

        switch ([sender tag])
        {
            case BUTTONLEFT:	m = ((control) ? -60*24 : -60); break;
            case BUTTONRIGHT:	m = ((control) ?  60*24 :  60);
        }
        date = [date dateByAddingYears:0 months:0 days:0 hours:0 minutes:m seconds:0];
        [cmpDateField setStringValue:[date descriptionWithCalendarFormat:calendarFormat]];
        [cmpZoneField setStringValue:[[date descriptionWithCalendarFormat:@"%z"] substringToIndex:3]];
        [cmpDateField display];
    }
    [self set:self];	// calculate chart
}

- (void)setCompositeEvent:(NSDictionary*)event
{   NSString		*string;
    NSCalendarDate	*date = nil;
    NSString		*dateFormat = [[AstroDate sharedInstance] dateFormat];
    NSString		*calendarFormat = [dateFormat stringByAppendingString:@" %H:%M"];

    [compositeEvent release];
    compositeEvent = [event retain];

    [cmpNameField setStringValue:[event stringForKey:@"name"]];
    [cmpCityField setStringValue:[event stringForKey:@"city"]];
    [cmpLatField setStringValue:vhfStringWithFloat([event floatForKey:@"lat"])];
    [cmpLonField setStringValue:vhfStringWithFloat([event floatForKey:@"lon"])];
    if ( (string = [event stringForKey:@"date"]) )
    {
        if ( isdigit([string characterAtIndex:[string length]-1]) )
            date = [NSCalendarDate dateWithString:string calendarFormat:@"%Y-%m-%d %H:%M %z"];
        else
            date = [NSCalendarDate dateWithString:string calendarFormat:@"%Y-%m-%d %H:%M %Z"];
    }
    if (!date)
        date = [NSCalendarDate date];
    [cmpDateField setStringValue:[date descriptionWithCalendarFormat:calendarFormat]];
    [cmpZoneField setStringValue:[[date descriptionWithCalendarFormat:@"%z"] substringToIndex:3]];
    [self set:self];
}

- (NSCalendarDate*)compositeDate
{   NSString        *dateStr = [cmpDateField stringValue], *tzStr = [cmpZoneField stringValue];
    NSString		*dateFormat = astroDateFormatForString([cmpDateField stringValue]);
    NSString		*calendarFormat = [dateFormat stringByAppendingString:@" %H:%M %z"];
    NSCalendarDate	*date;

    date = [NSCalendarDate dateWithString:[NSString stringWithFormat:@"%@ %@00", dateStr, tzStr]
                           calendarFormat:calendarFormat];
    if (!date)
        date = [NSCalendarDate date];
    return date;
}

- (NSDictionary*)compositeData
{   NSMutableDictionary	*dict = [NSMutableDictionary dictionary];

    //if (![self composite])
    //    return nil;
    [dict addEntriesFromDictionary:compositeEvent];
    [dict setObject:(([self composite]) ? @"1" : @"0") forKey:@"doCalculate"];
    [dict setObject:[cmpCityField stringValue] forKey:@"city"];
    [dict setObject:[cmpLatField stringValue]  forKey:@"lat"];
    [dict setObject:[cmpLonField stringValue]  forKey:@"lon"];
    [dict setObject:[cmpNameField stringValue] forKey:@"name"];
    [dict setObject:[self compositeDate]       forKey:@"date"];
    return dict;
}


/*
 * notifications
 */

/* notification send from EventManager (AstroUpdateEvent)
 * this can be send by other objects to calculate a chart
 * modified: 2005-08-06
 */
- (void)updateEvent:(NSNotification*)notification
{   NSDictionary    *event = [notification object];
    BOOL            setComposite = ([[notification userInfo] intForKey:@"composite"]) ? YES : NO;

    if ([event isKindOfClass:[NSDictionary class]])
    {   int	oldState = [compositeSwitch state];

        if (setComposite && oldState == 0)
            [compositeSwitch setState:1];	// FIXME: is there a better way to calculate composite ?
        if ([compositeSwitch state] || setComposite)
            [self setCompositeEvent:event]; // calculate composite chart
        else
            [self setEvent:event];          // calculate main chart
        if (setComposite && oldState == 0)
            [compositeSwitch setState:oldState];
    }
    else
        NSLog(@"APChart, notification send from object %@, shoult be NSDictionary!", event);
}


/* delegate methods
 */
- (void)windowDidBecomeKey:(NSNotification*)notification
{
    [self update:self];
}

/* created:  1993-07-22
 * modified: 2003-06-22
 */
- (void)dealloc
{
    [compositeEvent release];
    [super dealloc];
}

@end
