/*
 * AstroPanel.m
 *
 * Copyright 1999-2010 by vhf interservice GmbH
 * Author:   Georg Fleischmann
 *
 * Created:  1995-08-10
 * Modified: 2010-01-16 (move/resize in grid)
 *           2008-07-23 (awakeFromNib instead of init)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the vhf Public License as
 * published by vhf interservice GmbH. Among other things, the
 * License requires that the copyright notices and this notice
 * be preserved on all copies.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the vhf Public License for more details.
 *
 * You should have received a copy of the vhf Public License along
 * with this program; see the file LICENSE. If not, write to vhf.
 *
 * vhf interservice GmbH, Im Marxle 3, 72119 Altingen, Germany
 * eMail: info@vhf.de
 * http://www.vhf.de
 */

#include <VHFShared/vhfCommonFunctions.h>
#include <VHFShared/VHFSystemAdditions.h>
#include "../../Cenon/PreferencesMacros.h"
#include "../AstroPrincipal.h"
#include "../astroMessages.h"
#include "AstroPanel.h"
#include "APBasicLevel.h"
#if ! USE_TABVIEW	// OpenStep
#    include "../Cenon/UITabActionCell.subproj/UITabActionCell.h"
#endif

@implementation AstroPanel

#if USE_TABVIEW	// GNUstep, MAC-OS X
- (BOOL)canBecomeKeyWindow	{ return YES; }

- awakeFromNib
{   static BOOL             awake = NO;
    NSTabViewItem           *item;
    NSNotificationCenter    *notificationCenter = [NSNotificationCenter defaultCenter];

    if ( awake )    // we are already awake
        return self;
    awake = YES;
    //[super init];

    [tabView setDelegate:self];
    levelCnt = 0;

    /* add observer to add tabs */
    [notificationCenter addObserver:self
                           selector:@selector(addTabNotification:)
                               name:AstroPanelAddTabNotification
                             object:nil];

    item = [[NSTabViewItem alloc] initWithIdentifier:@"Chart"];
    [item setLabel:TABCHART_STRING];
    [item setView:[[self windowAt:AP_CHART] contentView]];
    [tabView addTabViewItem:item];
    windows[levelCnt] = activeWindow = chartWindow;
    levelCnt++;

    item = [[NSTabViewItem alloc] initWithIdentifier:@"Map"];
    [item setLabel:TABMAP_STRING];
    [item setView:[[self windowAt:AP_MAP] contentView]];
    [tabView addTabViewItem:item];
    windows[levelCnt] = mapWindow;
    levelCnt++;

    item = [[NSTabViewItem alloc] initWithIdentifier:@"Search"];
    [item setLabel:TABSEARCH_STRING];
    [item setView:[[self windowAt:AP_SEARCH] contentView]];
    [tabView addTabViewItem:item];
    windows[levelCnt] = searchWindow;
    levelCnt++;

    /*item = [[NSTabViewItem alloc] initWithIdentifier:@"FlowChart"];
    [item setLabel:TABFLOWCHART_STRING];
    [item setView:[[self windowAt:AP_FLOWCHART] contentView]];
    [tabView addTabViewItem:item];
    windows[levelCnt] = flowWindow;
    levelCnt++;*/

    return self;
}

/* delegate: update view of tabViewItem
 */
- (void)tabView:(NSTabView*)tabV willSelectTabViewItem:(NSTabViewItem*)tabViewItem
{   int	level = [tabV indexOfTabViewItem:tabViewItem];

    /* click on tab -> set parent view
     */
    if ( !setSubLevel )
    {
        if ( level < levelCnt && activeWindow != windows[level] )
        {
            [activeWindow displayWillEnd];
            [tabViewItem setView:[[self windowAt:level] contentView]];
            activeWindow = windows[level];
        }
        [activeWindow update:self];
    }
}

/* set view of tabView (sub tabs)
 */
- (void)setLevelAt:(int)level
{   NSTabViewItem   *item;
    NSView          *newView = [[self windowAt:level] contentView];

    [activeWindow displayWillEnd];
    setSubLevel = YES;
    if ( level < levelCnt )
    {
        item = [tabView tabViewItemAtIndex:level];
        if ( [item view] != newView )
        {   [[item view] retain];
#ifdef GNUSTEP_BASE_VERSION
            [tabView selectTabViewItem:nil];	// GNUstep: old views are not removed
#endif
            [item setView:newView];
            [tabView selectTabViewItemAtIndex:level];
        }
    }
    setSubLevel = NO;
    activeWindow = [self windowAt:level];
    [activeWindow update:self];
}

#else	// OpenStep

- awakeFromNib
{   static BOOL             awake = NO;
    NSNotificationCenter	*notificationCenter = [NSNotificationCenter defaultCenter];

    if ( awake )    // we are already awake
        return self;
    awake = YES;
    //[super init];

    /* add observer to add tabs */
    [notificationCenter addObserver:self
                           selector:@selector(addTabNotification:)
                               name:AstroPanelAddTabNotification
                             object:nil];

    levelCnt = [levelRadio numberOfColumns];
    [self clickLevel:self];

    return self;
}

- (void)clickLevel:sender
{
   return [self setLevelAt:[levelRadio selectedColumn]];
}

/* attention to the -init if it is not subclassed you will lose the window!
 */
- (void)setLevelAt:(int)level
{
    [activeWindow displayWillEnd];
    if (level < levelCnt)
        [levelRadio selectCellAtRow:0 column:level];

    if (level < levelCnt )
    {
        [self windowAt:level];
        [self setLevelView:[windows[level] contentView]];
        activeWindow = windows[level];
    }
    else
    {
        [self setLevelView:nil];
        activeWindow = self;
        return;
    }

    [activeWindow update:self];
    [self orderFront:self];
}

- (void)setLevelView:theView
{
    [levView setContentView:[theView retain]];
    [theView setAutoresizingMask:NSViewWidthSizable|NSViewHeightSizable];

    [self display];
    [self flushWindow]; 
}

#endif	// end OpenStep

- (void)update:sender
{   int	i;

    for (i=0; i < levelCnt; i++)
        [windows[i] update:sender];
}

- (id)activeWindow
{
    return activeWindow;
}

- windowAt:(int)level
{
    switch (level)
    {
        case AP_CHART:
            if ( !chartWindow )
            {	if (![NSBundle loadModelNamed:@"APChart" owner:self])
                    NSLog(@"Cannot load APChart interface");
                windows[level] = chartWindow;
                [[chartWindow init] setWindow:self];
                [[chartWindow contentView] retain];	// GNUstep
            }
            return chartWindow;
        case AP_MAP:
            if ( !mapWindow )
            {	if (![NSBundle loadModelNamed:@"APMap" owner:self])
                    NSLog(@"Cannot load APMap interface");
                windows[level] = mapWindow;
                [[mapWindow init] setWindow:self];
                [[mapWindow contentView] retain];	// GNUstep
            }
            return mapWindow;
        case AP_SEARCH:
            if ( !searchWindow )
            {	if (![NSBundle loadModelNamed:@"APSearch" owner:self])
                    NSLog(@"Cannot load APSearch interface");
                windows[level] = searchWindow;
                [[searchWindow init] setWindow:self];
                [[searchWindow contentView] retain];	// GNUstep
            }
            return searchWindow;
        /*case AP_FLOWCHART:
            if ( !flowWindow )
            {	if (![NSBundle loadModelNamed:@"APFlowChart" owner:self])
                    NSLog(@"Cannot load APFlowChart interface");
                windows[level] = flowWindow;
                break;
            }
            return windows[level];*/
        default:
            return (level < levelCnt) ? windows[level] : nil;
    }
    [[windows[level] init] setWindow:self];
    [[windows[level] contentView] retain];	// GNUstep
    return windows[level];
}

/*
 * we overwrite this to catch a keyDown before NSMatrix etc. does
 */
- (void)sendEvent:(NSEvent *)event
{
    if ( event && [event type] == NSKeyDown )
    {
        /* we remove the modifiers when we pass down the event,
         * so we can use control for the speed control of DPControl
         */
        event = [NSEvent keyEventWithType:[event type] location:[event locationInWindow] modifierFlags:0
                                timestamp:[event timestamp] windowNumber:[event windowNumber]
                                  context:[event context] characters:[event charactersIgnoringModifiers]
              charactersIgnoringModifiers:[event charactersIgnoringModifiers] isARepeat:[event isARepeat]
                                  keyCode:[event keyCode]];
        if ( [[event characters] isEqual:@"\t"] )
            tabEvent = YES;
    }
    [super sendEvent:event];
}

/* send for NSMatrix and NSTextField
 */
- (void)controlTextDidEndEditing:(NSNotification *)aNotification
{   id	matrix = [aNotification object];

    if ( tabEvent )
    {
        if ( [[matrix selectedCell] action] )
            [[[matrix selectedCell] target] performSelector:[[matrix selectedCell] action]
                                                 withObject:[matrix selectedCell]];
        else if ( [matrix action] )
            [[matrix target] performSelector:[matrix action] withObject:matrix];
        tabEvent = NO;
    }
}


/* delegate methods
 */
- (void)windowDidBecomeKey:(NSNotification*)notification
{
    [activeWindow update:self];
}

/* allow resizing the window in grid
 * created: 2010-01-16
 */
- (NSSize)windowWillResize:(NSWindow*)sender toSize:(NSSize)newSize
{   int gridSize = Prefs_WindowGrid;

    if ( gridSize ) // grid size
    {
        newSize.width  = floor((newSize.width +gridSize/2) / gridSize) * gridSize;
        newSize.height = floor((newSize.height+gridSize/2) / gridSize) * gridSize;
    }
    return newSize;
}
/* allow moving of window in grid
 * created: 2010-01-16
 */
- (NSRect)constrainFrameRect:(NSRect)frameRect toScreen:(NSScreen *)screen
{   int gridSize = Prefs_WindowGrid;

    frameRect = [super constrainFrameRect:frameRect toScreen:screen];
    if ( gridSize )
    {   frameRect.origin.x = floor((frameRect.origin.x+gridSize/2) / gridSize) * gridSize;
        frameRect.origin.y = floor((frameRect.origin.y+gridSize/2) / gridSize) * gridSize;
    }
    return frameRect;
}

/*
 * Notifications
 */

- (void)addTabNotification:(NSNotification*)notification
{   APBasicLevel    *panel = [notification object];
    int             level = levelCnt;

    windows[level] = [panel retain];
    levelCnt ++;
    [panel init];	// ???
    [panel setWindow:self];
    [[panel contentView] retain];	// GNUstep

#if USE_TABVIEW	// GNUstep, MAC-OS X
    {   NSTabViewItem   *item;

        item = [[NSTabViewItem alloc] initWithIdentifier:[panel name]];
        [item setLabel:[panel name]];
        [item setView:[panel contentView]];
        [tabView addTabViewItem:item];
    }
#else
    {   UITabActionCell *item;
        NSSize          size;

        [levelRadio addColumn];
        item = [levelRadio cellAtRow:0 column:level];
        [item setStringValue:[panel name]];
        size = [levelRadio bounds].size;
        [levelRadio setCellSize:NSMakeSize(size.width/levelCnt, size.height)];
        [self display];
        [self flushWindow];
    }
#endif
}

@end
