/* EventManager.m
 * controller class and files owner of the EventPanel
 *
 * Copyright (C) 2003-2010 by vhf interservice GmbH
 * Author:   Georg Fleischmann
 *
 * created:  2003-06-22
 * modified: 2010-04-26 (-renameFolder:, -removeFolder: fixed)
 *           2009-02-28 (-add, -modify, -remove: test for nil events)
 *           2006-01-15 (date formats)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the vhf Public License as
 * published by vhf interservice GmbH. Among other things, the
 * License requires that the copyright notices and this notice
 * be preserved on all copies.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the vhf Public License for more details.
 *
 * You should have received a copy of the vhf Public License along
 * with this program; see the file LICENSE. If not, write to vhf.
 *
 * vhf interservice GmbH, Im Marxle 3, 72119 Altingen, Germany
 * eMail: info@vhf.de
 * http://www.vhf.de
 */

#include "EventManager.h"
#include "../Cenon/VHFShared/VHFDictionaryAdditions.h"
#include "../Cenon/App.h"
#include "../Cenon/functions.h"
#include "../Cenon/messages.h"
#include "../Cenon/PreferencesMacros.h"
#include "astroLocations.h"
#include "astroMessages.h"
#include "AstroPrincipal.h"
#include "Astro.bproj/AstroController.h"
#include "AstroDate.h"	// date checks etc.
#include "AAFImport.h"
#include "CityManager.h"
#include "TimeZoneManager.h"

@interface EventManager(PrivateMethods)
- (void)loadEvents;
- (void)saveEvents;
@end

@implementation EventManager

/* created: 2003-06-22
 * sort event dictionaries by name
 */
static NSInteger sortEventDictionary(id event1, id event2, void *context)
{
    return [(NSString*)[event1 objectForKey:@"name"] compare:[event2 objectForKey:@"name"]];
}

static NSString *typeFromIndex(int index)
{
    switch (index)
    {
        case TYPEPOPUP_FEMALE:  return @"f";
        case TYPEPOPUP_MALE:    return @"m";
        case TYPEPOPUP_ORG:     return @"o";
        case TYPEPOPUP_COUNTRY: return @"c";
        case TYPEPOPUP_EVENT:   return @"e";
    }
    return nil;
}
static int indexFromType(NSString *typeString)
{
    switch ([typeString characterAtIndex:0])
    {
        case 'f': return TYPEPOPUP_FEMALE;
        case 'm': return TYPEPOPUP_MALE;
        case 'o': return TYPEPOPUP_ORG;
        case 'c': return TYPEPOPUP_COUNTRY;
        case 'e': return TYPEPOPUP_EVENT;
    }
    return TYPEPOPUP_UNKNOWN;
}

/*NSTimeZone *timeZoneWithString(NSString *string)
{   NSTimeZone	*tz;

    if ()
        tz = [NSTimeZone timeZoneWithName:string];
    return tz;
}*/

/* add tool menu item to main menu
 *
 * We add the menu item on creation
 * If we are to be displayed we load the panel itself
 */
- (id)init
{   NSNotificationCenter	*notificationCenter = [NSNotificationCenter defaultCenter];
/*#if !defined(GNUSTEP_BASE_VERSION) && !defined(__APPLE__)	// OpenStep 4.2
    NSMenu	*toolMenu = [[[NSApp mainMenu] itemWithTag:MENU_TOOLS] target];
#else
    NSMenu	*toolMenu = [[[NSApp mainMenu] itemAtIndex:MENU_TOOLS-1] submenu];
#endif*/

    [super init];
    /*[toolMenu addItemWithTitle:@"Astro Event-Manager ..."
                        action:@selector(showPanel:)
                 keyEquivalent:@""];
    [[[toolMenu itemArray] objectAtIndex:[[toolMenu itemArray] count]-1] setTarget:self];*/

    /* add notification to set city */
    [notificationCenter addObserver:self
                           selector:@selector(updateCity:)
                               name:AstroUpdateCity
                             object:nil];

    /* add notification to set time zone */
    [notificationCenter addObserver:self
                           selector:@selector(updateTimeZone:)
                               name:AstroTimeZoneChanged
                             object:nil];

    /* read only events */
    extEventDict = [[NSMutableDictionary dictionary] retain];

    return self;
}

/* load interface file and display panel
 */
- (void)showPanel:sender
{
    if (!eventPanel)
    {   NSBundle	*bundle = [NSBundle bundleForClass:[EventManager class]];

        /* load event panel */
        if ( ![bundle loadNibFile:@"EventPanel"
                externalNameTable:[NSDictionary dictionaryWithObject:self forKey:@"NSOwner"]
                         withZone:[self zone]] )
            NSLog(@"Cannot load EventPanel interface file");
        [eventPanel setFrameUsingName:@"EventPanel"];
        [eventPanel setFrameAutosaveName:@"EventPanel"];

        [self loadEvents];
        [eventBrowser loadColumnZero];
        [eventPanel setDelegate:self];
    }
    [eventPanel makeKeyAndOrderFront:sender];
}


/*
 * managing event folders
 */

- (void)addFolder:sender;
{   NSString	*name = [folderField stringValue];

    if (![folders containsObject:name])
    {   [folders addObject:name];
        [folders sortUsingSelector:@selector(compare:)];
        [eventDict setObject:[NSMutableArray array] forKey:name];
        [eventBrowser loadColumnZero];
        [eventBrowser selectRow:[folders count]-1 inColumn:0];
        [self saveEvents];
    }
    else
        NSLog(@"EventPanel, Add Folder: Name '%@' already in use!", name);
}
- (void)renameFolder:sender;
{   int         folderIx = [eventBrowser selectedRowInColumn:0];
    NSString    *name = [folderField stringValue];

    if (name)
    {   NSArray     *events;
        NSString    *oldFolderName = [folders objectAtIndex:folderIx];

        events = [eventDict objectForKey:oldFolderName];
        [eventDict setObject:events forKey:name];
        [eventDict removeObjectForKey:oldFolderName];

        [folders replaceObjectAtIndex:folderIx withObject:name];
        [eventBrowser loadColumnZero];
        [eventBrowser selectRow:folderIx inColumn:0];
        [self saveEvents];
    }
    else
        NSLog(@"EventPanel, Rename Folder: Name == nil");
}
- (void)removeFolder:sender;
{   int	folderIx = [eventBrowser selectedRowInColumn:0];

    if (! NSRunAlertPanel(@"", REALLYDELETEFOLDER_STRING, DELETE_STRING, CANCEL_STRING, nil,
                               [folders objectAtIndex:folderIx]) == NSAlertDefaultReturn)
        return;

    [eventDict removeObjectForKey:[folders objectAtIndex:folderIx]];
    [folders removeObjectAtIndex:folderIx];
    [eventBrowser loadColumnZero];
    [eventBrowser selectRow:(folderIx >= 1) ? folderIx-1 : 0 inColumn:0];
    [self clear:self];
    [self saveEvents];
}


/*
 * managing events
 */

/* ask CityPanel for the location
 */
- (void)getLocation:sender
{
    [[Astro_Principal cityManager] showPanel:self];
}

/* ask TimeZonePanel for the location
 */
- (void)getTimeZone:sender
{
    [[Astro_Principal timeZoneManager] showPanel:self];
}

/* add new entry
 * modified: 2009-02-28
 */
- (void)add:sender
{   NSMutableDictionary	*event = [NSMutableDictionary dictionary];
    int			folderIx = [eventBrowser selectedRowInColumn:0], eventIx, i;
    NSMutableArray	*events;
    NSString		*dateFormat = astroDateFormatForString([dateField stringValue]);
    NSString		*dateFormat_Z = [dateFormat stringByAppendingString:@" %H:%M %Z"];
    NSString		*dateFormat_z = [dateFormat stringByAppendingString:@" %H:%M %z"];
    NSString		*string;

    if (folderIx < 0)
        return;
    events = [eventDict objectForKey:[folders objectAtIndex:folderIx]];
    if (!events)
        return;

    /* make events list and entries mutable */
    if (![events isKindOfClass:[NSMutableArray class]])
    {
        events = [[events mutableCopy] autorelease];
        [eventDict setObject:events forKey:[folders objectAtIndex:folderIx]];

        for (i=0; i<(int)[events count]; i++)
        {
            if (![events isKindOfClass:[NSMutableDictionary class]])
            {   NSMutableDictionary	*e = [[[events objectAtIndex:i] mutableCopy] autorelease];
                [events replaceObjectAtIndex:i withObject:e];
            }
        }
    }

    if (![eventField stringValue])	// empty title
        return;

    /* name already exists -> clear fields (?) */
    for (i=0; i<[events count]; i++)
    {   NSString	*name = [[events objectAtIndex:i] objectForKey:@"name"];

        if ( [[eventField stringValue] isEqual:name] )
        {
            [self clear:self];
            return;
        }
    }

    /* add data fields */
    [event setObject:[eventField stringValue] forKey:@"name"];
    if ( dateFormat && [[timeField stringValue] length] == 5)
    {   NSString	*tzStr = [tzField stringValue];
        NSString	*string = [NSString stringWithFormat:@"%@ %@ %@",
                                            [dateField stringValue], [timeField stringValue], tzStr];
        NSCalendarDate	*date;

        if ([tzStr hasPrefix:@"+"] || [tzStr hasPrefix:@"-"])	// + = E, - = W
            date = [NSCalendarDate dateWithString:string calendarFormat:dateFormat_z];
        else
            date = [NSCalendarDate dateWithString:string calendarFormat:dateFormat_Z];

        if (!date)
        {   NSRunAlertPanel(@"", WRONGDATEFORMAT_STRING, OK_STRING, nil, nil, astroErrorDate());
            return;
        }

        if ([tzStr hasPrefix:@"+"] || [tzStr hasPrefix:@"-"])	// + = E, - = W
            string = [date descriptionWithCalendarFormat:@"%Y-%m-%d %H:%M %z"];
        else
            string = [date descriptionWithCalendarFormat:@"%Y-%m-%d %H:%M %Z"];
        [event setObject:string forKey:@"date"];
    }
    else
    {   NSRunAlertPanel(@"", WRONGDATEFORMAT_STRING, OK_STRING, nil, nil, astroErrorDate());
        return;
    }
    if ((string = typeFromIndex([typePopup indexOfSelectedItem])))
        [event setObject:string forKey:@"type"];
    [event setObject:[countryField stringValue] forKey:@"country"];
    [event setObject:[cityField stringValue] forKey:@"city"];
    [event setObject:[latField stringValue] forKey:@"lat"];
    [event setObject:[lonField stringValue] forKey:@"lon"];
    [event setObject:[[[textView string] copy] autorelease] forKey:@"text"];
    //if ( ![events containsObject:event] )	// is name already in evetn list ?
    [events addObject:event];

    [events sortUsingFunction:sortEventDictionary context:nil];
    eventIx = [events indexOfObject:event];
    [eventBrowser reloadColumn:1];
    [eventBrowser selectRow:eventIx inColumn:1];

    [self saveEvents];
}
- (void)clear:sender
{
    [typePopup selectItemAtIndex:TYPEPOPUP_UNKNOWN];
    [eventField setStringValue:@""];
    [timeField setStringValue:@"00:00"];
    [tzField setStringValue:@"+0000"];
    [countryField setStringValue:@""];
    [cityField    setStringValue:@""];
    [latField     setStringValue:@""];
    [lonField     setStringValue:@""];
    [textView     setString:@""];
    [eventBrowser selectRow:[eventBrowser selectedRowInColumn:0] inColumn:0];
}
- (void)modify:sender
{   int			folderIx = [eventBrowser selectedRowInColumn:0];
    int			eventIx  = [eventBrowser selectedRowInColumn:1], i;
    NSMutableArray	*events;
    NSMutableDictionary	*event;
    NSString		*dateFormat = astroDateFormatForString([dateField stringValue]);
    NSString		*dateFormat_Z = [dateFormat stringByAppendingString:@" %H:%M %Z"];
    NSString		*dateFormat_z = [dateFormat stringByAppendingString:@" %H:%M %z"];
    NSString		*string;

    if (eventIx < 0)
        return;
    events = [eventDict objectForKey:[folders objectAtIndex:folderIx]];
    if (!events)
        return;

    /* make event list and entries mutable */
    if ( ![events isKindOfClass:[NSMutableArray class]] )
    {
        events = [[events mutableCopy] autorelease];
        [eventDict setObject:events forKey:[folders objectAtIndex:folderIx]];

        for (i=0; i<(int)[events count]; i++)
        {
            if (![events isKindOfClass:[NSMutableDictionary class]])
            {   NSMutableDictionary	*e = [[[events objectAtIndex:i] mutableCopy] autorelease];
                [events replaceObjectAtIndex:i withObject:e];
            }
        }
    }

    event = [events objectAtIndex:eventIx];
    if (! NSRunAlertPanel(@"", REALLYMODIFY_STRING, MODIFY_STRING, CANCEL_STRING, nil,
                               [event objectForKey:@"name"]) == NSAlertDefaultReturn)
        return;

    /* update values */
    if ( ![[eventField stringValue] length] )
        return;
    if ([events containsObject:[eventField stringValue]])
    {   NSLog(@"Modify: Event already in database!", [eventField stringValue]);
        return;
    }

    if ( dateFormat && [[timeField stringValue] length] == 5 )
    {   NSString	*tzStr = [tzField stringValue];
        NSString	*string = [NSString stringWithFormat:@"%@ %@ %@",
                                            [dateField stringValue], [timeField stringValue], tzStr];
        NSCalendarDate	*date;

        if ([tzStr hasPrefix:@"+"] || [tzStr hasPrefix:@"-"])	// + = E, - = W
            date = [NSCalendarDate dateWithString:string calendarFormat:dateFormat_z];
        else
            date = [NSCalendarDate dateWithString:string calendarFormat:dateFormat_Z];

        if (!date)
        {   NSRunAlertPanel(@"", WRONGDATEFORMAT_STRING, OK_STRING, nil, nil, astroErrorDate());
            return;
        }

        if ([tzStr hasPrefix:@"+"] || [tzStr hasPrefix:@"-"])	// + = E, - = W
            string = [date descriptionWithCalendarFormat:@"%Y-%m-%d %H:%M %z"];
        else
            string = [date descriptionWithCalendarFormat:@"%Y-%m-%d %H:%M %Z"];
        [event setObject:string forKey:@"date"];
    }
    else if (![[dateField stringValue] length])	// this uses the current date
        [event removeObjectForKey:@"date"];
    else
    {   NSRunAlertPanel(@"", WRONGDATEFORMAT_STRING, OK_STRING, nil, nil, astroErrorDate());
        return;
    }

    [event setObject:[eventField stringValue] forKey:@"name"];
    if ((string = typeFromIndex([typePopup indexOfSelectedItem])))
        [event setObject:string forKey:@"type"];
    [event setObject:[countryField stringValue] forKey:@"country"];
    [event setObject:[cityField stringValue] forKey:@"city"];
    [event setObject:[latField stringValue] forKey:@"lat"];
    [event setObject:[lonField stringValue] forKey:@"lon"];
    if (![[textView string] isEqual:[event objectForKey:@"text"]])
    {
        if (![[textView string] length])
            [event removeObjectForKey:@"text"];
        else
            [event setObject:[[[textView string] copy] autorelease] forKey:@"text"];
    }

    [eventBrowser reloadColumn:1];
    [eventBrowser selectRow:eventIx inColumn:1];
    [self saveEvents];
}
- (void)remove:sender
{   int             folderIx = [eventBrowser selectedRowInColumn:0];
    int             eventIx  = [eventBrowser selectedRowInColumn:1], i;
    NSMutableArray  *events;
    NSDictionary    *event;

    if (eventIx < 0)
        return;
    events = [eventDict objectForKey:[folders objectAtIndex:folderIx]];
    if (!events)
        return;
    event = [events objectAtIndex:eventIx];
    if (! NSRunAlertPanel(@"", REALLYDELETEEVENT_STRING, DELETE_STRING, CANCEL_STRING, nil,
                               [event objectForKey:@"name"]) == NSAlertDefaultReturn)
        return;

    /* make event list and entries mutable */
    if (![events isKindOfClass:[NSMutableArray class]])
    {
        events = [[events mutableCopy] autorelease];
        [eventDict setObject:events forKey:[folders objectAtIndex:folderIx]];

        for (i=0; i<(int)[events count]; i++)
        {
            if (![events isKindOfClass:[NSMutableDictionary class]])
            {   NSMutableDictionary	*e = [[[events objectAtIndex:i] mutableCopy] autorelease];
                [events replaceObjectAtIndex:i withObject:e];
            }
        }
    }

    [eventField setStringValue:@""];
    [timeField setStringValue:@"00:00"];
    [tzField setStringValue:@"+0000"];
    [countryField setStringValue:@""];
    [cityField setStringValue:@""];
    [latField setStringValue:@""];
    [lonField setStringValue:@""];
    [textView setString:@""];

    [events removeObjectAtIndex:eventIx];
    //[eventBrowser selectRow:(eventIx >= 1) ? eventIx-1 : 0 inColumn:1];
    [eventBrowser reloadColumn:1];

    [self saveEvents];
}

/* returns the current event list or nil if none
 * created: 2005-08-31
 */
- (NSArray*)selectedEventList
{   int		folderIx = [eventBrowser selectedRowInColumn:0];
    int		eventIx  = [eventBrowser selectedRowInColumn:1];
    NSArray	*events;

    if (folderIx < 0)
        return nil;

    events = [eventDict objectForKey:[folders objectAtIndex:folderIx]];
    if (!events)
        events = [extEventDict objectForKey:[folders objectAtIndex:folderIx]];
    if ([events isKindOfClass:[NSDictionary class]])	// sub-folders
    {   NSDictionary	*subEvents;
        int		subFolderIx = eventIx;

        //eventIx = [eventBrowser selectedRowInColumn:2];
        //if (eventIx < 0)
        //    return nil;
        subEvents = [extEventDict objectForKey:[folders objectAtIndex:folderIx]];		// Met.db
        events = [subEvents objectForKey:[[subEvents allKeys] objectAtIndex:subFolderIx]];	// List of Rain
    }
    return events;
}
/* returns event row for a selected folder
 * created: 2005-08-31
 */
- (int)eventRow
{   int		folderIx = [eventBrowser selectedRowInColumn:0];
    NSArray	*events;

    if (folderIx < 0)
        return -1;
    events = [eventDict objectForKey:[folders objectAtIndex:folderIx]];
    if (!events)
        events = [extEventDict objectForKey:[folders objectAtIndex:folderIx]];
    if ([events isKindOfClass:[NSDictionary class]])	// sub-folders
        return 2;
    return 1;
}

/* fills the event fields from an event dictionary
 * created: 2005-08-31
 */
- (void)setEvent:(NSDictionary*)event
{   NSString	*string;

    if (!event)
    {   [self clear:self];
        return;
    }
    [eventField setStringValue:[event objectForKey:@"name"]];
    if ( (string = [event stringForKey:@"date"]) )
    {   NSCalendarDate	*date;

        if ( isdigit([string characterAtIndex:[string length]-1]) )
        {   date = [NSCalendarDate dateWithString:string calendarFormat:@"%Y-%m-%d %H:%M %z"];
            [tzField setStringValue:[date descriptionWithCalendarFormat:@"%z"]];
        }
        else
        {   date = [NSCalendarDate dateWithString:string calendarFormat:@"%Y-%m-%d %H:%M %Z"];
            [tzField setStringValue:[date descriptionWithCalendarFormat:@"%Z"]];
        }
        [dateField setStringValue:[date descriptionWithCalendarFormat:Prefs_DateFormat]];
        [timeField setStringValue:[date descriptionWithCalendarFormat:@"%H:%M"]];
    }
    else
    {   [dateField setStringValue:@""];
        [timeField setStringValue:@""];
        [tzField setStringValue:@"+0000"];
    }
    [typePopup selectItemAtIndex:indexFromType([event objectForKey:@"type"])];
    [countryField setStringValue:(string=[event objectForKey:@"country"]) ? string : @""];
    [cityField    setStringValue:(string=[event objectForKey:@"city"])    ? string : @""];
    [latField     setStringValue:(string=[event stringForKey:@"lat"])     ? string : @""];
    [lonField     setStringValue:(string=[event stringForKey:@"lon"])     ? string : @""];
    [textView     setString:     (string=[event objectForKey:@"text"])    ? string : @""];
}

/* search for an event (event name only)
 * created: 2005-08-31
 */
- (void)search:sender
{   int		i, eventIx = -1;
    NSString	*string;

    if ([string=[eventField stringValue] length])
    {   NSArray	*events = [self selectedEventList];

        for (i=0; i<[events count]; i++)
        {   NSString	*name = [[events objectAtIndex:i] objectForKey:@"name"];

            if ([name rangeOfString:string].length)
            {   eventIx = i;
                [eventBrowser selectRow:eventIx inColumn:[self eventRow]];
                [self setEvent:[events objectAtIndex:eventIx]];
                break;
            }
        }
    }
}

/* set selection in Chart-Panel etc. via notification
 * modified: 2005-07-25
 */
- (void)set:sender
{   int             folderIx = [eventBrowser selectedRowInColumn:0];
    int             eventIx  = [eventBrowser selectedRowInColumn:1];
    NSArray         *events;
    NSDictionary    *event, *infoDict = nil;

    if (eventIx < 0)
        return;
    events = [eventDict objectForKey:[folders objectAtIndex:folderIx]];
    if (!events)
        events = [extEventDict objectForKey:[folders objectAtIndex:folderIx]];
    if ([events isKindOfClass:[NSDictionary class]])	// sub-folders
    {   NSDictionary    *subEvents;
        int             subFolderIx = eventIx;

        eventIx = [eventBrowser selectedRowInColumn:2];
        if (eventIx < 0)
            return;
        subEvents = [extEventDict objectForKey:[folders objectAtIndex:folderIx]];               // Met.db
        events    = [subEvents objectForKey:[[subEvents allKeys] objectAtIndex:subFolderIx]];   // List of Rain
    }
    event = [events objectAtIndex:eventIx];
    if ([sender isEqual:setCompositeButton])
        infoDict = [NSDictionary dictionaryWithObject:@"1" forKey:@"composite"];
    [[NSNotificationCenter defaultCenter] postNotificationName:AstroUpdateEvent
                                                        object:event userInfo:infoDict];
}


/*
 * internal methods
 */

/* load events from file
 * modified: 2005-07-24
 */
- (void)loadEvents
{   NSString		*path;
    NSArray		*array;
    NSFileManager	*fileManager = [NSFileManager defaultManager];
    int			i, j;

    [eventBrowser setDelegate:self];
    [eventBrowser setTarget:self];
    [eventBrowser setAction:@selector(doClick:)];

    path = vhfPathWithPathComponents(userLibrary(), MODULENAME, @"event.db", nil);
    if ( ![fileManager fileExistsAtPath:path] )
        path = vhfPathWithPathComponents(localLibrary(), MODULENAME, @"event.db", nil);

    [eventDict release];
    [folders release];
    eventDict = [[NSMutableDictionary dictionaryWithContentsOfFile:path] retain];
    folders = [[eventDict allKeys] mutableCopy];

    /* load other archives */
    for (i=0; i<2; i++)
    {
        if (!i)
            path = vhfPathWithPathComponents(localLibrary(), MODULENAME, @"Data", nil);
        else
            path = vhfPathWithPathComponents(userLibrary(), MODULENAME, @"Data", nil);
        array = [fileManager directoryContentsAtPath:path];
        for (j=0; j<(int)[array count]; j++)
        {   NSString	*file     = [array objectAtIndex:j];
            NSString	*filePath = [path stringByAppendingPathComponent:file];

            if ([file hasSuffix:@"aaf"])	// AAF Archive
            {   NSMutableArray	*dataArray;

                dataArray = [AAFImport dictionaryWithAAFFromFile:filePath];
                [folders addObject:file];
                [extEventDict setObject:dataArray forKey:file];
            }
            else if ([file hasSuffix:@"db"])	// Cenon Archive
            {   NSMutableDictionary	*dataDict;

                dataDict = [NSMutableDictionary dictionaryWithContentsOfFile:filePath];
                if (dataDict)
                {
                    [folders addObject:file];
                    [extEventDict setObject:dataDict forKey:file];
                }
            }
        }
    }

    [folders sortUsingSelector:@selector(compare:)];
}
- (void)saveEvents
{   NSString	*path;

    path = vhfPathWithPathComponents(userLibrary(), MODULENAME, @"event.db", nil);
    [eventDict writeToFile:path atomically:YES];
}


/*
 * delegate methods
 */

/* browser delegate method
 * modified: 2005-07-24
 */
- (NSInteger)browser:(NSBrowser*)sender numberOfRowsInColumn:(NSInteger)column
{   int             folderIx, eventIx;
    NSArray         *events;
    NSDictionary	*subEvents;

    switch (column)
    {
        case 0:		// folders
            return [folders count];
        case 1:		// sub-folders or events
            folderIx = [sender selectedRowInColumn:0];
            events = [eventDict objectForKey:[folders objectAtIndex:folderIx]];
            if (!events)
                events = [extEventDict objectForKey:[folders objectAtIndex:folderIx]];
            if ([events isKindOfClass:[NSDictionary class]])	// sub-folders
            {
                [eventBrowser setHasHorizontalScroller:YES];
                return [[(NSDictionary*)events allKeys] count];
            }
            [eventBrowser setHasHorizontalScroller:NO];
            // FIXME: on Apple 10.4, when Scroller is removed, a white bar remains on top of the Browser,
            //        how to remove this? Resizing the Panel corrects the problem
            return [events count];
        case 2:		// events in sub-folders
            folderIx = [sender selectedRowInColumn:0];
            eventIx  = [sender selectedRowInColumn:1];
            subEvents = [extEventDict objectForKey:[folders objectAtIndex:folderIx]];		// Met.db
            events = [subEvents objectForKey:[[subEvents allKeys] objectAtIndex:eventIx]];	// Rain
            return [events count];
    }
    return 0;
}

/* browser delegate method
 * modified: 2005-07-24
 */
-(void)browser:(NSBrowser*)sender willDisplayCell:(id)cell atRow:(NSInteger)row column:(NSInteger)column
{   int             folderIx, eventIx;
    NSArray         *events;
    NSDictionary    *subEvents;

    switch (column)
    {
        case 0:		// folders
            [cell setStringValue:[folders objectAtIndex:row]];
            [cell setLeaf:NO];
            break;
        case 1:		// sub-folders or events
            folderIx = [sender selectedRowInColumn:0];
            events = [eventDict objectForKey:[folders objectAtIndex:folderIx]];
            if (!events)
                events = [extEventDict objectForKey:[folders objectAtIndex:folderIx]];
            if ([events isKindOfClass:[NSDictionary class]])	// sub-folders
            {   NSArray	*keys = [(NSDictionary*)events allKeys];

                [cell setStringValue:[keys objectAtIndex:row]];
                [cell setLeaf:NO];
            }
            else	// event
            {
                [cell setStringValue:[[events objectAtIndex:row] objectForKey:@"name"]];
                [cell setLeaf:YES];
            }
            break;
        case 2:		// events in sub-folders
            folderIx = [sender selectedRowInColumn:0];
            eventIx  = [sender selectedRowInColumn:1];
            subEvents = [extEventDict objectForKey:[folders objectAtIndex:folderIx]];		// Met.db
            events = [subEvents objectForKey:[[subEvents allKeys] objectAtIndex:eventIx]];	// List of Rain
            [cell setStringValue:[[events objectAtIndex:row] objectForKey:@"name"]];		// Rain event
            [cell setLeaf:YES];
    }

    [cell setLoaded:YES];
}

- (void)doClick:sender
{   int             folderIx = [eventBrowser selectedRowInColumn:0];
    int             eventIx  = [eventBrowser selectedRowInColumn:1];
    NSArray         *events;
    NSDictionary    *event;
    NSString        *string;
    BOOL            flag = YES;

    if (eventIx < 0)
    {
        [folderField setStringValue:[folders objectAtIndex:folderIx]];
        //[self clear:self];	// we let the user copy events between folders -> no clear
        if ([extEventDict objectForKey:[folders objectAtIndex:folderIx]])
            flag = NO;
        [removeFolderButton setEnabled:flag];
        [addButton setEnabled:flag];
        [modifyButton setEnabled:flag];
        [removeButton setEnabled:flag];
        return;
    }

    events = [eventDict objectForKey:[folders objectAtIndex:folderIx]];
    if (!events)
        events = [extEventDict objectForKey:[folders objectAtIndex:folderIx]];
    if ([events isKindOfClass:[NSDictionary class]])	// sub-folders
    {   NSDictionary    *subEvents;
        int             subFolderIx = eventIx;

        eventIx = [eventBrowser selectedRowInColumn:2];
        if (eventIx < 0)
            return;
        subEvents = [extEventDict objectForKey:[folders objectAtIndex:folderIx]];		// Met.db
        events = [subEvents objectForKey:[[subEvents allKeys] objectAtIndex:subFolderIx]];	// List of Rain
    }
    event = [events objectAtIndex:eventIx];
    [eventField setStringValue:[event objectForKey:@"name"]];
    if ( (string = [event stringForKey:@"date"]) )
    {   NSCalendarDate	*date;

        if ( isdigit([string characterAtIndex:[string length]-1]) )
        {   date = [NSCalendarDate dateWithString:string calendarFormat:@"%Y-%m-%d %H:%M %z"];
            [tzField setStringValue:[date descriptionWithCalendarFormat:@"%z"]];
        }
        else
        {   date = [NSCalendarDate dateWithString:string calendarFormat:@"%Y-%m-%d %H:%M %Z"];
            [tzField setStringValue:[date descriptionWithCalendarFormat:@"%Z"]];
        }
        [dateField setStringValue:[date descriptionWithCalendarFormat:Prefs_DateFormat]];
        [timeField setStringValue:[date descriptionWithCalendarFormat:@"%H:%M"]];
    }
    else
    {   [dateField setStringValue:@""];
        [timeField setStringValue:@""];
        [tzField setStringValue:@"+0000"];
    }
    [typePopup selectItemAtIndex:indexFromType([event objectForKey:@"type"])];
    [countryField setStringValue:(string=[event objectForKey:@"country"]) ? string : @""];
    [cityField    setStringValue:(string=[event objectForKey:@"city"])    ? string : @""];
    [latField     setStringValue:(string=[event stringForKey:@"lat"])     ? string : @""];
    [lonField     setStringValue:(string=[event stringForKey:@"lon"])     ? string : @""];
    [textView     setString:     (string=[event objectForKey:@"text"])    ? string : @""];
}


/*
 * notifications
 */

/* notification send from CityManager
 * this can be send by other objects to provide a city
 */
- (void)updateCity:(NSNotification*)notification
{   NSDictionary	*city = [notification object];
    NSString		*string;

    if ([city isKindOfClass:[NSDictionary class]])
    {
        [cityField    setStringValue:(string=[city objectForKey:@"city"])    ? string : @""];
        [countryField setStringValue:(string=[city objectForKey:@"country"]) ? string : @""];
        [latField     setStringValue:(string=[city objectForKey:@"lat"])     ? string : @""];
        [lonField     setStringValue:(string=[city objectForKey:@"lon"])     ? string : @""];
    }
    else
        NSLog(@"EventManager, notification send from object %@, shoult be NSDictionary!", city);
}

/* notification send from TimeZoneManager
 * this can be send by other objects to provide a time zone
 */
- (void)updateTimeZone:(NSNotification*)notification
{   NSTimeZone	*tz = [notification object];

    if ([tz isKindOfClass:[NSTimeZone class]])
    {   NSString	*dateStr = [dateField stringValue];
        NSString	*dateFormat = astroDateFormatForString([dateField stringValue]);
        NSString	*timeStr = [timeField stringValue];
        int		y, m, d, H, M;
        NSCalendarDate	*date;

        if (!dateFormat)
        {   NSRunAlertPanel(@"", WRONGDATEFORMAT_STRING, OK_STRING, nil, nil, astroErrorDate());
            return;
        }
        if ([timeStr length] != 5)
        {   NSRunAlertPanel(@"", WRONGDATEFORMAT_STRING, OK_STRING, nil, nil, @"HH:MM");
            return;
        }

        if (![[AstroDate sharedInstance] getYear:&y month:&m day:&d fromDateString:dateStr])
        {   NSRunAlertPanel(@"", WRONGDATEFORMAT_STRING, OK_STRING, nil, nil, astroErrorDate());
            return;
        }

        H = [[timeStr substringToIndex:  2] intValue];
        M = [[timeStr substringFromIndex:3] intValue];

        date = [NSCalendarDate dateWithYear:y month:m day:d hour:H minute:M second:0 timeZone:tz];
        if (!date)
        {   NSRunAlertPanel(@"", WRONGDATEFORMAT_STRING, OK_STRING, nil, nil, astroErrorDate());
            return;
        }

        [tzField setStringValue:(date) ? [date descriptionWithCalendarFormat:@"%z"] : @"+0000"];
    }
    else
        NSLog(@"EventManager, notification send from object %@, shoult be NSTimeZone!", tz);
}

/*
 * Delegate methods
 */

/* allow resizing the window in grid
 * created: 2010-01-16
 */
- (NSSize)windowWillResize:(NSWindow*)sender toSize:(NSSize)newSize
{   int gridSize = Prefs_WindowGrid;

    if ( gridSize ) // grid size
    {
        newSize.width  = floor((newSize.width +gridSize/2) / gridSize) * gridSize;
        newSize.height = floor((newSize.height+gridSize/2) / gridSize) * gridSize;
    }
    return newSize;
}
/* allow moving of window in grid
 * created: 2010-01-16
 */
/*- (NSRect)constrainFrameRect:(NSRect)frameRect toScreen:(NSScreen *)screen
{   int gridSize = Prefs_WindowGrid;

    frameRect = [super constrainFrameRect:frameRect toScreen:screen];
    if ( gridSize )
    {   frameRect.origin.x = floor((frameRect.origin.x+gridSize/2) / gridSize) * gridSize;
        frameRect.origin.y = floor((frameRect.origin.y+gridSize/2) / gridSize) * gridSize;
    }
    return frameRect;
}*/
/*- (void)windowDidMove:(NSNotification *)notification
{   int     gridSize = Prefs_WindowGrid;
    id      window = [notification object];
    NSRect  frameRect = [window frame];

    if ( gridSize )
    {   frameRect.origin.x = floor((frameRect.origin.x+gridSize/2) / gridSize) * gridSize;
        frameRect.origin.y = floor((frameRect.origin.y+gridSize/2) / gridSize) * gridSize;
    }
    [window setFrameOrigin:frameRect.origin];
}*/

- (void)dealloc
{
    [eventDict release];
    [folders release];
    [super dealloc];
}

@end
