/* LayerObject.h
 * Object managing a single layer and its attributes
 *
 * Copyright (C) 1996-2004 by vhf interservice GmbH
 * Author:   Georg Fleischmann
 *
 * Created:  1996-03-07
 * Modified: 2004-10-23
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the vhf Public License as
 * published by vhf interservice GmbH. Among other things, the
 * License requires that the copyright notices and this notice
 * be preserved on all copies.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the vhf Public License for more details.
 *
 * You should have received a copy of the vhf Public License along
 * with this program; see the file LICENSE. If not, write to vhf.
 *
 * vhf interservice GmbH, Im Marxle 3, 72119 Altingen, Germany
 * eMail: info@vhf.de
 * http://www.vhf.de
 */

#ifndef VHF_H_LAYEROBJECT
#define VHF_H_LAYEROBJECT

#include "GraphicObjects.subproj/PerformanceMap.h"

typedef struct
{
    float	step[100];
    int		count;
}OutputSteps;

#define CUT_INSIDE	0
#define CUT_OUTSIDE	1
#define CUT_NOSIDE	2
#define CUT_PICKOUT	3
#define CUT_ISOLATION	4	// calculate isolation of PCB
#define CUT_BLOWUP	5	// calculate blow up of PCB
#define CUT_RUBOUT	6	// calculate rubout of PCB

/* these shouldn't be changed, they are saved to the document !
 */
#define LAYER_STANDARD	0	// standard layer
#define LAYER_PASSIVE	1
#define LAYER_CLIPPING	2
#define LAYER_LEVELING	3
#define LAYER_FITTING	4	// a layer with two marks for flipping the working piece
//#define LAYER_PAGE	5	// page in multi page document

@interface LayerObject: NSObject
{
    NSString		*string;		// the name of the layer
    PerformanceMap	*performanceMap;	// grid for faster access of objects on layer
    NSMutableArray	*list;			// the graphic list
    int			tag;			// tag for layers with special purpose
    NSColor 		*color;			// the color of this layer
    int			state;			// if we have to display this layer
    BOOL		editable;		// whether we are editable
    int			type;			// type of layer: LAYER_STANDARD, LAYER_PASSIVE, ...
    BOOL		likeOtherLayers;	// YES = share parameters with other layers of state YES
    BOOL		useForTile;		// whether layer is used for duplicates
    BOOL		dirty;			// YES when changed

    /* CAM */
    int			toolIndex;		// the index of the tool
    NSString		*toolString;		// the name of the tool
    BOOL		filled;			// whether we are filled
    BOOL		mirrored;		// whether we are mirrored
    int			side;			// whether to cut inside, outside or on the line
    float		flatness;
    float		fillOverlap;		// (0.0 - 1.0) 0 = no overlap
    float		fillDirection;		// (0.0 - 360.0)
    BOOL		removeLoops;		// whether we have to remove garbage (loops) from contour
    BOOL		revertDirection;	// whether we revert the direction (ccw/cw)
    float		dippingDepth;		// the dippingDepth [mm]
    float		approachAngle;		// the approach angle for dipping, 0 - 89 [deg]
    BOOL		stepwise;		// weather stepwise cutting is active
    int			numSteps;		// number of middle steps
    float		step[3];		// first, step width, final step [mm]
    OutputSteps		calcSteps;		// calculated steps [mm]
    float		settle;			// [mm]
    BOOL		settleBefore;		// settlement before last step z
    int			levelingX, levelingY;	// number of test points for leveling */
    BOOL		inlay;

    /* CAM output */
    NSArray		*webList;		// used to reach web list when creating output
    int			tileCount;		// used to reach tile count when creating output
    id			clipObject;		// used to reach clip object when creating output
}

+ layerObject;
+ layerObjectWithFrame:(NSRect)bRect;
- init;
- initWithFrame:(NSRect)bRect;

- (void)createPerformanceMapWithFrame:(NSRect)rect;
- (PerformanceMap*)performanceMap;
- (void)setList:(NSMutableArray*)aList;
- (NSMutableArray*)list;
- (void)insertObject:(id)obj atIndex:(unsigned)ix;
- (void)addObject:(id)obj;
- (void)addObjectWithoutCheck:(id)obj;
- (void)addObjectsFromArray:(NSArray*)array;
- (void)updateObject:(VGraphic*)g;
- (void)removeAllObjects;
- (void)removeObject:(id)obj;
- (void)draw:(NSRect)rect inView:(id)view;

- (BOOL)isFillable;
- (BOOL)hasDip;
- (BOOL)isPassive;
- (BOOL)fastSideSelection;

- (void)setString:(NSString *)aString;
- (NSString*)string;

- (void)setTag:(int)newTag;
- (int)tag;

- (void)setColor:(NSColor *)aColor;
- (NSColor *)color;

- (void)setState:(int)flag;
- (int)state;

- (void)setEditable:(BOOL)flag;
- (BOOL)editable;

- (void)setDirty:(BOOL)flag calculate:(BOOL)cflag;
- (void)setDirty:(BOOL)flag;
- (BOOL)dirty;

- (int)type;
- (void)setType:(int)newType;

- (BOOL)useForTile;
- (void)setUseForTile:(BOOL)flag;
- (void)setTileCount:(int)c;
- (int)tileCount;

- (BOOL)likeOtherLayers;
- (void)setLikeOtherLayers:(BOOL)flag;


/* CAM */
- (void)setToolIndex:(int)aTool;
- (int)toolIndex;
- (void)setToolString:(NSString*)theToolString;
- (NSString*)toolString;

- (void)setFilled:(BOOL)flag;
- (BOOL)filled;

- (void)setMirrored:(BOOL)flag;
- (BOOL)mirrored;

- (void)setInlay:(BOOL)flag;
- (BOOL)inlay;

- (void)setSide:(int)s;
- (int)side;

- (float)flatness;
- (void)setFlatness:(float)v;

- (float)fillOverlap;
- (void)setFillOverlap:(float)v;
- (float)fillDirection;
- (void)setFillDirection:(float)v;

- (void)setDippingDepth:(float)d;
- (float)dippingDepth;

- (void)setApproachAngle:(float)angle;
- (float)approachAngle;

- (void)setStepwise:(BOOL)flag;
- (BOOL)stepwise;
- (void)setFirstStep:(float)step;
- (void)setStepHeight:(float)height;
- (float)stepHeight;
- (void)setNumSteps:(int)n;
- (int)numSteps;
- (void)setFinalStep:(float)step;
- (float)stepWithNum:(int)n;
- (OutputSteps)steps;
- (OutputSteps)stepsForDip:(float)dip;

- (void)setSettle:(float)value;
- (float)settle;
- (void)setSettleBefore:(BOOL)flag;
- (BOOL)settleBefore;

- (BOOL)removeLoops;
- (void)setRemoveLoops:(BOOL)flag;

- (BOOL)revertDirection;
- (void)setRevertDirection:(BOOL)flag;

- (void)setLevelingPointsX:(int)x y:(int)y;
- (int)levelingPointsX;
- (int)levelingPointsY;

/* CAM output, used to reach web list when creating output */
- (void)setWebList:(NSArray*)array;
- (NSArray*)webList;
- (void)setClipObject:obj;
- clipObject;


/* archiving */
- (void)dealloc;

- (id)initWithCoder:(NSCoder *)aDecoder;
- (void)encodeWithCoder:(NSCoder *)aCoder;

@end

#endif // VHF_H_LAYEROBJECT
