/* APChart.m
 * Controls the chart generation
 *
 * Copyright (C) 2001-2004 by vhf interservice GmbH
 * Author:   Georg Fleischmann
 *
 * Created:  2001-05-18
 * Modified: 2004-05-22
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the vhf Public License as
 * published by vhf interservice GmbH. Among other things, the
 * License requires that the copyright notices and this notice
 * be preserved on all copies.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the vhf Public License for more details.
 *
 * You should have received a copy of the vhf Public License along
 * with this program; see the file LICENSE. If not, write to vhf.
 *
 * vhf interservice GmbH, Im Marxle 3, 72119 Altingen, Germany
 * eMail: info@vhf.de
 * http://www.vhf.de
 */

#include <VHFShared/vhfCommonFunctions.h>
#include "../Cenon/App.h"
#include "../Cenon/locations.h"
#include "../Cenon/messages.h"
#include "../Cenon/functions.h"
#include "AstroPanel.h"
#include "APChart.h"
#include "../AstroChart.h"
#include "../astroLocations.h"
#include "../EventManager.h"
#include "../AstroPrincipal.h"

#define EVENTFILE	@"event.data"	// list of events

@interface APChart(PrivateMethods)
@end

@implementation APChart

- init
{   NSNotificationCenter	*notificationCenter = [NSNotificationCenter defaultCenter];

    [self setDelegate:self];

    /* add notification to set event */
    [notificationCenter addObserver:self
                           selector:@selector(updateEvent:)
                               name:AstroUpdateEvent
                             object:nil];

    [self setComposite:self];

    return [super init];
}

/* start calculation of chart
 */
- (void)set:(id)sender
{   NSCalendarDate	*utc;
    float		lat = [latField floatValue], lon = [lonField floatValue];
    Document		*doc = [(App*)NSApp currentDocument];
    DocView		*view = [doc documentView];

    utc = [NSCalendarDate dateWithString:[NSString stringWithFormat:@"%@ UTC", [dateField stringValue]]
                          calendarFormat:@"%d.%m.%Y-%H:%M %Z"];

    if (!astroChart && view && [[doc name] hasPrefix:CHART_PREFIX] )
        astroChart = [[AstroChart astroChartWithView:view] retain];
    else if (astroChart && view && [[doc name] hasPrefix:CHART_PREFIX] )
        [astroChart setView:view];
    [astroChart radix:utc longitude:lon latitude:lat city:[self city]
                title:[nameField stringValue] composite:[self compositeData]];

    /* calculate chart for date */
    [[NSNotificationCenter defaultCenter] postNotificationName:AstroDateSetNotification object:utc];
}

- (id)astroChart
{
    return astroChart;
}

/*
 * chart methods
 */

/* open the database panel
 */
- (void)openDatabase:sender
{
    [[Astro_Principal eventManager] showPanel:self];
}

/* set current time and date
 */
- (void)setNow:sender
{   NSCalendarDate	*date = [NSCalendarDate date];

    [date setTimeZone:[NSTimeZone timeZoneWithName:@"UTC"]];
    if ([compositeSwitch state])
    {
        [cmpDateField setStringValue:[date descriptionWithCalendarFormat:@"%d.%m.%Y-%H:%M"]];
        [cmpDateField display];
    }
    else
    {
        [dateField setStringValue:[date descriptionWithCalendarFormat:@"%d.%m.%Y-%H:%M"]];
        [dateField display];
    }
    [self set:self];
}

- (void)setDate:sender
{   BOOL		control = [(App*)NSApp control];
    int			m = 0;
    NSCalendarDate	*date = [self date];

    if ([sender isKindOfClass:[NSButton class]])	// buttons
    {
        switch ([sender tag])
        {
            case BUTTONLEFT:	m = ((control) ? -60*24 : -60); break;
            case BUTTONRIGHT:	m = ((control) ?  60*24 :  60);
        }
        date = [date dateByAddingYears:0 months:0 days:0 hours:0 minutes:m seconds:0];
        [dateField setStringValue:[date descriptionWithCalendarFormat:@"%d.%m.%Y-%H:%M"]];
        [dateField display];
    }
    [self set:self];
}

- (void)setEvent:(NSDictionary*)event
{   NSString		*string;
    NSCalendarDate	*date;

    [nameField setStringValue:[event stringForKey:@"name"]];
    [cityField setStringValue:[event stringForKey:@"city"]];
    [latField setStringValue:vhfStringWithFloat([event floatForKey:@"lat"])];
    [lonField setStringValue:vhfStringWithFloat([event floatForKey:@"lon"])];
    string = [event stringForKey:@"date"];
    if ( isdigit([string characterAtIndex:[string length]-1]) )
        date = [NSCalendarDate dateWithString:string calendarFormat:@"%Y-%m-%d %H:%M %z"];
    else
        date = [NSCalendarDate dateWithString:string calendarFormat:@"%Y-%m-%d %H:%M %Z"];
    if (!date)
        date = [NSCalendarDate date];
    [date setTimeZone:[NSTimeZone timeZoneWithName:@"UTC"]];
    [dateField setStringValue:[date descriptionWithCalendarFormat:@"%d.%m.%Y-%H:%M"]];
    [self set:self];
}


- (NSString*)title
{
    return [nameField stringValue];
}
- (NSString*)city
{
    return [cityField stringValue];
}
- (float)latitude
{
    return [latField floatValue];
}
- (float)longitude
{
    return [lonField floatValue];
}
- (NSCalendarDate*)date
{
    return [NSCalendarDate dateWithString:[NSString stringWithFormat:@"%@ UTC", [dateField stringValue]]
                           calendarFormat:@"%d.%m.%Y-%H:%M %Z"];
}

- (NSMutableDictionary*)data
{   NSMutableDictionary	*dict = [NSMutableDictionary dictionary];

    [dict setObject:[cityField stringValue] forKey:@"city"];
    [dict setObject:[latField stringValue] forKey:@"lat"];
    [dict setObject:[lonField stringValue] forKey:@"lon"];
    [dict setObject:[nameField stringValue] forKey:@"name"];
    [dict setObject:[self date] forKey:@"date"];
    return dict;
}



/*
 * composite methods
 */

- (void)setComposite:sender
{
    [cmpNameField setEnabled:[self composite]];
    [cmpCityField setEnabled:[self composite]];
    [cmpLatField setEnabled:[self composite]];
    [cmpLonField setEnabled:[self composite]];
    [cmpDateField setEnabled:[self composite]];
    [cmpBCButton setEnabled:[self composite]];

    /* enable/disable composite layer */
    [astroChart setCompositeLayerEnabled:[self composite]];
}

- (BOOL)composite
{
    return [compositeSwitch state];
}

- (void)setCompositeDate:sender
{   BOOL		control = [(App*)NSApp control];
    int			m = 0;
    NSCalendarDate	*date = [self compositeDate];

    if ([sender isKindOfClass:[NSButton class]])	// buttons
    {
        switch ([sender tag])
        {
            case BUTTONLEFT:	m = ((control) ? -60*24 : -60); break;
            case BUTTONRIGHT:	m = ((control) ?  60*24 :  60);
        }
        date = [date dateByAddingYears:0 months:0 days:0 hours:0 minutes:m seconds:0];
        [cmpDateField setStringValue:[date descriptionWithCalendarFormat:@"%d.%m.%Y-%H:%M"]];
        [cmpDateField display];
    }
    [self set:self];	// calculate chart
}

- (void)setCompositeEvent:(NSDictionary*)event
{   NSString		*string;
    NSCalendarDate	*date;

    [compositeEvent release];
    compositeEvent = [event retain];

    [cmpNameField setStringValue:[event stringForKey:@"name"]];
    [cmpCityField setStringValue:[event stringForKey:@"city"]];
    [cmpLatField setStringValue:vhfStringWithFloat([event floatForKey:@"lat"])];
    [cmpLonField setStringValue:vhfStringWithFloat([event floatForKey:@"lon"])];
    string = [event stringForKey:@"date"];
    if ( isdigit([string characterAtIndex:[string length]-1]) )
        date = [NSCalendarDate dateWithString:string calendarFormat:@"%Y-%m-%d %H:%M %z"];
    else
        date = [NSCalendarDate dateWithString:string calendarFormat:@"%Y-%m-%d %H:%M %Z"];
    if (!date)
        date = [NSCalendarDate date];
    [date setTimeZone:[NSTimeZone timeZoneWithName:@"UTC"]];
    [cmpDateField setStringValue:[date descriptionWithCalendarFormat:@"%d.%m.%Y-%H:%M"]];
    [self set:self];
}

- (NSCalendarDate*)compositeDate
{
    return [NSCalendarDate dateWithString:[NSString stringWithFormat:@"%@ UTC", [cmpDateField stringValue]]
                           calendarFormat:@"%d.%m.%Y-%H:%M %Z"];
}

- (NSDictionary*)compositeData
{   NSMutableDictionary	*dict = [NSMutableDictionary dictionary];

    if (![self composite])
        return nil;
    [dict addEntriesFromDictionary:compositeEvent];
    [dict setObject:[cmpCityField stringValue] forKey:@"city"];
    [dict setObject:[cmpLatField stringValue] forKey:@"lat"];
    [dict setObject:[cmpLonField stringValue] forKey:@"lon"];
    [dict setObject:[cmpNameField stringValue] forKey:@"name"];
    [dict setObject:[self compositeDate] forKey:@"date"];
    return dict;
}


/*
 * notifications
 */

/* notification send from EventManager
 * this can be send by other objects to calculate a chart
 */
- (void)updateEvent:(NSNotification*)notification
{   NSDictionary	*event = [notification object];

    if ([event isKindOfClass:[NSDictionary class]])
    {
        if ([compositeSwitch state])
            [self setCompositeEvent:event];
        else
            [self setEvent:event];
    }
    else
        NSLog(@"APChart, notification send from object %@, shoult be NSDictionary!", event);
}


/* created:  1993-07-22
 * modified: 2003-06-22
 */
- (void)dealloc
{
    [compositeEvent release];
    [super dealloc];
}

@end
