/* EphemerisSearch.m
 * search Ephemeris data
 *
 * Copyright (C) 2004 by vhf interservice GmbH
 * Author:   Georg Fleischmann
 *
 * created:  2004-04-12
 * modified: 2004-05-20
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the vhf Public License as
 * published by vhf interservice GmbH. Among other things, the
 * License requires that the copyright notices and this notice
 * be preserved on all copies.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the vhf Public License for more details.
 *
 * You should have received a copy of the vhf Public License along
 * with this program; see the file LICENSE. If not, write to vhf.
 *
 * vhf interservice GmbH, Im Marxle 3, 72119 Altingen, Germany
 * eMail: info@vhf.de
 * http://www.vhf.de
 */

#include <Foundation/Foundation.h>
#include <VHFShared/types.h>
#include "EphemerisSearch.h"
#include "astroCommon.h"

@interface EphemerisSearch(PrivateMethods)
@end

@implementation EphemerisSearch

+ newWithEphemeris:(Ephemeris*)eph
{
    return [[self alloc] initWithEphemeris:eph];
}

- (id)initWithEphemeris:(Ephemeris*)eph
{
    [super init];
    ephemeris = [eph retain];

    return self;
}

/* return a dictionary with arrays containing dates and the events
 *
 * features:
 * {
 *    planets = ("Sun", ...);
 *    radix = { date = NSCalendarDate; lat = latitude; lon = longitude; };
 * }
 *
 * returned:
 * {
 *    planet = 
 *    {
 *      entrySign       = (date1 = sign#; date2 = sign#;);
 *      entryRadixHouse = (date1 = house#; date2 = house#;);
 *      aspects         = (date1 = (planet, devider); date2 = (planet, devider););
 *      aspectsRadix    = (date1 = (planet, devider); date2 = (planet, devider););
 *      mirrors         = (date1 = (planet, devider); date2 = (planet, devider););
 *      mirrorsRadix    = (date1 = (planet, devider); date2 = (planet, devider););
 *    };
 * }
 */
#define HOUSESIZE	15
- (NSDictionary*)searchWithFeatures:(NSDictionary*)features
                           fromDate:(NSCalendarDate*)dateFrom toDate:(NSCalendarDate*)dateTo
                     searchInterval:(int)minutes
{   NSMutableDictionary	*dict = [NSMutableDictionary dictionary];
    NSCalendarDate	*date = dateFrom, *date0;
    NSArray		*planets = [features objectForKey:@"planets"];
    int			planetCnt = [planets count];
    int			p, p1, planetIx[planetCnt], i;
    /* sign entry */
    BOOL		featureEnterSigns = ([features objectForKey:@"entrySign"]) ? YES : NO;
    int			signNo[planetCnt];
    double		lastDeg[planetCnt];
    /* aspects (angles, mirrors) */
    BOOL		featureAspects = ([features objectForKey:@"aspects"])   ? YES : NO;
    BOOL		featureMirrors = ([features objectForKey:@"mirrors"])   ? YES : NO;
    NSArray		*dividers = [features objectForKey:@"dividers"];
    int			divCnt = [dividers count];
    double		target[planetCnt][planetCnt][divCnt],
                        lastAngle [planetCnt][planetCnt][divCnt], closestAngle [planetCnt][planetCnt][divCnt];
    double		lastMirror[planetCnt][planetCnt][divCnt], closestMirror[planetCnt][planetCnt][divCnt];

    if (!(featureEnterSigns || featureAspects || featureMirrors) || ![planets count])	// nothing to do
        return nil;

NSLog(@"Search from: %@ to: %@\nfeatures = %@", dateFrom, dateTo, features);

    /* create dictionaries for planets */
    for (p=0; p<planetCnt; p++)
    {   NSMutableDictionary	*planetDict = [NSMutableDictionary dictionary];
        NSString		*planet = [planets objectAtIndex:p];

        [dict setObject:planetDict forKey:planet];
        planetIx[p] = [ephemeris indexForObjectName:planet];

        if (featureEnterSigns)
        {   [planetDict setObject:[NSMutableDictionary dictionary] forKey:@"entrySign"];
            signNo[p] = -1;
            lastDeg[p] = -1.0;
        }
        if (featureAspects || featureMirrors)
        {
            if (p < planetCnt-1)
                [planetDict setObject:[NSMutableDictionary dictionary] forKey:@"aspects"];
            dividers = [dividers sortedArrayUsingSelector:@selector(compare:)];
            for (p1=0; p1<[planets count]; p1++)
                for (i=0; i<divCnt; i++)
                {
                    target[p][p1][i] = 360.0 / [[dividers objectAtIndex:i] floatValue];
                    lastAngle[p][p1][i]  = closestAngle[p][p1][i]  = 1000.0;
                    lastMirror[p][p1][i] = closestMirror[p][p1][i] = 1000.0;
                }
        }
    }

    /* search through range */
    while ([date compare:dateTo] != NSOrderedDescending)	// date <= dateTo
    {   EphObject	*objects = [ephemeris objectsAtUTC:date];

        /* entry of sign */
        if (featureEnterSigns)
        {
            for (p=0; p<[planets count]; p++)
            {   double  deg = objects[planetIx[p]].lon;
                int	sign = deg / HOUSESIZE;

                if (signNo[p] > -1 && sign != signNo[p])	// sign changed
                {   NSMutableDictionary	*entryDict;
                    double		v = deg, vPrev = lastDeg[p], delta, delta1;
                    int			dSec, targetSign;

//printf("%s: deg=%f (%f) sign = %d (%d)\n", [[date description] cString], deg, lastDeg[p], sign, signNo[p]);
                    /* average exact time */
                    if (Diff(v, vPrev) >= 180.0)	// flip over 0
                    {
                        if (v < vPrev) vPrev -= 360.0;
                        else           v     -= 360.0;
                    }
                    targetSign = (v >= vPrev) ? sign : signNo[p];	// direct/retrograde
                    delta  = Diff(vPrev, v);				// delta interval
                    delta1 = Diff(vPrev, targetSign*HOUSESIZE);		// delta target
                    dSec = minutes*60 * delta1 / delta  - (minutes*60.0);
                    date0 = [date dateByAddingYears:0 months:0 days:0 hours:0 minutes:0 seconds:dSec];

                    //printf("sign changed %s: signNo[%d]=%d sign=%d", [[date0 description] cString], p, signNo[p], sign);
                    entryDict = [[dict objectForKey:[planets objectAtIndex:p]] objectForKey:@"entrySign"];
                    targetSign = ((deg >= lastDeg[p]) ? 1.0 : -1.0) * targetSign;	// retrograde = negative
                    [entryDict setObject:[NSNumber numberWithInt:targetSign*HOUSESIZE] forKey:date0];
                }
                signNo[p] = sign;
                lastDeg[p] = deg;
            }
        }

        /* aspects */
        if (featureAspects || featureMirrors)
        {
            for (p=0; p<planetCnt-1; p++)
            {   double  deg = objects[planetIx[p]].lon;

                for (p1=p+1; p1<planetCnt; p1++)
                {   double  deg1 = objects[planetIx[p1]].lon;

                    if (featureAspects)
                    {   double  a = angle(deg, deg1);

                        for (i=divCnt-1; i>=0; i--)	// use the biggest divider, if more than one divider fit
                        {   double	is  = a - floor(a / target[p][p1][i]) * target[p][p1][i];

                            // FIXME: we should allow approach from both directions: 0 and target !
                            //        closestAngle should be the diff to the node (0 or target)
                            if ( is <= lastAngle[p][p1][i] )
                                closestAngle[p][p1][i] = is;
                            else if (closestAngle[p][p1][i] < 3.0)
                            {   double			v, delta, delta1;
                                int			dSec;
                                NSMutableDictionary	*entryDict;
                                NSArray			*entry;

                                /* get exact date: date += deltaTime * delta1 / delta */
                                v = (is < closestAngle[p][p1][i]) ? is+target[p][p1][i] : is;
                                delta  = Diff(closestAngle[p][p1][i], target[p][p1][i]) + Diff(v, target[p][p1][i]);
                                delta1 = Diff(closestAngle[p][p1][i], target[p][p1][i]);
                                dSec = minutes*60 * delta1 / delta  - (minutes*60.0);
                                date0 = [date dateByAddingYears:0 months:0 days:0 hours:0 minutes:dSec/60 seconds:0];
                                //printf("%s %s  v=%f delta=%f delta1 = %f\n",
                                //[[date descriptionWithCalendarFormat:@"%Y%m%d %H:%M:%S"] cString],
                                //[[date0 descriptionWithCalendarFormat:@"%Y%m%d %H:%M:%S"] cString],
                                //v, delta, delta1);

                                v = floor((a+target[p][p1][i]/2.0)/target[p][p1][i]) * target[p][p1][i];
                                if (v >= 360.0) v = 0.0;
                                //printf("a=%f v=%f target[p][p1][i] = %f\n", a, v, target[p][p1][i]);
                                entry = [NSArray arrayWithObjects:[planets objectAtIndex:p1],
                                        [dividers objectAtIndex:i], [NSNumber numberWithDouble:v], nil];
                                entryDict = [[dict objectForKey:[planets objectAtIndex:p]] objectForKey:@"aspects"];
                                [entryDict setObject:entry forKey:date0];
                                //printf("%f: is=%f last=%f closest=%f\n",
                                //       target[p][p1][i], is, lastAngle[p][p1][i], closestAngle[p][p1][i]);
                                //printf("* * * %s * * *\n", [[entry description] cString]);
                                closestAngle[p][p1][i] = 1000.0;
                            }
                            lastAngle[p][p1][i] = is;
                        }
                    } // end angles
                    if (featureMirrors)
                    {   double  m = mirror(deg, deg1);

                        for (i=divCnt-1; i>=0; i--)	// use the biggest divider, if more than one divider fit
                        {   double	is = m - floor(m / target[p][p1][i]) * target[p][p1][i];

                            if ( Diff(is, target[p][p1][i]) <= Diff(lastMirror[p][p1][i], target[p][p1][i]) )
                                closestMirror[p][p1][i] = is;
                            else if (closestMirror[p][p1][i] < 360.0
                                     && Diff(closestMirror[p][p1][i], target[p][p1][i]) < 3.0)
                            {   double			v, delta, delta1;
                                int			dSec;
                                NSMutableDictionary	*entryDict;
                                NSArray			*entry;

                                /* get exact date: date += deltaTime * delta1 / delta */
                                v = (is < closestMirror[p][p1][i]) ? is+target[p][p1][i] : is;
                                delta  = Diff(closestMirror[p][p1][i], target[p][p1][i]) +
                                         Diff(v, target[p][p1][i]);
                                delta1 = Diff(closestMirror[p][p1][i], target[p][p1][i]);
                                dSec = minutes*60 * delta1 / delta  - (minutes*60.0);
                                date0 = [date dateByAddingYears:0 months:0 days:0 hours:0 minutes:dSec/60 seconds:0];

                                v = floor((m+target[p][p1][i]/2.0)/target[p][p1][i])*target[p][p1][i];
                                if (v >= 360.0) v = 0.0;
                                entry = [NSArray arrayWithObjects:[planets objectAtIndex:p1],
                                        [dividers objectAtIndex:i], [NSNumber numberWithDouble:v], @"Mirror", nil];
                                entryDict = [[dict objectForKey:[planets objectAtIndex:p]] objectForKey:@"aspects"];
                                [entryDict setObject:entry forKey:date0];
                                closestMirror[p][p1][i] = 1000.0;
                            }
                            lastMirror[p][p1][i] = is;
                        }
                    } // end mirrors
                }
            }
        } // end aspects

        date = [date dateByAddingYears:0 months:0 days:0 hours:0 minutes:minutes seconds:0];
    }

NSLog(@"Search dict = %@", dict);
    return dict;
}


/* search aspects
 * created:  2004-05-20
 * modified: 2004-05-22
 *
 * return
 * {
 *    date = (planet1, planet2, divider, angle);
 *    date = (planet1, planet2, divider, mirror, "Mirror");
 * }
 */
- (NSDictionary*)searchAspect:(NSString*)planet1 :(NSString*)planet2
                     dividers:(NSArray*)dividers
                 searchAngles:(BOOL)searchAngles searchMirrors:(BOOL)searchMirrors
                     fromDate:(NSCalendarDate*)dateFrom toDate:(NSCalendarDate*)dateTo
               searchInterval:(int)minutes
{   NSMutableDictionary	*dict = [NSMutableDictionary dictionary];
    NSCalendarDate	*date = dateFrom, *date0;
    int			p1 = [ephemeris indexForObjectName:planet1];
    int			p2 = [ephemeris indexForObjectName:planet2];
    int			i, divCnt = [dividers count];
    double		target[divCnt], lastAngle[divCnt], closestAngle[divCnt];
    double		lastMirror[divCnt], closestMirror[divCnt];
    NSArray		*entry;

    if (!searchAngles && !searchMirrors)
        return dict;
    dividers = [dividers sortedArrayUsingSelector:@selector(compare:)];
    for (i=0; i<divCnt; i++)
    {   target[i] = 360.0 / [[dividers objectAtIndex:i] floatValue];
        lastAngle[i]  = closestAngle[i]  = 1000.0;
        lastMirror[i] = closestMirror[i] = 1000.0;
    }

    while ([date compare:dateTo] != NSOrderedDescending)	// date <= dateTo
    {   NSAutoreleasePool	*pool = [NSAutoreleasePool new];
        EphObject		*objects = [ephemeris objectsAtUTC:date];

        [date retain];
        if (searchAngles)
        {   double	a = angle(objects[p1].lon, objects[p2].lon);

            for (i=divCnt-1; i>=0; i--)	// use the biggest divider, if more than one divider fit
            {   double	is = a - floor(a / target[i]) * target[i];

                // FIXME: we should ignore angles also covered by higher dividers !!!
                if ( Diff(is, target[i]) <= Diff(lastAngle[i], target[i]) )
                    closestAngle[i] = is;
                else if (closestAngle[i] < 360.0 && Diff(closestAngle[i], target[i]) < 1.0)
                {   double	v, delta, delta1;
                    int		dSec;

                    /* get exact date: date += deltaTime * delta1 / delta */
                    v = (is < closestAngle[i]) ? is+target[i] : is;
                    delta  = Diff(closestAngle[i], target[i]) + Diff(v, target[i]);
                    delta1 = Diff(closestAngle[i], target[i]);
                    dSec = minutes*60 * delta1 / delta  - (minutes*60.0);
                    date0 = [date dateByAddingYears:0 months:0 days:0 hours:0 minutes:dSec/60 seconds:0];

//printf("%s %s  v=%f delta=%f delta1 = %f\n",
//[[date descriptionWithCalendarFormat:@"%Y%m%d %H:%M:%S"] cString],
//[[date0 descriptionWithCalendarFormat:@"%Y%m%d %H:%M:%S"] cString],
//v, delta, delta1);

                    v = floor((a+target[i]/2.0)/target[i])*target[i];	// target angle
                    entry = [NSArray arrayWithObjects:planet1, planet2,
                                                     [dividers objectAtIndex:i],
                                                     [NSNumber numberWithDouble:v],
                                                     nil];
                    [dict setObject:entry forKey:date0];
                    //printf("%f: is=%f last=%f closest=%f\n", target[i], is, lastAngle[i], closestAngle[i]);
                    //printf("* * * %s * * *\n", [[entry description] cString]);
                    closestAngle[i] = 1000.0;
                }
                lastAngle[i] = is;
            }
        }

        if (searchMirrors)
        {   double	m = mirror(objects[p1].lon, objects[p2].lon);

            for (i=0; i<divCnt; i++)
            {   double	is = m - floor(m / target[i]) * target[i];

                // FIXME: we should ignore mirror angles covered by higher dividers !!!
                if ( Diff(is, target[i]) <= Diff(lastMirror[i], target[i]) )
                    closestMirror[i] = is;
                else if (closestMirror[i] < 360.0 && Diff(closestMirror[i], target[i]) < 1.0)
                {   double	v, delta, delta1;
                    int		dSec;

                    /* get exact date: date += deltaTime * delta1 / delta */
                    v = (is < closestMirror[i]) ? is+target[i] : is;
                    delta  = Diff(closestMirror[i], target[i]) + Diff(v, target[i]);
                    delta1 = Diff(closestMirror[i], target[i]);
                    dSec = minutes*60 * delta1 / delta  - (minutes*60.0);
                    date0 = [date dateByAddingYears:0 months:0 days:0 hours:0 minutes:dSec/60 seconds:0];

                    v = floor((m+target[i]/2.0)/target[i])*target[i];	// target mirror angle
                    entry = [NSArray arrayWithObjects:planet1, planet2,
                                                      [dividers objectAtIndex:i],
                                                      [NSNumber numberWithDouble:v],
                                                      @"Mirror", nil];
                    [dict setObject:entry forKey:date];
                    //printf("%f: is=%f last=%f closest=%f\n", target[i], is, lastMirror[i], closestMirror[i]);
                    //printf("* * * %s * * *\n", [[entry description] cString]);
                    closestMirror[i] = 1000.0;
                }
                lastMirror[i] = is;
            }
        }

        [date autorelease];
        date = [[date dateByAddingYears:0 months:0 days:0 hours:0 minutes:minutes seconds:0] retain];
        [pool release];
    }

//NSLog(@"Search dict = %@", dict);

    return dict;
}

@end
