/* AstroMandala.m
 * Fractal Astrological Mandalas
 * Beautiful patterns generated by interfering
 * wave harmonics in relation to Planets, Ecliptic, and location.
 * The fractal patterns also display key knowledge of the
 * harmonic relation between angles, mirrors and the
 * various interfering cycles.
 *
 * Copyright (C) 2003-2005 by vhf interservice GmbH
 * Author:   Georg Fleischmann
 *
 * created:  2003-11-23
 * modified: 2005-08-24
 *
 * vhf interservice GmbH, Im Marxle 3, 72119 Altingen, Germany
 * eMail: info@vhf.de
 * http://www.vhf.de
 */

#include <AppKit/AppKit.h>
#include <VHFShared/VHFStringAdditions.h>
#include <VHFShared/VHFDictionaryAdditions.h>
#include "../Cenon/functions.h"		// vhfRGBColorFromString()
#include "../Cenon/Graphics.h"
#include "DocViewAstro.h"
#include "AstroPrincipal.h"
#include "AstroMandala.h"
#include "astroCommon.h"
#include "colorFunctions.h"
#include "TopocentricCycle.h"
#include "SwissEphemeris.subproj/Ephemeris.h"
#include "Astro.bproj/AstroController.h"

#define RADIUS_1	240.0/204.0	//1.0	// Ecliptic
#define RADIUS_2	1.00		//0.85	// scale Event 1
#define RADIUS_3	144.0/204.0	//0.60	// scale Event 2
#define RADIUS_4	84.0/204.0	//0.35
#define SCALE_MIN	((RADIUS_2-RADIUS_3) / 10.0)
#define SCALE_MAX	(SCALE_MIN * 2.0)
#define SYMBOL_SIZE	0.12
#define FONT_SIZE	10.0	// fix
#define LINE_HEIGHT	11.0	// fix

/* Private methods
 */
@interface AstroMandala(PrivateMethods)
@end

@implementation AstroMandala

+ (id)mandalaWithView:(DocView*)aView
{
    return [[[AstroMandala alloc] initWithView:aView] autorelease];
}
- (id)initWithView:(DocView*)aView
{
    [super init];
    view = [aView retain];

    /* defaults */
    mandalaType   = TYPE_FRACTAL_SPIRAL;
    octaveAngle   = 180;
    recursions    = 2;
    maxMapDivider = 12;

    return self;
}
- (void)setView:(id)aView
{
    [view release];
    view = [aView retain];
}



/*
 * General methods
 */

- (void)setFeatures:(NSDictionary*)features
{
    mandalaType = [features intForKey:@"mandalaType"];
    components = [features intForKey:@"components"];

    dividersFrom = [features intForKey:@"dividersFrom"];
    dividersTo   = [features intForKey:@"dividersTo"];

    bgColor = [features objectForKey:@"bgColor"];
    bgColor = [bgColor colorUsingColorSpaceName:NSDeviceRGBColorSpace];

    dimensionX = [features intForKey:@"dimensionX"];
    dimensionY = [features intForKey:@"dimensionY"];

    recursions    = [features intForKey:@"recursions"];
    octaveAngle   = [features intForKey:@"octaveAngle"];
    maxMapDivider = [features intForKey:@"maxMapDivider"];

    mapping = [features intForKey:@"mapping"];	// ???
}

- (BOOL)getMarksFromTemplate:(NSArray*)list
                      center:(NSPoint*)center radius:(float*)radius start:(NSPoint*)start
                      bounds:(NSRect*)rect
{   int		i, okNum = 0;
    BOOL	okCenter = NO, okStart = NO;
    NSPoint	ur   = {600.0, 600.0};

    *rect = NSMakeRect(0.0, 0.0, 600.0, 600.0);

    for (i=0; i<[list count]; i++)
    {   VMark	*g = [list objectAtIndex:i];

        if ([g isKindOfClass:[VMark class]])
        {
            if ([[g name] isEqual:@"ORIGIN_LL"])
            {   rect->origin = [g origin]; }
            if ([[g name] isEqual:@"ORIGIN_UR"])
            {   ur = [g origin]; }
            if ([[g name] isEqual:@"CENTER"])
            {   *center = [g origin]; okCenter = YES; }
            else if ([[g name] isEqual:@"START"])
            {
                *start  = [g origin];
                *radius = sqrt( SqrDistPoints(*center, [g origin]) );
                okStart = YES;
            }
            if ((++okNum) >= 2)
                break;
        }
    }
    rect->size = NSMakeSize(ur.x - rect->origin.x, ur.y - rect->origin.y);
    if (!okCenter)
        NSLog(@"Mark with name 'CENTER' missing on template layer");
    if (!okStart)
        NSLog(@"Mark with name 'START' missing on template layer");
    return okCenter && okStart;
}



static __inline__ void setBitmapColorAtIndex(unsigned char *planes[], int i, RGB rgb)
{
    planes[0][i] = (unsigned char)(rgb.r * 255.0);	// r
    planes[1][i] = (unsigned char)(rgb.g * 255.0);	// g
    planes[2][i] = (unsigned char)(rgb.b * 255.0);	// b
}

/* set image to rgb
 */
static void clearImage(unsigned char *planes[], NSSize imageSize, RGB rgb)
{   int	x, y, i;

    for (x=0; x<imageSize.width; x++)
    {
        for (y=0; y<imageSize.height; y++)
        {
            i = y * imageSize.width + x;
            planes[0][i] = (unsigned char)(rgb.r * 255.0);	// r
            planes[1][i] = (unsigned char)(rgb.g * 255.0);	// g
            planes[2][i] = (unsigned char)(rgb.b * 255.0);	// b
        }
    }
}

/* color for degree
 */
static RGB rgbColorForDegree(double deg, float saturation, float brightness)
{   NSColor	*color;
    RGB		rgb;

    color = [NSColor colorWithCalibratedHue:mod360(deg)/360.0
                                 saturation:saturation brightness:brightness alpha:1.0];
    [color getRed:&rgb.r green:&rgb.g blue:&rgb.b alpha:NULL];
    return rgb;
}

/* return n for 2^(n-1) < d <= 2^n
 * for d = 1, 2, 3, 4, ...
 *     n = 0, 1,    2, ...
 */
int octaveForDivider(int d, int *octave)
{   int	n, oct = 1;

    for (n=0; oct < d; n++)
        oct *= 2;
    if (octave != NULL)
        *octave = oct;
    return n;
}

/* angle with fixed octave size
 * (otherwise we get fern like structures instead of neat spirals)
 * octaves: 1, 2, 3/4, 5/6/7/8, ...
 */
static double alphaForDividerWithFixedOctaveSize(int d, double octaveSize)
{   int		n, octave;
    double	octAngle, alpha = 360.0 / d, f;

    if (octaveSize == 0.0)
        return alpha;
    n = octaveForDivider(d, &octave);
    octAngle = 360.0 / octave;
    f = (d == 1) ? 0.0 : (2.0*octAngle - alpha) / octAngle;
    /*printf("d = %d  n = %d  a = %.2f  octA = %.2f  f = %.2f -> %.2f\n",
            d, n, alpha, octAngle, f, mod360((((n) ? (n-1) : 0) * octaveSize) + f * octaveSize));*/
    alpha = ((n) ? (n-1) : 0) * octaveSize;	// end angle of previous octave
    alpha = mod360(alpha + f * octaveSize);	// delta degree / size of octave
    return alpha;
}

static VMark *marking(NSPoint center, float diameter, NSString *name)
{   VMark	*mark = [VMark markWithOrigin:center diameter:diameter];

    if (name)
        [mark setName:name];
    return mark;
}

/*
 * TYPE: Fractal Spirals
 */

/* recursive spriral at center
 * created:  2005-08-22
 * modified: 2005-08-24
 */
typedef struct
{
    unsigned char	*planes[5];	// planes
    NSSize		size;		// image size
    //int		recursions;	// number of recursions
    //int		octaveAngle;
    //BOOL		counterSpiral;
}ImageInfo;

typedef struct
{
    NSPoint	center;	// center point of our parent spiral
    float	rad;	// radius of our parent spiral
    //int		d;	// divider of our parent spirale
    float	offset;	// 0 for normal and 180 degree for the counter spirals
    int		cnt;	// recursion count
}FifoEntry;
- (void)spiralLoopAtCenter:(NSPoint)cp radius:(float)radius angle:(float)offsetAlpha
                 imageInfo:(ImageInfo*)info
{   int		d = dividersFrom, d1, dCnt = dividersTo - dividersFrom, cnt = 0;
    float	colorEnd, colorRange, r = radius;
    FifoEntry	*fifo, *fifoPtr;
    int		fifoStart = -1, fifoEnd = -1, fifoSize, fifoCnt = 0;
    BOOL	counterSpiral = YES;
    int		fifoMax = 500000;	// limit memory use to about 50 MB
    float	octAngle = (mandalaType == TYPE_FRACTAL_SPIRAL) ? octaveAngle : 0;

    if (!dCnt)
        return;

    /* FIFO
     * we add new entries to the end of the fifo
     * and remove entries from the beginning of the fifo
     */
    fifoSize = Min((int)pow((double)dCnt, (double)recursions), fifoMax);
    fifo = NSZoneCalloc(NSDefaultMallocZone(), fifoSize, sizeof(FifoEntry));

    while (1)
    {
        /* color shading ???
         * 1. from d-1 to d   (d determines center color)
         * 2. from d   to d+1 (d+1 = center color)
         * 3. from 0   to d   (always starts at top)
         * 4. from d   to 0   (always ends at bottom)
         */
        colorEnd   = 360.0 / d;					// 1. color end at d
        colorRange = 360.0 / ((d<=1) ? 1.0 : ((d-1)*d));	// 1. color range of spiral from d-1 to d
        for (d1 = d; d1 <= d+dCnt; d1++)
        {   float	alpha = alphaForDividerWithFixedOctaveSize(d1, octAngle);
            NSPoint	sp = NSMakePoint(cp.x + r/d1, cp.y), p;

            p = vhfPointRotatedAroundCenter(sp, alpha+offsetAlpha, cp);
            if (p.x >= 0.0 && p.y >= 0.0 &&
                p.x < info->size.width && p.y < info->size.height)
            {   float	a = colorEnd + colorRange / (d1-d+1);	// 1. color segment from d-1 to d

                setBitmapColorAtIndex(info->planes,
                                      (int)(((int)p.y * info->size.width) + p.x),
                                      rgbColorForDegree(a, 1.0, 1.0));	// colors shade from d-1 to center color (d)
                //[group addObject:marking(p, 2.0, [NSString stringWithFormat:@"d = %d, a = %.0f", d1, alpha])];
            }
            if (cnt < recursions && r/d1 > 1.0 && fifoCnt < fifoSize)	// add parameters to recursion list
            {
                fifoEnd = (fifoEnd+1 < fifoSize) ? fifoEnd+1 : 0;
                fifoPtr = fifo+fifoEnd;
                fifoPtr->center = p;
                fifoPtr->rad    = r/d1; //r/(d1 * (d1+1));	// r/d1;
                //fifoPtr->d      = d;
                fifoPtr->offset = alpha+offsetAlpha;	// offsetAlpha;
                fifoPtr->cnt    = cnt + 1;
                fifoCnt ++;
            }

            if (counterSpiral)
            {
                p = vhfPointRotatedAroundCenter(sp, alpha+offsetAlpha+180, cp);
                if (p.x >= 0.0 && p.y >= 0.0 &&
                    p.x < info->size.width && p.y < info->size.height)
                {   float	a = colorEnd + colorRange / (d1-d+1);	// 1. color segment from d-1 to d

                    setBitmapColorAtIndex(info->planes,
                                          (int)(((int)p.y * info->size.width) + p.x),
                                          rgbColorForDegree(a, 1.0, 1.0));	// colors shade from d-1 to center (d)
                    //[group addObject:marking(p, 2.0, [NSString stringWithFormat:@"d = %d, a = %.0f", d1, alpha])];
                }
                if (cnt < recursions && r/d1 > 1.0 && fifoCnt < fifoSize)	// add parameters to recursion list
                {
                    fifoEnd = (fifoEnd+1 < fifoSize) ? fifoEnd+1 : 0;
                    fifoPtr = fifo+fifoEnd;
                    fifoPtr->center = p;
                    fifoPtr->rad    = r/d1; //r/(d1 * (d1+1));
                    //fifoPtr->d      = d;
                    fifoPtr->offset = alpha+offsetAlpha+180;	// offsetAlpha+180;
                    fifoPtr->cnt    = cnt + 1;
                    fifoCnt ++;
                }
            }
        }

        if (fifoStart == fifoEnd)
            break;
        fifoStart = (fifoStart < fifoSize-1) ? fifoStart+1 : 0;		// remove entry from fifo
        if (fifoStart >= 0)
            fifoCnt --;

        fifoPtr = fifo+fifoStart;	// get first entry from recursion fifo
        cp = fifoPtr->center;
        r  = fifoPtr->rad;
        //d  = fifoPtr->d;
        offsetAlpha = fifoPtr->offset;
        cnt = fifoPtr->cnt;
    }
    NSZoneFree(NSZoneFromPointer(fifo), fifo);
}

- (VGroup*)mapAtCenter:(NSPoint)center radius:(float)radius angle:(float)offsetAlpha
            maxDivider:(int)dMax recursions:(int)recur
{   int		d = dividersFrom, d1, dCnt = dMax - dividersFrom, cnt = 0;
    NSPoint	cp = center;
    float	r = radius;
    FifoEntry	*fifo, *fifoPtr;
    int		fifoStart = -1, fifoEnd = -1, fifoSize, fifoCnt = 0;
    BOOL	counterSpiral = NO, showDivider = YES;
    NSString	*string;
    int		fifoMax = 500000;	// limit memory use to about 50 MB
    VGroup	*group = [VGroup group];
    VText	*text;
    VLine	*line;
    NSFont	*systemFont12 = [NSFont systemFontOfSize:12.0];
    float	octAngle = (mandalaType == TYPE_FRACTAL_SPIRAL) ? octaveAngle : 0;

    if (!dCnt)
        return nil;

    /* FIFO */
    fifoSize = Min((int)pow((double)dCnt, (double)recur), fifoMax);
    fifo = NSZoneCalloc(NSDefaultMallocZone(), fifoSize, sizeof(FifoEntry));

    while (1)
    {
        for (d1 = d; d1 <= d+dCnt; d1++)
        {   float	alpha = alphaForDividerWithFixedOctaveSize(d1, octAngle);
            NSPoint	sp = NSMakePoint(cp.x + r/d1, cp.y), p;

            p = vhfPointRotatedAroundCenter(sp, alpha+offsetAlpha, cp);
            if (p.x >= 0.0 && p.y >= 0.0)
            {   NSColor	*color = [NSColor colorWithCalibratedWhite:1.0-1.0/(cnt+1) alpha:1.0];

                line = [VLine lineWithPoints:NSMakePoint(cp.x, 2*center.y-cp.y)
                                            :NSMakePoint( p.x, 2*center.y- p.y)];
                [line setColor:color];
                [group addObject:line];
                if ( r/d1 > 1*12 )
                {   NSPoint	p1 = p;

                    if (d1 > 1)
                        p1 = vhfPointRotatedAroundCenter(NSMakePoint(cp.x + r/d1 + 1*12, cp.y), alpha+offsetAlpha, cp);
                    string = (showDivider) ? [NSString stringWithFormat:@"%d", d1]
                                           : [NSString stringWithFormat:@"%.0f", alpha];
                    text = [view text:string center:NSMakePoint(p1.x, 2*center.y-p1.y)
                                color:color angle:0.0 center:p1 font:systemFont12];
                    [group addObject:text];
                }
                //[group addObject:marking(p, 2.0, [NSString stringWithFormat:@"d = %d, a = %.0f", d1, alpha])];
            }
            if (cnt < recur && r/d1 > 1.0 && fifoCnt < fifoSize)	// add parameters to recursion list
            {
                fifoEnd = (fifoEnd+1 < fifoSize) ? fifoEnd+1 : 0;
                fifoPtr = fifo+fifoEnd;
                fifoPtr->center = p;
                fifoPtr->rad    = r/d1; //r/(d1 * (d1+1));
                //fifoPtr->d      = d;
                fifoPtr->offset = alpha+offsetAlpha;	//offsetAlpha
                fifoPtr->cnt    = cnt + 1;
                fifoCnt ++;
            }

            if (counterSpiral)
            {
                p = vhfPointRotatedAroundCenter(sp, alpha+offsetAlpha+180, cp);
                if (p.x >= 0.0 && p.y >= 0.0)
                {
                }
                if (cnt < recur && r/d1 > 1.0 && fifoCnt < fifoSize)	// add parameters to recursion list
                {
                    fifoEnd = (fifoEnd+1 < fifoSize) ? fifoEnd+1 : 0;
                    fifoPtr = fifo+fifoEnd;
                    fifoPtr->center = p;
                    fifoPtr->rad    = r/d1;
                    //fifoPtr->d      = d1;
                    fifoPtr->offset = offsetAlpha+180;
                    fifoPtr->cnt    = cnt + 1;
                    fifoCnt ++;
                }
            }
        }

        if (fifoStart == fifoEnd)
            break;
        fifoStart = (fifoStart < fifoSize-1) ? fifoStart+1 : 0;		// remove entry from fifo
        if (fifoStart >= 0)
            fifoCnt --;

        fifoPtr = fifo+fifoStart;	// get first entry from recursion fifo
        cp = fifoPtr->center;
        r  = fifoPtr->rad;
        //d  = fifoPtr->d;
        offsetAlpha = fifoPtr->offset;
        cnt = fifoPtr->cnt;
    }
    NSZoneFree(NSZoneFromPointer(fifo), fifo);
    return group;
}

/* calculate fractal mandala (kaleidoscope)
 * created:  2005-08-12
 * modified: 2005-08-24
 *
 * we use 'bounds' from the template for our image
 * we use dividersFrom - dividersTo
 *
 * 1. we draw spirals for each harmonic approach for all n = 1, 2, 3, 4, ...
 *    n - n/2 - n/3 - n/4 ...
 * 2. Mirrors ???
 *    a) we mirror from an inner starting geometry radial to the outside.
 *    b) we start with large dividers (small) and mirror the accumulated
 *       geometry by the next smaller divider.
 */
- (VGraphic*)mandalaFractal
{   VGroup		*group = [VGroup group];
    VImage		*image = [[VImage new] autorelease];
    NSPoint		center;//, sp, cp, cp1;
    NSSize		imageSize = bounds.size;
    int			w = imageSize.width, h = imageSize.height;//, d, i;
    NSBitmapImageRep	*bitmap;
    NSImage		*nsImage;
    unsigned char	*planes[5];	// RGB
    ImageInfo		imageInfo;
    double		radius;
    RGB			rgb;

    degreeDict = [ephemeris objectDictAtUTC:utc];

    /* create image and bitmap */
    nsImage = [[NSImage allocWithZone:(NSZone *)[self zone]] initWithSize:imageSize];
    bitmap = [[NSBitmapImageRep alloc] initWithBitmapDataPlanes:NULL
                                                     pixelsWide:w pixelsHigh:h
                                                  bitsPerSample:8
                                                samplesPerPixel:3	// RGB
                                                       hasAlpha:NO
                                                       isPlanar:YES
                                                 colorSpaceName:NSCalibratedRGBColorSpace
                                                    bytesPerRow:w
                                                   bitsPerPixel:8];
    [bitmap getBitmapDataPlanes:planes];
    [bitmap getBitmapDataPlanes:imageInfo.planes];
    //image = [[[VImage alloc] initWithRepresentations:[NSArray arrayWithObjects:bitmap, nil]] autorelease];
    imageInfo.size          = imageSize;
    //imageInfo.recursions    = recursions;
    //imageInfo.octaveAngle   = (mandalaType == TYPE_FRACTAL_SPIRAL) ? octaveAngle : 0;
    //imageInfo.counterSpiral = counterSpiral;

    [bgColor getRed:&rgb.r green:&rgb.g blue:&rgb.b alpha:NULL];
    clearImage(planes, imageSize, rgb /*VHFMakeRGB(1.0, 1.0, 1.0)*/);

    /* draw spirals */
    center = NSMakePoint(imageSize.width/2.0, imageSize.height/2.0);	// center point for spiral
    radius = imageSize.height*0.50;
    [self spiralLoopAtCenter:center radius:radius angle:0 imageInfo:&imageInfo];

    /* map
     * FIXME: this should go to the main method
     */
    {   NSArray	*layerList = [view layerList];

        if ([layerList count] >= 4)	// 0, 1, 2, 3, or more
        {   int		ixMap = 2;
            LayerObject	*mapLayer = [layerList objectAtIndex:ixMap];

            if (mapLayer >= 0 && [mapLayer state])
            {   VGroup	*mapGroup;

                [mapLayer removeAllObjects];
                mapGroup = [self mapAtCenter:center radius:radius angle:0
                                  maxDivider:Min(dividersTo, maxMapDivider)
                                  recursions:Min(recursions, 2)];
                [mapLayer addObject:mapGroup];
            }
        }
    }

    [nsImage setDataRetained:YES];
    [nsImage addRepresentation:bitmap]; [bitmap release];
    [nsImage setScalesWhenResized:YES];
    [image setImage:nsImage]; [nsImage release];
    [image setSize:imageSize];
    [image setOrigin:bounds.origin];
    //return image;

    [group addObject:image];
    return group;
}



/*
 * TYPE: color Dots
 */
static NSArray *colorWheel(float s)
{   NSArray	*colorWheel = nil;

    //if (!colorWheel)
        colorWheel = [NSArray arrayWithObjects:
            [NSColor colorWithCalibratedHue:0.000 saturation:s brightness:1.0 alpha:1.0], // aries
            [NSColor colorWithCalibratedHue:0.028 saturation:s brightness:1.0 alpha:1.0], // taurus
            [NSColor colorWithCalibratedHue:0.056 saturation:s brightness:1.0 alpha:1.0], // gemini
            [NSColor colorWithCalibratedHue:0.083 saturation:s brightness:1.0 alpha:1.0], // cancer
            [NSColor colorWithCalibratedHue:0.167 saturation:s brightness:1.0 alpha:1.0], // leo
            [NSColor colorWithCalibratedHue:0.250 saturation:s brightness:1.0 alpha:1.0], // virgo
            [NSColor colorWithCalibratedHue:0.333 saturation:s brightness:1.0 alpha:1.0], // libra
            [NSColor colorWithCalibratedHue:0.500 saturation:s brightness:1.0 alpha:1.0], // scorpio
            [NSColor colorWithCalibratedHue:0.583 saturation:s brightness:1.0 alpha:1.0], // sagittarius
            [NSColor colorWithCalibratedHue:0.667 saturation:s brightness:1.0 alpha:1.0], // capricorn
            [NSColor colorWithCalibratedHue:0.750 saturation:s brightness:1.0 alpha:1.0], // aquarius
            [NSColor colorWithCalibratedHue:0.916 saturation:s brightness:1.0 alpha:1.0], // pisces
            [NSColor colorWithCalibratedHue:0.000 saturation:s brightness:1.0 alpha:1.0], // aries
            nil];
    return colorWheel;
}

/* 1. Houses
 */
- (VGroup*)housesWithCenter:(NSPoint)center radius:(float)radius
                        lat:(float)latitude lon:(float)longitude gmt:(NSCalendarDate*)gmt
{   NSPoint		p;
    VGroup		*group = [VGroup group];
    VPath		*path;
    NSMutableArray	*list;
    VLine		*line;
    VArc		*arc;
    int			i;
    float		rotateAC = 0.0;
    int			houseCnt = 12;
    double		*house = [ephemeris houseAtUTC:gmt latitude:latitude longitude:longitude
                                            houseCount:houseCnt];
    NSArray		*houseColors = colorWheel(1.0);

    if (Prefs_ACFix)
        rotateAC = 180.0 - house[1];	// rotation of chart

    for ( i=1; i<=houseCnt; i++ )
    {   double	a = house[i], a1 = house[(i<houseCnt) ? i+1 : 1];

        if (rotateAC)
        {   a  = standardAngle(a  + rotateAC);
            a1 = standardAngle(a1 + rotateAC);
        }

        path = [VPath path];
        [path setFillColor:[houseColors objectAtIndex:i-1]];
        [path setFilled:YES optimize:NO];
        list = [path list];

        arc = [VArc arc];
        p = vhfPointRotatedAroundCenter(NSMakePoint(center.x+radius, center.y), a, center);
        [arc setCenter:center start:p angle:angle(a1, a)];
        [list addObject:arc];

        line = [VLine lineWithPoints:[arc pointWithNum:PT_END] :center];
        [list addObject:line];

        line = [VLine lineWithPoints:center :p];
        [list addObject:line];

        [group addObject:path];
    }

    return group;
}

/* 2. Ecliptic
 */
- (VGroup*)eclipticWithCenter:(NSPoint)center radius:(float)radius rotate:(float)angle
{   int			i;
    VArc		*arc;
    VLine		*line;
    VPath		*path;
    VGroup		*group = [VGroup group];
    NSMutableArray	*list;
    NSPoint		p0, p1;
    NSArray		*zodiacColors = colorWheel(1.0);

    /* 0 - 30 deg segments
     */
    for (i=0; i<12; i++)
    {   float	radius1 = radius * 0.9, radius2 = radius * 0.3;

        path = [VPath path];
        [path setFillColor:[zodiacColors objectAtIndex:i]];
        [path setFilled:YES optimize:NO];
        list = [path list];
        arc = [VArc arc];
        p0 = vhfPointRotatedAroundCenter(NSMakePoint(center.x+radius1, center.y), angle+i*30.0,     center);
        [arc setCenter:center start:p0 angle:30.0];
        [list addObject:arc];
        p1 = vhfPointRotatedAroundCenter(NSMakePoint(center.x+radius2, center.y), angle+(i+1)*30.0, center);
        line = [VLine lineWithPoints:[arc pointWithNum:PT_END] :p1];
        [list addObject:line];
        arc = [VArc arc];
        [arc setCenter:center start:p1 angle:-30.0];
        [list addObject:arc];
        line = [VLine lineWithPoints:[arc pointWithNum:PT_END] :p0];
        [list addObject:line];
        [group addObject:path];
    }

    return group;
}

/* 3. Planets
 */
- (VGroup*)planetsWithCenter:(NSPoint)center radius:(float)radius
                         lat:(float)latitude lon:(float)longitude gmt:(NSCalendarDate*)gmt
{   //NSDictionary	*degreeDict;
    //NSMutableArray	*planets = [NSMutableArray arrayWithObjects:@"Mars", @"Venus", @"Mercury", @"Moon", @"Sun", @"Mercury", @"Venus", @"Pluto", @"Jupiter", @"Saturn", @"Uranus", @"Neptune", nil];
    NSPoint		p;
    VArc		*arc;
    VGroup		*group = [VGroup group];
    int			i;
    float		rotateAC = 0.0, r = radius / (float)[planets count] / 1.0;
    NSArray		*planetColors = colorWheel(1.0);

    degreeDict = [ephemeris objectDictAtUTC:gmt lat:latitude lon:longitude];
    if (Prefs_ACFix)
        rotateAC = 180.0 - [degreeDict floatForKey:@"AC"];	// rotation of chart

    for ( i=0; i<(int)[planets count]; i++ )
    {   NSString	*planet = [planets objectAtIndex:i];
        float		angle = [degreeDict floatForKey:planet];

        if (rotateAC)
            angle = standardAngle(angle + rotateAC);
        p = vhfPointRotatedAroundCenter(NSMakePoint(center.x + radius-((float)i*(r/2.0))-r, center.y), angle, center);
        arc = [VArc arc];
        [arc setCenter:p start:NSMakePoint(p.x + r, p.y) angle:360.0];
        [arc setFilled:YES];
        [arc setFillColor:[planetColors objectAtIndex:i]];

        [group addObject:arc];
    }

    return group;
}

- (VText*)textWithSubstring:(NSString*)name fromTemplate:(NSArray*)list
{   unsigned	i;

    for (i=0; i<[list count]; i++)
    {   VText	*g = [list objectAtIndex:i];

        if ([g isKindOfClass:[VText class]])
        {   NSRange	range = [[g string] rangeOfString:name];

            if (range.length)
               return [[g copy] autorelease];
        }
    }
    NSLog(@"Substring '%@' missing in Template", name);
    return nil;
}

/* 4. Text
 */
- (VGroup*)textWithCenter:(NSPoint)cp radius:(float)radius
                 template:(NSArray*)template title:(NSString*)title
                      lat:(float)latitude lon:(float)longitude city:(NSString*)city gmt:(NSCalendarDate*)gmt
{   NSRect		r;
    VGroup		*group = [VGroup group];
    VText		*text;
    NSFont		*systemFont12 = [NSFont systemFontOfSize:12.0];
    NSFont		*systemFont20 = [NSFont systemFontOfSize:20.0];

    /* title */
    if (template && (text = [self textWithSubstring:@"#TITLE#" fromTemplate:template]))
        [text replaceTextWithString:title];
    else
        text = [view text:[title stringByAppendingString:@" "] origin:NSMakePoint(20.0, 790.0)
                    color:[NSColor blackColor] angle:0.0 center:cp
               lineHeight:0.0 align:NSLeftTextAlignment font:systemFont20];
    [group addObject:text];

    /* title, date, city, lat, lng
     */
    if (template && (text = [self textWithSubstring:@"#NAME#" fromTemplate:template]))
    {   NSMutableString	*string = [[[text string] mutableCopy] autorelease];
        NSRange		range;
        NSString	*ns, *ew;

        range = [string rangeOfString:@"#NAME#"];
        [string replaceCharactersInRange:range withString:title];
        range = [string rangeOfString:@"#DATE#"];
        [string replaceCharactersInRange:range
                              withString:[gmt descriptionWithCalendarFormat:@"%d.%m.%Y-%H:%M %Z"]];
        range = [string rangeOfString:@"#LOCATION#"];
        ns = (latitude  >= 0.0) ? @"N" : @"S";
        ew = (longitude >= 0.0) ? @"E" : @"W";
        if (Prefs_DegreesMinutes)
            [string replaceCharactersInRange:range
                                  withString:[NSString stringWithFormat:@"%@ %@ %@ %@ %@",
                                      city, stringFromDeg(Abs(latitude)),  ns,
                                            stringFromDeg(Abs(longitude)), ew]];
        else
            [string replaceCharactersInRange:range
                                  withString:[NSString stringWithFormat:@"%@ %.3f %@, %.3f %@",
                                      city, Abs(latitude), ns, Abs(longitude), ew]];
        [text replaceTextWithString:string];
    }
    else
    {   text = [view text:[NSString stringWithFormat:@"%@ \n%@ \n%@ %.3f N, %.3f E ", title,
                          [gmt descriptionWithCalendarFormat:@"%d.%m.%Y-%H:%M %Z"], city, latitude, longitude]
                   origin:NSMakePoint(20.0, cp.y - radius*RADIUS_1) color:[NSColor blackColor] angle:0.0
                   center:cp lineHeight:0.0 align:NSLeftTextAlignment font:systemFont12];
        r = [text bounds];
        [text moveTo:NSMakePoint(r.origin.x, r.origin.y - r.size.height)];
    }
    [group addObject:text];

    return group;
}



/*
 * TYPE: Image
 */

/* return the amplitude at a given position on a planetary wave on ecliptic.
 * The planetary wave is relative to the planet.
 * At the planet the amplitude is maximal and the gradient is maximal too (1 - Sin(0)).
 * That is because the planet is actually an accumulation at a nodal point.
 * planetDeg	degree of planet on ecliptic
 * targetDeg	degree of the position we need the amplitude in the planet cycle
 * d		divider for which we need the amplitude
 */
static double planetAmplitude(double targetDeg, double planetDeg, double d)
{   double	waveLen = 360.0 / d;
    double	a = angle(targetDeg, planetDeg);

    if (a >= waveLen)
        a -= (double)((int)(a/waveLen)) * waveLen;	// faster than floor()
    a *= d / 2.0;			// transform to [0.0, 180.0] degree interval
    return 1.0 - Sin(a);		// amplitude = [0.0, 1.0]
}

- (RGB)mandalaColorAtLocation:(NSPoint)p size:(NSSize)imageSize
{   float	eclipticDeg = 0.0, localDeg, dSum = 0.0;
    int		d, i, cnt;
    RGB		rgbColor = VHFMakeRGB(0.0, 0.0, 0.0);
    //RGB		rgbColor = VHFMakeRGB(1.0, 1.0, 1.0);
    //static int	primeNumbers[20] = {1, 2, 3, 5, 7, 11};

    /* image position expressed in cycle degrees
     * - cartesian: x and y axis
     * - circular:  radial and axial
     */
    switch (mapping)
    {
        case MAP_CARTESIAN:	// x + y
            if (dimensionX)
                eclipticDeg += p.x * 360.0 / imageSize.width;
            if (dimensionY)
                eclipticDeg += p.y * 360.0 / imageSize.height;
            localDeg = eclipticDeg;	// FIXME: use conversion functions
            break;
        case MAP_CIRCULAR:	// radial + axial
        {   NSPoint	c = NSMakePoint(imageSize.width/2.0, imageSize.height/2.0);
            float	r = Min(imageSize.width, imageSize.height)/2.0;
            float	dx, dy, r1;
            float	deg = 0, deg1 = 0;

            dx = p.x - c.x;
            dy = p.y - c.y;
            r1 = sqrt(dx*dx + dy*dy);
            if (dimensionX)
            {   deg = (r1 != 0.0) ? Asin(dy / r1) : 0.0;	// radial (rays)
                if (dx < 0.0)			// 2. Q + 3. Q
                    deg = 180.0 - deg;
                else if (dx > 0.0 && dy < 0.0)	// 4. Q
                    deg += 360.0;
            }
            if (dimensionY)
                deg1 = r1 * 360.0 / r;				// axial  (concentric)
            eclipticDeg = deg + deg1;
        }
    }
    if (eclipticDeg >= 360.0)
        eclipticDeg -= 360.0;

    /* obtain color at cycle degree
     * Combining colors:
     *  - add:    (rgb + rgb) / n
     *  - filter: rgb * rgb
     * The Cycles:
     *  - harmonics add up
     *  - Ecliptic and Houses filter planets
     *  - planets add up
     * Options:
     *  - saturation wave
     *  - brightness wave
     */
    for (d=dividersFrom; d<=dividersTo; d++)	// 12-1 is completely different to 1-12 !!!
    {
        /* Planet waves - single color intensity wave */
        //for (i=0; i<[planets count]; i++)
        i = 1;
        {   NSString	*planet = [planets objectAtIndex:i];
            float	deg = [degreeDict floatForKey:planet];
            HSI		hsi = VHFMakeHSI(planetDeg[i], 1.0, 1.0);
            RGB		rgb;

            hsi.i = planetAmplitude(eclipticDeg, deg, d);	// 1.0 at planet
            rgb = RGBFromHSI(hsi);
            rgbColor.r += rgb.r;
            rgbColor.g += rgb.g;
            rgbColor.b += rgb.b;
            /* planets are like spotlights
             * if a red and a green one add up, they are bright yellow, not dim yellow !
             * Wie mache ich das?
             * (rgb+rgb)/2 = Halbgelb
             * rgb*rgb = schwarz
             * Am Schluss alles wieder anheben, so dass irgendwo was 1.0 ist ? Geht es besser? Direkt?
             * 
             */
            //rgbColor.r = (rgbColor.r + rgb.r) / 2.0;
            //rgbColor.g = (rgbColor.g + rgb.g) / 2.0;
            //rgbColor.b = (rgbColor.b + rgb.b) / 2.0;
        }

        /* Ecliptic waves */

        /* Houses waves */

        //dSum += 1.0/(float)d;
    }
    cnt = ((dividersTo-dividersFrom)+1) /** [planets count]*/;
    rgbColor.r /= cnt;
    rgbColor.g /= cnt;
    rgbColor.b /= cnt;
     //hsi = HSIFromRGB(rgb);
     //hsi.b = 1.0;
     //hsi.s = 1.0;
if (rgbColor.r > 1.0 || rgbColor.g > 1.0 || rgbColor.b > 1.0)
    printf("dSum=%f  %f %f %f\n", dSum, rgbColor.r, rgbColor.g, rgbColor.b);

    return rgbColor;
}
- (VImage*)mandalaImage
{   VImage		*image = [[VImage new] autorelease];
    NSPoint		p;
    NSSize		imageSize = NSMakeSize(200, 200);
    int			w = imageSize.width, h = imageSize.height;
    NSBitmapImageRep	*bitmap;
    NSImage		*nsImage;
    unsigned char	*planes[5];	// RGB

    degreeDict = [ephemeris objectDictAtUTC:utc];

    /* create image and bitmap */
    nsImage = [[NSImage allocWithZone:(NSZone *)[self zone]] initWithSize:imageSize];
    bitmap = [[NSBitmapImageRep alloc] initWithBitmapDataPlanes:NULL
                                                     pixelsWide:w pixelsHigh:h
                                                  bitsPerSample:8
                                                samplesPerPixel:3	// RGB
                                                       hasAlpha:NO
                                                       isPlanar:YES
                                                 colorSpaceName:NSCalibratedRGBColorSpace
                                                    bytesPerRow:w
                                                   bitsPerPixel:8];
    [bitmap getBitmapDataPlanes:planes];
    //image = [[[VImage alloc] initWithRepresentations:[NSArray arrayWithObjects:bitmap, nil]] autorelease];

    /* add segment colors for each divider and cycle to image
     * we ask for the color with reference to the image coordinates
     */
    for (p.y=0; p.y<imageSize.height; p.y++)
    {   NSAutoreleasePool	*pool = [NSAutoreleasePool new];

//printf("%.0f\n", p.y);
        for (p.x=0; p.x<imageSize.width; p.x++)
        {   int		i = p.y * imageSize.width + p.x;
            RGB		rgb;

            rgb = [self mandalaColorAtLocation:p size:imageSize];
            planes[0][i] = (unsigned char)(rgb.r * 255.0);	// r
            planes[1][i] = (unsigned char)(rgb.g * 255.0);	// g
            planes[2][i] = (unsigned char)(rgb.b * 255.0);	// b
//if (p.x == 10)
//    printf("%d: %d %d %d\n", i, planes[0][i], planes[1][i], planes[2][i]);
        }
        [pool release];
    }

    [nsImage addRepresentation:[bitmap autorelease]];
    [nsImage setDataRetained:YES];
    [nsImage setScalesWhenResized:YES];
    [image setImage:[nsImage autorelease]];
    [image setSize:imageSize];
    [image setOrigin:NSZeroPoint];

    return image;
}


/*
 * Main Method
 */
- (void)mandala:(NSCalendarDate*)gmt
      longitude:(float)longitude latitude:(float)latitude city:(NSString*)city
          title:(NSString*)title
{   NSArray	*layerList = [view layerList];
    LayerObject	*layer;
    VGraphic	*g;
    NSPoint	cp = NSMakePoint(296.0, 535.0);
    NSPoint	start;
    float	radius = 204.0, rotateAC = 0.0;
    int		ixMaskMandala = -1, ixMandala = 0, ixTemplate = 1, ixMap = -1, i;
    NSArray	*template = nil;	// list of marks and text showing positions

    if (!gmt || ![[[view window] title] hasPrefix:MANDALA_PREFIX])
        return;

    utc = gmt;
    lat = latitude;
    lon = longitude;

    [planets release];
    planets = [Prefs_Objects retain];

    ephemeris = [Astro_Principal ephemeris];

    /* get positions
     */
    if ([layerList count] == 0)		// no way
    {
        NSLog(@"Not enough layers (%d) to calculate mandala !", [layerList count]);
        return;
    }
    else if ([layerList count] == 1)	// use defaults
    {
        ixMandala  = 0;
    }
    else if ([layerList count] < 3)	// use template and calculate mask defaults
    {
        ixMandala  = 0;
        ixTemplate = [layerList count]-1;
        template = [[layerList objectAtIndex:ixTemplate] list];
        [self getMarksFromTemplate:template center:&cp radius:&radius start:&start bounds:&bounds];
    }
    else				// use template and masks
    {
        ixMaskMandala = 0;
        ixMandala     = 1;
        if ([layerList count] >= 4)	// 0, 1, 2, 3, or more
            ixMap = 2;
        ixTemplate    = [layerList count]-1;
        template = [[layerList objectAtIndex:ixTemplate] list];
        [self getMarksFromTemplate:template center:&cp radius:&radius start:&start bounds:&bounds];
    }


    layer = [layerList objectAtIndex:ixMandala];
    [layer removeAllObjects];

    /* Fractal */
    if (mandalaType == TYPE_FRACTAL_SPIRAL || mandalaType == TYPE_FRACTAL_LEAF)
    {
        g = [self mandalaFractal];
        [layer addObject:g];
    }

    /* mandala image */
# if 0
    if (0)
    {   VImage	*image;

        [planets release];
        planets = [[NSArray arrayWithObjects:@"Mars", @"Venus", @"Mercury", @"Moon", @"Sun", @"Pluto", @"Jupiter", @"Saturn", @"Uranus", @"Neptune", nil] retain];
        planetDeg[0] = 0.0;		// Mars
        planetDeg[1] = 30.0/360.0;	// Venus
        planetDeg[2] = 60.0/360.0;	// Mercury
        planetDeg[3] = 90.0/360.0;	// Moon
        planetDeg[4] = 120.0/360.0;	// Sun
        planetDeg[5] = 210.0/360.0;	// Pluto
        planetDeg[6] = 240.0/360.0;	// Jupiter
        planetDeg[7] = 270.0/360.0;	// Saturn
        planetDeg[8] = 300.0/360.0;	// Uranus
        planetDeg[9] = 330.0/360.0;	// Neptune

        image = [self mandalaImage];
        //layer = [layerList objectAtIndex:ixMandala];
        //[layer removeAllObjects];
        [layer addObject:image];
    }
#endif

    /* mandala with color dots
     */
    if (mandalaType == TYPE_DOTS)
    {
        //layer = [layerList objectAtIndex:ixMandala];
        //[layer removeAllObjects];
        if (Prefs_ACFix)
            rotateAC = 180.0 - ac(sideralTime(gmt, longitude), latitude);

        /* no mask -> calculate basics
         */
        if (ixMaskMandala < 0)		// calculate basics
        {
        }
        /* rotate AC to the left
         * we create a rotated copy of the mask and turn off the mask
         */
        else if (rotateAC)	// rotate Mask
        {   NSArray	*list = [[layerList objectAtIndex:ixMaskMandala] list];

            [[layerList objectAtIndex:ixMaskMandala] setState:0];
            for (i=0; i<(int)[list count]; i++)
            {   VGraphic	*g = [[[list objectAtIndex:i] copy] autorelease];

                if (![g isLocked])
                {   [g setAngle:-rotateAC withCenter:cp];
                    if ([g isKindOfClass:[VText class]] && [(VText*)g isSerialNumber])
                        [g rotate:[g rotAngle]];
                }
                [layer addObject:g];
            }
        }
        else		// no rotation -> enable mask
            [[layerList objectAtIndex:ixMaskMandala] setState:1];

        /* 1. houses   */
        g = [self housesWithCenter:cp radius:radius lat:latitude lon:longitude gmt:gmt];
        [layer addObjectsFromArray:[(VGroup*)g list]];
        /* 2. ecliptic */
        g = [self eclipticWithCenter:cp radius:radius rotate:rotateAC];
        [layer addObjectsFromArray:[(VGroup*)g list]];
        /* 3. planets  */
        g = [self planetsWithCenter:cp radius:radius lat:latitude lon:longitude gmt:gmt];
        [layer addObjectsFromArray:[(VGroup*)g list]];
        /* 4. text     */
        g = [self textWithCenter:cp radius:radius template:template
                           title:title lat:latitude lon:longitude city:city gmt:gmt];
        [layer addObjectsFromArray:[(VGroup*)g list]];
    }

    [view drawAndDisplay];
}

- (void)dealloc
{
    [view release];
    [planets release];

    [super dealloc];
}

@end
