/* AstroChart.m
 * The well known circular Astrology chart
 *
 * Copyright (C) 1998-2006 by vhf interservice GmbH
 * Author:   Georg Fleischmann
 *
 * created:  1998-12-19
 * modified: 2006-05-05
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the vhf Public License as
 * published by vhf interservice GmbH. Among other things, the
 * License requires that the copyright notices and this notice
 * be preserved on all copies.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the vhf Public License for more details.
 *
 * You should have received a copy of the vhf Public License along
 * with this program; see the file LICENSE. If not, write to vhf.
 *
 * vhf interservice GmbH, Im Marxle 3, 72119 Altingen, Germany
 * eMail: info@vhf.de
 * http://www.vhf.de
 */

#include <AppKit/AppKit.h>
#include <VHFShared/VHFStringAdditions.h>
#include <VHFShared/VHFDictionaryAdditions.h>
#include "../Cenon/functions.h"
#include "../Cenon/Graphics.h"
#include "VTextAstroAdditions.h"	// convenient text methods
#include "DocViewAstro.h"
#include "AstroPrincipal.h"
#include "AstroChart.h"
#include "AspectLines.h"
#include "astroCommon.h"
#include "TopocentricCycle.h"
#include "SwissEphemeris.subproj/Ephemeris.h"
#include "Astro.bproj/AstroController.h"

#define GRAY_COLOR	[NSColor colorWithDeviceRed:0.85 green:0.85 blue:0.8 alpha:1.0]
#define RADIUS_1	240.0/204.0	//1.0	// Ecliptic
#define RADIUS_2	1.00		//0.85	// scale Event 1
#define RADIUS_3	144.0/204.0	//0.60	// scale Event 2
#define RADIUS_4	84.0/204.0	//0.35
#define SCALE_MIN	((RADIUS_2-RADIUS_3) / 10.0)
#define SCALE_MAX	(SCALE_MIN * 2.0)
#define SYMBOL_SIZE	0.12
#define STAR_SIZE	0.25 //0.22
#define FONT_SIZE	10.0	// fix
#define LINE_HEIGHT	11.0	// fix
#define TAB_SPACING	32.0	// fix

/* Private methods
 */
@interface AstroChart(PrivateMethods)
@end

@implementation AstroChart

+ (id)astroChartWithView:(DocView*)aView
{
    return [[[AstroChart alloc] initWithView:aView] autorelease];
}
- (id)initWithView:(DocView*)aView
{
    [super init];
    view = [aView retain];
    return self;
}
- (void)setView:(id)aView
{
    [view release];
    view = [aView retain];
}
- (id)view
{
    return view;
}

/* angle deg1 - deg2 */
static NSString *formAngle(float deg1, float deg2)
{   float	a = angle(deg1, deg2);

    //angle = Diff(deg1, deg2);
    if (a>180.0)
        a = 360.0 - a;
    return [[NSString stringWithFormat:@"%.1f", a] stringWithLength:-5];
}
/* mirror deg1 - deg2 */
static NSString *formMirror(float deg1, float deg2)
{   float	m = mirror(deg1, deg2);

    if (m>180.0)
        m = 360.0 - m;
    return [[NSString stringWithFormat:@"%.1f", m] stringWithLength:-5];
}

/* aspect color
 * deg = [0, 360]
 */
static float angles[21] = {0.0, 30.0, 45.0, 60.0, 72.0, 90.0, 120.0, 135.0, 144.0, 150.0, 180.0, 210.0, 216.0, 225.0, 240.0, 270.0, 288.0, 300.0, 315.0, 330.0, 360.0};
NSColor *orbColor(float deg, float orb1, float orb2)
{   int		i;
    float	orb = MAXCOORD;

    for (i=0; i<21; i++)
        orb = Min(orb, Diff(deg, angles[i]));
    if (orb <= orb1)
        return [NSColor blackColor];
    else if (orb <= orb2)
        return [NSColor darkGrayColor];
    return [NSColor lightGrayColor];
}

/* draws a center star
 */
- (VPath*)starWithCenter:(NSPoint)center radius:(float)radius rotate:(float)angle
{   int			i;
    VLine		*line;
    VPath		*path = [VPath path];
    NSMutableArray	*list = [path list];
    NSPoint		p0, p1, p2, p3;

    for (i=0; i<4; i++)
    {
        p0 = vhfPointRotatedAroundCenter(NSMakePoint(center.x + radius,     center.y), angle+i*90.0,      center);
        p1 = vhfPointRotatedAroundCenter(NSMakePoint(center.x + radius*0.1, center.y), angle+i*90.0+22.5, center);
        p2 = vhfPointRotatedAroundCenter(NSMakePoint(center.x + radius*0.6, center.y), angle+i*90.0+45.0, center);
        p3 = vhfPointRotatedAroundCenter(NSMakePoint(center.x + radius*0.1, center.y), angle+i*90.0+67.5, center);
        line = [VLine lineWithPoints:p0 :p1];
        [list addObject:line];
        line = [VLine lineWithPoints:p1 :p2];
        [list addObject:line];
        line = [VLine lineWithPoints:p2 :p3];
        [list addObject:line];
        line = [VLine lineWithPoints:p3 :vhfPointRotatedAroundCenter(p0, 90.0, center)];
        [list addObject:line];
    }
    return path;
}

static NSArray *colorWheel(float s)
{   NSArray	*colorWheel = nil;

    //if (!colorWheel)
        colorWheel = [NSArray arrayWithObjects:
            [NSColor colorWithCalibratedHue:0.000 saturation:s brightness:1.0 alpha:1.0], // aries
            [NSColor colorWithCalibratedHue:0.028 saturation:s brightness:1.0 alpha:1.0], // taurus
            [NSColor colorWithCalibratedHue:0.056 saturation:s brightness:1.0 alpha:1.0], // gemini
            [NSColor colorWithCalibratedHue:0.083 saturation:s brightness:1.0 alpha:1.0], // cancer
            [NSColor colorWithCalibratedHue:0.167 saturation:s brightness:1.0 alpha:1.0], // leo
            [NSColor colorWithCalibratedHue:0.250 saturation:s brightness:1.0 alpha:1.0], // virgo
            [NSColor colorWithCalibratedHue:0.333 saturation:s brightness:1.0 alpha:1.0], // libra
            [NSColor colorWithCalibratedHue:0.500 saturation:s brightness:1.0 alpha:1.0], // scorpio
            [NSColor colorWithCalibratedHue:0.583 saturation:s brightness:1.0 alpha:1.0], // sagittarius
            [NSColor colorWithCalibratedHue:0.667 saturation:s brightness:1.0 alpha:1.0], // capricorn
            [NSColor colorWithCalibratedHue:0.750 saturation:s brightness:1.0 alpha:1.0], // aquarius
            [NSColor colorWithCalibratedHue:0.916 saturation:s brightness:1.0 alpha:1.0], // pisces
            nil];
    return colorWheel;
}

/*
 * Ecliptic circle: outside ring with global houses and signs
 */
- (VGroup*)eclipticWithCenter:(NSPoint)center
                     radius1:(float)radius1 radius2:(float)radius2
                  symbolSize:(float)symbolSize rotate:(float)angle
{   int			i;
    VArc		*arc;
    VLine		*line;
    VPath		*path;
    VText		*text;
    VGroup		*group = [VGroup group];
    NSMutableArray	*list;
    NSPoint		p0, p1;
    NSFont		*symbolFont = [NSFont fontWithName:astroFont size:symbolSize];

    /* filled ring */
    if (1)	// colored zodiac ring
    {   NSArray	*zodiacColors = colorWheel(0.5);

        for (i=0; i<12; i++)
        {
            path = [VPath path];
            [path setFillColor:[zodiacColors objectAtIndex:i]];
            [path setFilled:YES optimize:NO];
            list = [path list];
            arc = [VArc arc];
            p0 = vhfPointRotatedAroundCenter(NSMakePoint(center.x+radius1, center.y), angle+i*30.0,     center);
            [arc setCenter:center start:p0 angle:30.0];
            [list addObject:arc];
            p1 = vhfPointRotatedAroundCenter(NSMakePoint(center.x+radius2, center.y), angle+(i+1)*30.0, center);
            line = [VLine lineWithPoints:[arc pointWithNum:PT_END] :p1];
            [list addObject:line];
            arc = [VArc arc];
            [arc setCenter:center start:p1 angle:-30.0];
            [list addObject:arc];
            line = [VLine lineWithPoints:[arc pointWithNum:PT_END] :p0];
            [list addObject:line];
            [group addObject:path];
        }
    }
    else	// gray zodiac ring
    {
        path = [VPath path];
        [path setFillColor:GRAY_COLOR/*[NSColor lightGrayColor]*/];
        [path setFilled:YES optimize:NO];
        list = [path list];
        arc = [VArc arc];
        [arc setCenter:center start:NSMakePoint(center.x+radius1, center.y) angle:360.0];
        [list addObject:arc];
        arc = [VArc arc];
        [arc setCenter:center start:NSMakePoint(center.x+radius2, center.y) angle:360.0];
        [list addObject:arc];
        [group addObject:path];
    }

    /* black border */
    arc = [VArc arc];
    [arc setCenter:center start:NSMakePoint(center.x+radius1, center.y) angle:360.0];
    [group addObject:arc];
    arc = [VArc arc];
    [arc setCenter:center start:NSMakePoint(center.x+radius2, center.y) angle:360.0];
    [group addObject:arc];

    /* global houses (signs) */
    for (i=0; i<12; i++)
    {
        p0 = NSMakePoint(center.x+radius1, center.y);
        p1 = NSMakePoint(center.x+radius2, center.y);
        line = [VLine lineWithPoints:p0 :p1];
        [line setAngle:(i*30)-angle withCenter:center];
        [group addObject:line];

        /* signs */
        text = [VText text:zodiacSign(i)
                   center:NSMakePoint(p0.x-(radius1-radius2)/2.0, center.y)
                    color:[NSColor darkGrayColor] angle:angle+(i*30+15) center:center font:symbolFont];
        [text rotate:-((angle+i*30)-(90-15))];
        [group addObject:text];
    }


    /* black inner border */
    arc = [VArc arc];
    [arc setCenter:center start:NSMakePoint(center.x+radius2*RADIUS_4, center.y) angle:360.0];
    [group addObject:arc];

    return group;
}

/* scale, degrees */
typedef enum
{
    SCALEDEG_NONE = 0,	// no scale
    SCALEDEG_360  = 1,	// 360 degree
    SCALEDEG_DEC  = 2	// declination
}ScaleDegreeType;
- (VGroup*)scaleWithCenter:(NSPoint)center radius:(float)radius
                scaleSize1:(float)scaleSize1 scaleSize2:(float)scaleSize2
                degreeType:(ScaleDegreeType)degreeType rotate:(float)angle
{   int		i;
    VLine	*line;
    VGroup	*group = [VGroup group];
    NSPoint	p0, p1;
    NSFont	*systemFont8 = [NSFont systemFontOfSize:8.0];

    for (i=0; i<360; i++)
    {
        p0 = NSMakePoint(center.x+radius - ((i%5) ? scaleSize1 : scaleSize2), center.y);
        if (!degreeType && i%30 == 0)	// longer scales every 30 degrees without text
            p0.x -= scaleSize1;
        p1 = NSMakePoint(center.x+radius, center.y);
        line = [VLine lineWithPoints:p0 :p1];
        [line setAngle:angle-i withCenter:center];
        //[line setColor:color];
        [group addObject:line];

        if ( degreeType && !(i%30) )
        {   int	i1 = i;

            if (degreeType == SCALEDEG_DEC)
            {
                if (i1 > 180 && i1 <= 360)	// 3Q, 4Q -> 1Q, 2Q
                    i1 = 360 - i1;
                if (i1 > 90 && i1 <= 180)	// 2Q, (3Q, 4Q) -> 1Q
                    i1 = 180 - i1;
                i1 /= 3;			// display 10, 20, 30
                if (i > 180)			// 2Q, 3Q
                    i1 = -i1;
            }
            [group addObject:[VText text:[NSString stringWithFormat:@"%d", i1]
                                 center:NSMakePoint(p0.x-scaleSize1, p0.y)
                                  color:[NSColor blackColor]
                                  angle:angle+i center:center font:systemFont8]];
        }
    }
    return group;
}

/* local scale, degrees
 * we draw the local scale to the outside of the radius
 */
- (VGroup*)localScaleWithCenter:(NSPoint)center radius:(float)radius
                     scaleSize1:(float)scaleSize1 scaleSize2:(float)scaleSize2
                    showDegrees:(BOOL)showDegrees rotate:(float)angle
                            utc:(NSCalendarDate*)utc lat:(float)lat lon:(float)lon
{   int			i;
    VLine		*line;
    VGroup		*group = [VGroup group];
    NSPoint		p0, p1;
    TopocentricCycle	*cycle = [TopocentricCycle topocentricCycleAtDate:utc latitude:lat longitude:lon];
    double		*degrees = [cycle houses:360];
    NSFont		*systemFont8 = [NSFont systemFontOfSize:8.0];

    for (i=0; i<360; i++)
    {
        p0 = NSMakePoint(center.x+radius + ((i%5) ? scaleSize1 : scaleSize2), center.y);
        if (!showDegrees && i%30 == 0)	// longer scales every 30 degrees without text
            p0.x += scaleSize1;
        p1 = NSMakePoint(center.x+radius, center.y);
        line = [VLine lineWithPoints:p0 :p1];
        //[line setColor:[NSColor darkGrayColor]];
        [line setAngle:angle-degrees[i] withCenter:center];
        [group addObject:line];

        if ( showDegrees && !(i%30) )
            [group addObject:[VText text:[NSString stringWithFormat:@"%d", i]
                                 center:NSMakePoint(p0.x-scaleSize1, p0.y)
                                  color:[NSColor blackColor]
                                  angle:angle+degrees[i] center:center font:systemFont8]];
    }
    return group;
}

- (VText*)textWithSubstring:(NSString*)name fromTemplate:(NSArray*)list
{   unsigned	i;

    for (i=0; i<[list count]; i++)
    {   VText	*g = [list objectAtIndex:i];

        if ([g isKindOfClass:[VText class]])
        {   NSRange	range = [[g string] rangeOfString:name];

            if (range.length)
               return [[g copy] autorelease];
        }
    }
    //NSLog(@"Substring '%@' missing in Template", name);
    return nil;
}

/* house cusps
 * created: 2006-06-05 (extracted)
 */
- (void)addHousesAtCenter:(NSPoint)cp radius:(float)radius radiusCenter:(float)radCenter
                     date:(NSCalendarDate*)gmt lat:(float)lat lon:(float)lon
                 rotateAC:(float)rotateAC toLayer:(LayerObject*)layer
{   NSFont	*systemFont10 = [NSFont systemFontOfSize:10.0];
    int		houseCnt = 12, i;
    double	*house = [ephemeris houseAtUTC:gmt latitude:lat longitude:lon houseCount:houseCnt];
    NSArray	*houseColors = colorWheel(1.0);
    VGroup	*group = [VGroup group];
    VLine	*line;

    for ( i=1; i<=houseCnt; i++ )
    {   double		a = house[i];
        NSString	*string;
        NSPoint		p0, p1;

        if (rotateAC)
            a = standardAngle(a + rotateAC);

        p0 = NSMakePoint(cp.x + radius*((i==1 || i==4 || i==7 || i==10) ? RADIUS_2+SCALE_MIN : RADIUS_2),
                         cp.y);
        p1 = NSMakePoint(cp.x + radCenter, cp.y);

        line = [VLine lineWithPoints:p0 :p1];
        [line setAngle:-a withCenter:cp];
        if (Prefs_Colorful)
            [line setColor:[houseColors objectAtIndex:i-1]];
        if (i==1 || i==4 || i==7 || i==10)	// AC/MC thicker
            [line setWidth:0.7];
        else
            [line setWidth:0.51];
        [group addObject:line];

        if (i==1)		// AC
            string = @"AC ";
        else if (i==10)		// MC
            string = @"MC ";
        else
            string = [NSString stringWithFormat:@"%d ", i];
        [group addObject:[VText text:string
                              center:NSMakePoint(cp.x+(radius*RADIUS_3)+(radius*SCALE_MAX), cp.y+5.0)
                               color:[NSColor blackColor] angle:a center:cp font:systemFont10]];
    }
    [layer addObject:group];
}

/* modified: 2006-04-15 (aspect lines added)
 */
- (VGroup*)horoscopeWithCenter:(NSPoint)cp radius:(float)radius
                   tableOrigin:(NSPoint)tableOrigin
                      template:(NSArray*)template color:(NSColor*)color
                         title:(NSString*)title
                           lat:(float)latitude lon:(float)longitude city:(NSString*)city
                           gmt:(NSCalendarDate*)gmt
{   NSDictionary	*degreeDict, *nodeDict, *speedLonDict, *decDict, *speedDecDict;
    NSMutableArray	*planets = [NSMutableArray arrayWithObjects:@"AC", @"MC", nil];
    NSArray		*nodes = Prefs_Nodes;
    NSPoint		p0, p1;
    NSRect		r;
    NSMutableString	*mString = [NSMutableString string];
    NSString		*string;
    VGroup		*group = [VGroup group];
    VLine		*line;
    VText		*text;
    int			i, j, cnt;
    BOOL		retro = Prefs_ShowRetrograde;
    float		rotateAC = 0.0;
    NSFont		*systemFont10 = [NSFont systemFontOfSize:10.0];
    NSFont		*systemFont12 = [NSFont systemFontOfSize:12.0];
    NSFont		*systemFont20 = [NSFont systemFontOfSize:20.0];
    NSFont		*nodeFont   = [NSFont fontWithName:astroFont size:radius*SYMBOL_SIZE*0.8];
    NSFont		*planetFont = [NSFont fontWithName:astroFont size:radius*SYMBOL_SIZE];
    NSString		*dateFormat = [Prefs_DateFormat stringByAppendingFormat:@" %%H:%%M %%z"];

    [planets addObjectsFromArray:Prefs_Objects];

    [ephemeris setMeanNodes:Prefs_MeanNodes];
    degreeDict   = [ephemeris objectDictAtUTC:gmt lat:latitude lon:longitude elev:0.0
                                  topocentric:Prefs_Topocentric];
    if (Prefs_ACFix)
        rotateAC = 180.0 - [degreeDict floatForKey:@"AC"];	// rotation of chart
    nodeDict     = [ephemeris currentNodeDict];
    speedLonDict = [ephemeris currentSpeedLonDict];
    decDict      = [ephemeris currentDeclinationDict];
    speedDecDict = [ephemeris currentSpeedDecDict];

    /* background color */
    //[view setBackgroundColor:[NSColor colorWithCalibratedRed:1.0 green:1.0 blue:0.87 alpha:1.0]];
    [view setBackgroundColor:Prefs_BGColor];

    /* title */
    if (template)
    {
        if ( (text = [self textWithSubstring:@"#TITLE#" fromTemplate:template]) )
        {   [text replaceTextWithString:title];
            [group addObject:text];
        }
    }
    else
    {   text = [VText text:[title stringByAppendingString:@" "] origin:NSMakePoint(20.0, 790.0)
                    color:color angle:0.0 center:cp
               lineHeight:0.0 align:NSLeftTextAlignment font:systemFont20];
        [group addObject:text];
    }

    /* scale, planets and local houses
     */

    /* black border */
    /*arc = [VArc arc];
    [arc setCenter:cp start:NSMakePoint(cp.x+radius*RADIUS_4, cp.y) angle:360.0];
    [group addObject:arc];*/

    /* scale */
    /*[group addObject:[self scaleWithCenter:cp radius:radius*RADIUS_2
                                scaleSize1:radius*SCALE_MIN scaleSize2:radius*SCALE_MAX
                                degreeType:SCALEDEG_360]];*/

    /* local scale */
    if ( Prefs_LocalScale && chartTemplate.components & CHART_TOPOSCALE )
        [group addObject:[self localScaleWithCenter:cp radius:radius*RADIUS_2
                                         scaleSize1:radius*SCALE_MIN*0.5 scaleSize2:radius*SCALE_MAX*0.5
                                        showDegrees:NO rotate:rotateAC
                                                utc:gmt lat:latitude lon:longitude]];

    /* Objects */
    if ( chartTemplate.components & CHART_OBJECTS )
    {
        /* nodes */
        for ( i=0; i<(int)[nodes count]; i++ )
        {   NSString	*node = [nodes objectAtIndex:i];
            float		angle = [nodeDict floatForKey:node];
            
        //NSLog(@"%d %@ = %f", i, node, angle);
        //if (angle == 0.0)
        //    continue;
            if (rotateAC)
                angle = standardAngle(angle + rotateAC);
            p0 = NSMakePoint(cp.x + radius*RADIUS_2 - radius*(SCALE_MAX+SCALE_MIN), cp.y);
            p1 = NSMakePoint(cp.x + radius*RADIUS_2 + radius*SCALE_MIN, cp.y);
            line = [VLine lineWithPoints:p0 :p1];
            [line setAngle:-angle withCenter:cp];
            [line setColor:[NSColor darkGrayColor]];
            [group addObject:line];

            [group addObject:[VText text:planetSymbol(node) // nodeSymbols(node)
                                  center:NSMakePoint(p0.x-(radius*SYMBOL_SIZE*1.2), cp.y)
                                   color:[NSColor grayColor] angle:angle center:cp font:nodeFont]];
        }

        /* planets */
        for ( i=2; i<(int)[planets count]; i++ )
        {   NSString	*planet = [planets objectAtIndex:i];
            float		angle = [degreeDict floatForKey:planet];
            float		v = [speedLonDict floatForKey:planet];

            if (rotateAC)
                angle = standardAngle(angle + rotateAC);
            p0 = NSMakePoint(cp.x + radius*RADIUS_2 - radius*(SCALE_MAX+SCALE_MIN), cp.y);
            p1 = NSMakePoint(cp.x + radius*RADIUS_2 + radius*SCALE_MIN, cp.y);
            line = [VLine lineWithPoints:p0 :p1];
            [line setAngle:-angle withCenter:cp];
            [line setColor:color];
            [group addObject:line];

            string = (retro && v < 0.0) ? [NSString stringWithFormat:@"%@X", planetSymbol(planet)]
                                        : planetSymbol(planet);
            [group addObject:[VText text:string
                                  center:NSMakePoint(p0.x-(radius*SCALE_MAX), cp.y)
                                   color:color angle:angle center:cp font:planetFont]];
        }
    }

    /* show aspect lines */
    if ( Prefs_ShowAspectGeo && (chartTemplate.components & CHART_ASPECTLINES)	// enabled
         && !Prefs_ShowDeclination )	// not together with declination
    {   AspectLines	*aspectGeo = [AspectLines sharedInstance];
        id		g;
        int		objCnt = [planets count];
        double		objs[objCnt];

        for ( i=0; i<objCnt; i++ )		// AC, MC, Planets
        {   float	angle = [degreeDict floatForKey:[planets objectAtIndex:i]];

            if (rotateAC)
                angle = standardAngle(angle + rotateAC);
            objs[i] = angle;
        }
        [aspectGeo setObjects:objs cnt:objCnt];
        [aspectGeo setRadius:radius*RADIUS_3];
        [aspectGeo setStrengthOfOrb:YES];
        [aspectGeo setColoredDividers:Prefs_Colorful];
        [aspectGeo setDisplayMirrors:YES];
        //[aspectGeo setOrb:0.03];
        g = [aspectGeo calculateAtCenter:cp];	// aspect star (resonant)
        [group addObject:g];
    }

    /* planet declination */
    if ( Prefs_ShowDeclination && (chartTemplate.components & CHART_DECLINATION) )
    {   //BOOL	retro = NO;
        VArc	*arc = [VArc arc];

        [arc setCenter:cp start:NSMakePoint(cp.x+radius*RADIUS_3, cp.y) angle:360.0];
        [group addObject:arc];
        [group addObject: [self scaleWithCenter:cp radius:radius*RADIUS_3
                                     scaleSize1:radius*SCALE_MIN scaleSize2:radius*SCALE_MAX
                                     degreeType:SCALEDEG_DEC rotate:rotateAC]];
        for ( i=2; i<(int)[planets count]; i++ )
        {   NSString	*planet = [planets objectAtIndex:i];
            float	dec  = [decDict      floatForKey:planet] * 3.0;
            float	lon  = [degreeDict   floatForKey:planet];
            float	vDec = [speedDecDict floatForKey:planet];
            //float	vLon = [speedLonDict floatForKey:planet];
            //BOOL	flip = (vDec * vLon < 0.0) ? YES : NO;	// (vDec < 0.0 && vLon >= 0.0);
            BOOL	flip = ( lon >= 90.0 && lon <= 270.0 ) ? YES : NO;

            /* we place the declination symbol on the side of the longitude.
             * We add an R to the declination, if the direction is counter the direction of the side
             */
            if (flip)	// display retrograde on the other side
                dec = mod360(180.0 - dec);
            //if (rotateAC)
            //    dec = standardAngle(dec + rotateAC);
            p0 = NSMakePoint(cp.x + radius*RADIUS_3 - radius*(SCALE_MAX+SCALE_MIN), cp.y);
            p1 = NSMakePoint(cp.x + radius*RADIUS_3 /*+ radius*SCALE_MIN*/, cp.y);
            line = [VLine lineWithPoints:p0 :p1];
            [line setAngle:-dec withCenter:cp];
            [line setColor:color];
            [group addObject:line];

            /* retrograde means against the regular direction */
            string = ( ((vDec < 0.0 && !flip) || (vDec > 0.0 && flip)) )
                     ? [NSString stringWithFormat:@"%@X", planetSymbol(planet)]
                     : planetSymbol(planet);
            [group addObject:[VText text:string
                                 center:NSMakePoint(p0.x-(radius*SCALE_MAX), cp.y)
                                  color:color angle:dec center:cp font:planetFont]];
        }
    }

    /* AC / DC */
    /*{
        ac = [self ac:sideralTime latitude:latitude];
        p0 = NSMakePoint(cp.x+radius*RADIUS_2, cp.y);
        p1 = NSMakePoint(cp.x+radius*RADIUS_4, cp.y);
        line = [VLine lineWithPoints:p0 :p1];
        [line setAngle:-ac withCenter:cp];
        [line setWidth:0.5];
        [line setColor:color];
        [group addObject:line];
        [group addObject:[VText text:@"AC " center:NSMakePoint(cp.x+(radius*RADIUS_3)+(radius*SCALE_MAX), cp.y)
                               color:[NSColor blackColor] angle:ac center:cp font:systemFont10]];
    }*/
    /* MC / IC */
    /*{
        mc = [self mc:sideralTime];
        p0 = NSMakePoint(cp.x+radius*RADIUS_2, cp.y);
        p1 = NSMakePoint(cp.x+radius*RADIUS_4, cp.y);
        line = [VLine lineWithPoints:p0 :p1];
        [line setAngle:-mc withCenter:cp];
        [line setWidth:0.5];
        [line setColor:color];
        [group addObject:line];
        [group addObject:[VText text:@"MC " center:NSMakePoint(cp.x+(radius*RADIUS_3)+(radius*SCALE_MAX), cp.y)
                               color:[NSColor blackColor] angle:mc center:cp font:systemFont10]];
    }*/

    /* house cusps */
    if (chartTemplate.components & CHART_HOUSES)
    {   int	houseCnt = 12;
        double	*house = [ephemeris houseAtUTC:gmt latitude:latitude longitude:longitude houseCount:houseCnt];
        NSArray	*houseColors = colorWheel(1.0);

        for ( i=1; i<=houseCnt; i++ )
        {   double	a = house[i];
            float	radCenter = radius*RADIUS_4;
            NSString	*string;

            if (rotateAC)
                a = standardAngle(a + rotateAC);
            if ( Prefs_ShowAspectGeo || Prefs_ShowDeclination )
                radCenter = radius*RADIUS_3;	// shorter with aspect lines or Declination

            p0 = NSMakePoint(cp.x + radius*((i==1 || i==4 || i==7 || i==10) ? RADIUS_2+SCALE_MIN : RADIUS_2),
                             cp.y);
            p1 = NSMakePoint(cp.x + radCenter, cp.y);

            line = [VLine lineWithPoints:p0 :p1];
            [line setAngle:-a withCenter:cp];
            if (Prefs_Colorful)
                [line setColor:[houseColors objectAtIndex:i-1]];
            if (i==1 || i==4 || i==7 || i==10)	// AC/MC thicker
                [line setWidth:0.7];
            else
                [line setWidth:0.51];
            [group addObject:line];

            if (i==1)		// AC
                string = @"AC ";
            else if (i==10)	// MC
                string = @"MC ";
            else
                string = [NSString stringWithFormat:@"%d ", i];
            [group addObject:[VText text:string
                                 center:NSMakePoint(cp.x+(radius*RADIUS_3)+(radius*SCALE_MAX), cp.y+5.0)
                                  color:[NSColor blackColor] angle:a center:cp font:systemFont10]];
        }
    }

    /* title, date, city, lat, lng
     */
    if (template && (text = [self textWithSubstring:@"#NAME#" fromTemplate:template]))
    {   NSMutableString	*string = [[[text string] mutableCopy] autorelease];
        NSRange		range;
        NSString	*ns, *ew;

        range = [string rangeOfString:@"#NAME#"];
        [string replaceCharactersInRange:range withString:title];
        range = [string rangeOfString:@"#DATE#"];
        [string replaceCharactersInRange:range
                              withString:[gmt descriptionWithCalendarFormat:dateFormat]];
        range = [string rangeOfString:@"#LOCATION#"];
        ns = (latitude  >= 0.0) ? @"N" : @"S";
        ew = (longitude >= 0.0) ? @"E" : @"W";
        if (Prefs_DegreesMinutes)
            [string replaceCharactersInRange:range
                                  withString:[NSString stringWithFormat:@"%@ %@ %@ %@ %@",
                                      city, stringFromDeg(Abs(latitude)),  ns,
                                            stringFromDeg(Abs(longitude)), ew]];
        else
            [string replaceCharactersInRange:range
                                  withString:[NSString stringWithFormat:@"%@ %.3f %@, %.3f %@",
                                      city, Abs(latitude), ns, Abs(longitude), ew]];
        [text replaceTextWithString:string];
    }
    else
    {   text = [VText text:[NSString stringWithFormat:@"%@ \n%@ \n%@ %.3f N, %.3f E ", title,
                          [gmt descriptionWithCalendarFormat:dateFormat], city, latitude, longitude]
                   origin:NSMakePoint(20.0, cp.y - radius*RADIUS_1) color:color angle:0.0
                   center:cp lineHeight:0.0 align:NSLeftTextAlignment font:systemFont12];
        r = [text bounds];
        [text moveTo:NSMakePoint(r.origin.x, r.origin.y - r.size.height)];
    }
    [group addObject:text];

    /*
     * table with angles and mirrors
     */
    if ( DiffPoint(tableOrigin, NSZeroPoint) > 0.0 )	// aspect symbols
    {   VGroup		*subGroup = [VGroup group];
        float		x = tableOrigin.x, y = tableOrigin.y, w, h;
        float		orb = 6.0, cellSize = LINE_HEIGHT + 2.0;
        NSFont		*symbolFont = [NSFont fontWithName:astroFont size:FONT_SIZE];
        //VRectangle	*rectangle;

        /* table: degrees of planets */
        {
            /* symbols vertical (1st leading column) */
            /*[mString setString:@""];
            for ( i=0, cnt=(int)[planets count]; i<cnt; i++ )
                [mString appendFormat:@"%@%@", planetSymbol([planets objectAtIndex:i]), (i<cnt-1) ? @"\n" : @""];
            text = [VText text:mString base:NSMakePoint(x, y) color:color angle:0.0
                       center:cp lineHeight:cellSize font:symbolFont];
            r = [text bounds];
#ifdef __APPLE__
            if (r.size.height > cnt * cellSize + 2)	// workaround Apple -> reduce lineHeight
            {   text = [VText text:mString base:NSMakePoint(x, y) color:color angle:0.0
                           center:cp lineHeight:cellSize-1 font:symbolFont];
                r = [text bounds];
            }
#endif
            [subGroup addObject:text];
            [text moveTo:NSMakePoint(r.origin.x + (cellSize - r.size.width)/2.0, y)];*/

            /* degrees */
            [mString setString:@""];
            for ( i=0, cnt=(int)[planets count]; i<cnt; i++ )
            {   NSString	*planet = [planets objectAtIndex:i], *string;
                float		deg = [degreeDict floatForKey:planet];
                float		v = [speedLonDict floatForKey:planet];

                string = (Prefs_DegreesMinutes) ? stringFromDeg(deg)
                                                : [[NSString stringWithFormat:@"%.5f", deg] stringWithLength:-9];
                string = [NSString stringWithFormat:@"%@ %@%@", string, (v < 0) ? @"R" : @" ",
                                                                (i<cnt-1) ? @"\n" : @""];
                [mString appendString:string];
            }
            text = [VText text:mString origin:NSMakePoint(x /*+ cellSize*/ + 3.0, y) color:color angle:0.0
                       center:cp lineHeight:cellSize align:NSLeftTextAlignment
                         font:[NSFont userFixedPitchFontOfSize:FONT_SIZE]];
            r = [text bounds];
#ifdef __APPLE__
            if (r.size.height > cnt * cellSize + 2)	// workaround Apple -> reduce lineHeight
                text = [VText text:mString origin:NSMakePoint(x /*+ cellSize*/ + 3.0, y) color:color angle:0.0
                           center:cp lineHeight:cellSize-1 align:NSLeftTextAlignment
                             font:[NSFont userFixedPitchFontOfSize:FONT_SIZE]];
#endif
            [subGroup addObject:text];
            w = [text bounds].origin.x - x + [text bounds].size.width + 3.0;

            /* table grid */
            for (i=0, cnt=(int)[planets count]; i<=cnt; i++ )
            {
                /* horicontal lines */
                p0 = NSMakePoint(x, y + i * cellSize);
                p1 = NSMakePoint(x + w, p0.y);
                line = [VLine lineWithPoints:p0 :p1];
                [subGroup addObject:line];
            }
            /* vertical lines */
            p0 = NSMakePoint(x, y);
            p1 = NSMakePoint(x, y + [planets count] * cellSize);
            line = [VLine lineWithPoints:p0 :p1];
            [subGroup addObject:line];
            /*p0.x = p1.x = x + cellSize;	// line for leading symbols
            line = [VLine lineWithPoints:p0 :p1];
            [subGroup addObject:line];*/
            p0.x = p1.x = x + w;
            line = [VLine lineWithPoints:p0 :p1];
            [subGroup addObject:line];

            x += w;
        }

        /* symbols vertical (2nd column) */
        [mString setString:@""];
        for ( i=0, cnt=(int)[planets count]; i<cnt; i++ )
            [mString appendFormat:@"%@%@", planetSymbol([planets objectAtIndex:i]), (i<cnt-1) ? @"\n" : @""];
        text = [VText text:mString base:NSMakePoint(x, y) color:color angle:0.0
                   center:cp lineHeight:cellSize font:symbolFont];
        r = [text bounds];
#ifdef __APPLE__
        if (r.size.height > cnt * cellSize + 2)	// workaround Apple -> reduce lineHeight
        {   text = [VText text:mString base:NSMakePoint(x, y) color:color angle:0.0
                       center:cp lineHeight:cellSize-1 font:symbolFont];
            r = [text bounds];
        }
#endif
        [text moveTo:NSMakePoint(r.origin.x + (cellSize - r.size.width)/2.0, y)];

        /* symbols background (vertical) */
        /*rectangle = [VRectangle rectangleWithOrigin:NSMakePoint(x, y)
                                              size:NSMakeSize(cellSize, [planets count]*LINE_HEIGHT)];
        [rectangle setColor:GRAY_COLOR];
        [rectangle setFilled:YES];
        [subGroup addObject:rectangle];*/

        [subGroup addObject:text];

        /* symbol background (horicontal) */
        /*rectangle = [VRectangle rectangleWithOrigin:NSMakePoint(x, y + [planets count]*cellSize)
                                              size:NSMakeSize(cellSize + [planets count]*cellSize, cellSize)];
        [rectangle setColor:GRAY_COLOR];
        [rectangle setFilled:YES];
        [subGroup addObject:rectangle];*/

        /* background of mirrors */
        /*for (i=1; i<[planets count]; i++ )
        {   Rectangle	*rectangle;

            p0 = NSMakePoint(x + cellSize + ([planets count]-i)*TAB_SPACING, y + i*cellSize);
            rectangle = [VRectangle rectangleWithOrigin:p0 size:NSMakeSize(i * cellSize, cellSize)];
            [rectangle setColor:GRAY_COLOR];
            [rectangle setFilled:YES];
            [subGroup addObject:rectangle];
        }*/

        /* aspect symbols */
        w = cellSize;			// width of vertical symbols (cellSize)
        h = cellSize * [planets count];	// height of all planet entries
        for (i=0, cnt=(int)[planets count]; i<cnt; i++ )	// rows
        {   float	deg1 = [degreeDict floatForKey:[planets objectAtIndex:i]];

            /* symbols horicontal (one by one) */
            mString = [NSMutableString string];
            [mString appendFormat:@"%@", planetSymbol([planets objectAtIndex:i])];
            text = [VText text:mString
                       center:NSMakePoint(x + w + i*cellSize+cellSize/2.0, y + h + cellSize/2.0)
                        color:color angle:0.0 center:cp lineHeight:LINE_HEIGHT font:symbolFont];
            [subGroup addObject:text];

            /* table columns */
            //[mString setString:@""];
            for (j=0, cnt=(int)[planets count]; j<cnt; j++)	// columns
            {   float	deg2 = [degreeDict floatForKey:[planets objectAtIndex:j]];
                NSColor	*orbCol;

                if (i < j)		// angles
                {
                    if ( !(string = aspectSymbol(angle(deg1, deg2), orb)) )
                        continue;
                    orbCol = orbColor(angle(deg1, deg2), 2.0, 4.0);
                    //[mString appendFormat:@"%@", aspectSymbol(angle(deg1, deg2), orb)];
                    //if (j < [planets count]-1)	// last entry without line break
                    //    [mString appendString:@"\n"];
                }
                else if (i > j)		// mirrors
                {
                    if ( !(string = aspectSymbol(mirror(deg1, deg2), orb)) )
                        continue;
                    orbCol = orbColor(mirror(deg1, deg2), 2.0, 4.0);
                    //[mString appendFormat:@"%@\n", aspectSymbol(mirror(deg1, deg2), orb)];
                }
                else			// empty
                    continue;
                //else if (i != 0)	// empty
                    //[mString appendString:@" \n"];
                text = [VText text:string
                           center:NSMakePoint(x + w + i*cellSize+cellSize/2.0, y + h - j*cellSize - cellSize/2.0)
                            color:orbCol angle:0.0 center:cp lineHeight:cellSize font:symbolFont];
                [subGroup addObject:text];
            }
            /*text = [VText text:mString
                         base:NSMakePoint(x + w + i*cellSize+cellSize/2.0, y)
                        color:color angle:0.0 center:cp size:FONT_SIZE lineHeight:cellSize fontName:astroFont];
            r = [text bounds];
            [text moveTo:NSMakePoint(r.origin.x - r.size.width/2.0, y)];
            [subGroup addObject:text];*/
        }

        /* table grid */
        for (i=0; i<=(int)[planets count] /*+1*/; i++ )
        {
            /* horicontal lines */
            p0 = NSMakePoint(x, y + i * cellSize);
            p1 = NSMakePoint(x + w + [planets count] * cellSize, p0.y);
            line = [VLine lineWithPoints:p0 :p1];
            [subGroup addObject:line];

            /* vertical lines */
            p0 = NSMakePoint(x + ((i <= (int)[planets count]) ? (w+i*cellSize) : 0.0), y);
            p1 = NSMakePoint(p0.x, y + ([planets count]+1) * cellSize);
            line = [VLine lineWithPoints:p0 :p1];
            [subGroup addObject:line];
        }
        /* diagonal seperating line */
        p0 = NSMakePoint(x + w, y + [planets count] * cellSize);
        p1 = NSMakePoint(p0.x + [planets count] * cellSize, y);
        line = [VLine lineWithPoints:p0 :p1];
        [subGroup addObject:line];
        [group addObject:subGroup];
    }
    else if ( DiffPoint(tableOrigin, NSZeroPoint) > 0.0 )	// aspects in degrees
    {   float		w, h;
        NSFont		*symbolFont = [NSFont fontWithName:astroFont size:FONT_SIZE];
        //Rectangle	*rectangle;

        /* symbols vertical */
        mString = [NSMutableString string];
        for ( i=0; i<(int)[planets count]; i++ )
            [mString appendFormat:@"%@\n", planetSymbol([planets objectAtIndex:i])];
        text = [VText text:mString base:NSMakePoint(20.0+5.0, 645.0) color:color angle:0.0
                   center:cp lineHeight:LINE_HEIGHT font:symbolFont];
        w = [text bounds].size.width;	// width of vertical symbols
        h = [text bounds].size.height;	// height of all planet entries

        /* symbols background (vertical) */
        /*rectangle = [VRectangle rectangleWithOrigin:NSMakePoint(20.0, 645.0)
                                              size:NSMakeSize(w + 10.0, [planets count]*LINE_HEIGHT)];
        [rectangle setFillColor:GRAY_COLOR];
        [rectangle setFilled:YES];
        [group addObject:rectangle];*/

        [group addObject:text];

        /* symbol background (horicontal) */
        /*rectangle = [VRectangle rectangleWithOrigin:NSMakePoint(20.0, 645.0 + [planets count]*LINE_HEIGHT)
                                              size:NSMakeSize(w+10.0 + [planets count]*TAB_SPACING, LINE_HEIGHT)];
        [rectangle setFillColor:GRAY_COLOR];
        [rectangle setFilled:YES];
        [group addObject:rectangle];*/

        /* background of mirrors */
        for (i=1; i<(int)[planets count]; i++ )
        {   VRectangle	*rectangle;

            p0 = NSMakePoint(20.0+w+10 + ([planets count]-i)*TAB_SPACING, 645.0 + i*LINE_HEIGHT);
            rectangle = [VRectangle rectangleWithOrigin:p0 size:NSMakeSize(i * TAB_SPACING, LINE_HEIGHT)];
            [rectangle setFillColor:GRAY_COLOR];
            [rectangle setFilled:YES];
            [group addObject:rectangle];
        }

        for (i=0; i<(int)[planets count]; i++ )	// rows
        {   float	deg1 = [degreeDict floatForKey:[planets objectAtIndex:i]];

            /* symbols horicontal (one by one) */
            mString = [NSMutableString string];
            [mString appendFormat:@"%@", planetSymbol([planets objectAtIndex:i])];
            text = [VText text:mString base:NSMakePoint(20.0+w+20.0+i*TAB_SPACING, 645.0 + h)
                        color:color angle:0.0 center:cp lineHeight:LINE_HEIGHT font:symbolFont];
            [group addObject:text];

            /* table columns */
            [mString setString:@""];
            for (j=0; j<(int)[planets count]; j++)	// columns
            {   float	deg2 = [degreeDict floatForKey:[planets objectAtIndex:j]];

                if (i < j)		// angles
                {
                    [mString appendFormat:@"%@ ", formAngle(deg1, deg2)];
                    if (j < (int)[planets count]-1)	// last entry without line break
                        [mString appendString:@"\n"];
                }
                else if (i > j)		// mirrors
                    [mString appendFormat:@"%@ \n", formMirror(deg1, deg2)];
                else if (i != 0)	// empty
                    [mString appendString:@" \n"];
            }
            [group addObject:[VText text:mString base:NSMakePoint(20.0+w+10+i*TAB_SPACING, 645.0)
                                  color:color angle:0.0 center:cp lineHeight:LINE_HEIGHT
                                   font:[NSFont userFixedPitchFontOfSize:FONT_SIZE]
                                        /*[NSFont fontWithName:@"Courier" size:FONT_SIZE]*/]];
        }

        /* table grid */
        for (i=0; i<=(int)[planets count] /*+1*/; i++ )
        {
            /* horicontal lines */
            p0 = NSMakePoint(20.0, 645.0 + i * LINE_HEIGHT);
            p1 = NSMakePoint(20.0 + w + 10.0 + [planets count] * TAB_SPACING, p0.y);
            line = [VLine lineWithPoints:p0 :p1];
            [group addObject:line];

            /* vertical lines */
            p0 = NSMakePoint(20.0 + ((i <= (int)[planets count]) ? (w+10.0+i*TAB_SPACING) : 0.0), 645.0);
            p1 = NSMakePoint(p0.x, 645.0 + ([planets count]+1) * LINE_HEIGHT);
            line = [VLine lineWithPoints:p0 :p1];
            [group addObject:line];
        }

        /*
         * table: degrees of planets
         */
        {
            /* symbols */
            [mString setString:@""];
            for ( i=0; i<(int)[planets count]; i++ )
                [mString appendFormat:@"%@\n", planetSymbol([planets objectAtIndex:i])];
            [group addObject:[VText text:mString base:NSMakePoint(20.0, 470.0) color:color angle:0.0
                                 center:cp lineHeight:LINE_HEIGHT font:symbolFont]];

            /* coordinates */
            [mString setString:@""];
            for ( i=0; i<(int)[planets count]; i++ )
            {   float	deg = [degreeDict floatForKey:[planets objectAtIndex:i]];

                [mString appendString:[[NSString stringWithFormat:@"%.5f\n", deg] stringWithLength:-10]];
            }
            //[mString appendString:@" \n"];
            [group addObject:[VText text:mString base:NSMakePoint(45.0, 470.0) color:color angle:0.0
                                 center:cp lineHeight:LINE_HEIGHT
                                   font:[NSFont userFixedPitchFontOfSize:FONT_SIZE]]];
        }
    }

//NSLog(@"%f %@", longitude, [sideralTime description]);
    return group;
}

/* draw composite chart inside chart
 * radius is the ouiter radius of the chart
 */
- (VGroup*)compositeWithCenter:(NSPoint)cp radius:(float)radius
                      template:(NSArray*)template  color:(NSColor*)color title:(NSString*)title
                  lat:(float)latitude lon:(float)longitude city:(NSString*)city gmt:(NSCalendarDate*)gmt
{   NSDictionary	*degreeDict;
    NSArray		*planets = Prefs_Objects;
    NSPoint		p0, p1;
    VGroup		*group = [VGroup group];
    VLine		*line;
    int			i;
    float		rotateAC = 0.0;
    NSColor		*planetColor = [NSColor colorWithDeviceRed:0.0 green:0.0 blue:0.0 alpha:1.0];
    NSFont		*systemFont10 = [NSFont systemFontOfSize:10.0];
    NSFont		*systemFont12 = [NSFont systemFontOfSize:12.0];
    NSFont		*planetFont = [NSFont fontWithName:astroFont size:radius*SYMBOL_SIZE];
    NSString		*dateFormat = [Prefs_DateFormat stringByAppendingFormat:@" %%H:%%M %%z"];

    degreeDict = [ephemeris objectDictAtUTC:gmt];

    /* house cusps */
    {   int	houseCnt = 12;
        double	*house = [ephemeris houseAtUTC:gmt latitude:latitude longitude:longitude houseCount:houseCnt];

        if (Prefs_ACFix)
            rotateAC = 180.0 - house[1];	// rotation of chart

        for ( i=1; i<=houseCnt; i++ )
        {   double	a = house[i];
            NSString	*string;

            if (rotateAC)
                a = standardAngle(a + rotateAC);

            p0 = NSMakePoint(cp.x + radius*RADIUS_3, cp.y);
            p1 = NSMakePoint(cp.x + radius*RADIUS_4, cp.y);

            line = [VLine lineWithPoints:p0 :p1];
            [line setAngle:-a withCenter:cp];
            if (i==1 || i==4 || i==7 || i==10)	// AC/MC thicker
                [line setWidth:0.5];
            [group addObject:line];

            if (i==1)	// AC
            {
                string = @"AC ";
                //ac = a;
            }
            else if (i==10)	// MC
            {
                string = @"MC ";
                //mc = a;
            }
            else
                string = [NSString stringWithFormat:@"%d ", i];
            [group addObject:[VText text:string
                                 center:NSMakePoint(cp.x+(radius*RADIUS_4)+(radius*SCALE_MAX), cp.y+5.0)
                                  color:[NSColor blackColor] angle:a center:cp font:systemFont10]];
        }
    }

    /* planets */
    for ( i=0; i<(int)[planets count]; i++ )
    {   NSString	*planet = [planets objectAtIndex:i];
        float		angle = [degreeDict floatForKey:planet];

        if (rotateAC)
            angle = standardAngle(angle + rotateAC);
        p0 = NSMakePoint(cp.x + radius*RADIUS_3 - radius*(SCALE_MAX+SCALE_MIN), cp.y);
        p1 = NSMakePoint(cp.x + radius*RADIUS_3 + radius*SCALE_MIN, cp.y);
        line = [VLine lineWithPoints:p0 :p1];
        [line setAngle:-angle withCenter:cp];
        [group addObject:line];

        [group addObject:[VText text:planetSymbol(planet) center:NSMakePoint(p0.x-(radius*SCALE_MAX), cp.y)
                              color:planetColor angle:angle center:cp font:planetFont]];
    }

    /* date, lat, lng */
    {   VText		*text;
        VRectangle	*rectangle;
        NSRect		r;
        float		viewWidth = [view bounds].size.width;

        if (template && (text = [self textWithSubstring:@"#NAME_C#" fromTemplate:template]))
        {   NSMutableString	*string = [[[text string] mutableCopy] autorelease];
            NSRange		range;
            NSString		*ns, *ew;

            range = [string rangeOfString:@"#NAME_C#"];
            [string replaceCharactersInRange:range withString:title];
            range = [string rangeOfString:@"#DATE_C#"];
            [string replaceCharactersInRange:range
                                  withString:[gmt descriptionWithCalendarFormat:dateFormat]];
            range = [string rangeOfString:@"#LOCATION_C#"];
            ns = (latitude  >= 0.0) ? @"N" : @"S";
            ew = (longitude >= 0.0) ? @"E" : @"W";
            if (Prefs_DegreesMinutes)
                [string replaceCharactersInRange:range
                                      withString:[NSString stringWithFormat:@"%@ %@ %@ %@ %@",
                                      city, stringFromDeg(Abs(latitude)),  ns,
                                            stringFromDeg(Abs(longitude)), ew]];
            else
                [string replaceCharactersInRange:range
                                      withString:[NSString stringWithFormat:@"%@ %.3f N, %.3f E",
                                          city, latitude, longitude]];
            [text replaceTextWithString:string];
        }
        else
        {   text = [VText text:[NSString stringWithFormat:@"%@ \n%@ \n%@ %.3f N, %.3f E ", title, 
                              [gmt descriptionWithCalendarFormat:dateFormat], city, latitude, longitude]
                       origin:NSMakePoint(cp.x+viewWidth/2.0-20.0, cp.y - radius*RADIUS_1)
                        color:color angle:0.0 center:cp
                   lineHeight:0.0 align:NSRightTextAlignment font:systemFont12];
            r = [text bounds];
            [text moveTo:NSMakePoint(r.origin.x, r.origin.y - r.size.height)];
        }

        /* background */
        if (ixMaskComposite < 0)
        {
            r = [text bounds];
            rectangle = [VRectangle rectangleWithOrigin:r.origin size:r.size];
            [rectangle setFillColor:GRAY_COLOR];
            [rectangle setFilled:YES];
            [group addObject:rectangle];
        }
        [group addObject:text];
    }

    return group;
}

/* Tansite aspect table between horoscope and composite
 * created:  2005-07-03
 * modified: 2005-09-19 (show symbol row on the left for symmetry)
 * Author:  Ilonka Fleischmann
 */
- (VGroup*)compositeTable:(NSPoint)cp tableOrigin:(NSPoint)tableOrigin
                     latH:(float)latitudeH lonH:(float)longitudeH gmtH:(NSCalendarDate*)gmtH
                     latC:(float)latitudeC lonC:(float)longitudeC gmtC:(NSCalendarDate*)gmtC
{   NSDictionary	*degreeDictH, *degreeDictC, *speedLonDict;
    NSMutableArray	*planets  = [NSMutableArray arrayWithObjects:@"AC", @"MC", nil];
    NSMutableArray	*planetsH = [NSMutableArray array];
    NSMutableArray	*planetsC = [NSMutableArray array];
    //NSPoint		tableOrigin = NSMakePoint(312.0, 37.0);
    NSMutableString	*mString = [NSMutableString string];
    NSString		*string;
    NSColor		*color = [NSColor blackColor];
    NSPoint		p0, p1;
    NSRect		r;
    VLine		*line;
    VText		*text;
    int			i, j, cnt;
    BOOL		showCombiAspects = NO;	// YES = show inner/outer + outer/inner
    BOOL		show2ndSymbolRow = NO;	// YES displays 2nd row of symbols to the right of degrees

    speedLonDict = [ephemeris currentSpeedLonDict];

    [planets  addObjectsFromArray:Prefs_Objects];	// AC, MC, Planets
    [planetsH addObjectsFromArray:Prefs_Objects];	// Planets 1 (outside)
    [planetsC addObjectsFromArray:Prefs_Objects];	// Planets 2 (inside)

    degreeDictH = [ephemeris objectDictAtUTC:gmtH lat:latitudeH lon:longitudeH elev:0.0 topocentric:Prefs_Topocentric];
    degreeDictC = [ephemeris objectDictAtUTC:gmtC lat:latitudeC lon:longitudeC elev:0.0 topocentric:Prefs_Topocentric];

    /*
     * Tansite aspect table between horoscope and composite
     */

    {   VGroup		*subGroup = [VGroup group];
        //float		x = tableOrigin.x, y = tableOrigin.y, w, h;
        float		x = 0.0 , y = tableOrigin.y, w, h;
        float		orb = 1.5, cellSize = LINE_HEIGHT + 2.0;
        NSFont		*symbolFont     = [NSFont fontWithName:astroFont size:FONT_SIZE];
        NSFont		*symbolFontMini = [NSFont fontWithName:astroFont size:FONT_SIZE-2];

        /* table: degrees of planets */
        {
            /* degrees */
            [mString setString:@""];
            for ( i=0, cnt=(int)[planets count]; i<cnt; i++ )
            {   NSString	*planet = [planets objectAtIndex:i], *string;
                float		deg = [degreeDictC floatForKey:planet];
                float		v = [speedLonDict floatForKey:planet];

                string = (Prefs_DegreesMinutes) ? stringFromDeg(deg)
                                                : [[NSString stringWithFormat:@"%.5f", deg] stringWithLength:-9];
                string = [NSString stringWithFormat:@"%@ %@%@", string, (v < 0) ? @"R" : @" ",
                                                                (i<cnt-1) ? @"\n" : @""];
                [mString appendString:string];
            }
            text = [VText text:mString origin:NSMakePoint(x + 3.0, y) color:color angle:0.0
                       center:cp lineHeight:cellSize align:NSLeftTextAlignment
                         font:[NSFont userFixedPitchFontOfSize:FONT_SIZE]];
            r = [text bounds];
#ifdef __APPLE__
            if (r.size.height > cnt * cellSize + 2)	// workaround Apple -> reduce lineHeight
                text = [VText text:mString origin:NSMakePoint(x + 3.0, y) color:color angle:0.0
                           center:cp lineHeight:cellSize-1 align:NSLeftTextAlignment
                             font:[NSFont userFixedPitchFontOfSize:FONT_SIZE]];
#endif
            /* correct x to lower left of table */
            if (showCombiAspects)	// show aspect table
                x = tableOrigin.x - ([planetsH count]*cellSize + cellSize + [text bounds].size.width + 6.0);
            else			// show degrees only
            {   x = tableOrigin.x - ([text bounds].size.width + 6.0);
                if ( show2ndSymbolRow )	// room for 2nd row of symbols
                    x -= cellSize;
            }

            /* move text to real origin */
            [text moveBy:NSMakePoint((x + 3.0) - r.origin.x, 0.0)];

            [subGroup addObject:text];
            w = [text bounds].origin.x - x + [text bounds].size.width + 3.0;

            /* table grid */
            for (i=0, cnt=(int)[planets count]; i<=cnt; i++ )
            {
                /* horicontal lines */
                p0 = NSMakePoint(x, y + i * cellSize);
                p1 = NSMakePoint(x + w, p0.y);
                line = [VLine lineWithPoints:p0 :p1];
                [subGroup addObject:line];
            }
            /* vertical lines */
            p0 = NSMakePoint(x, y);
            p1 = NSMakePoint(x, y + [planets count] * cellSize);
            line = [VLine lineWithPoints:p0 :p1];
            [subGroup addObject:line];
            p0.x = p1.x = x + w;
            line = [VLine lineWithPoints:p0 :p1];
            [subGroup addObject:line];

            x += w;
        }

        /* symbols vertical (1st leading column) */
        if ( !showCombiAspects )
        {
            x -= w + cellSize;
            [mString setString:@""];
            for ( i=0, cnt=(int)[planets count]; i<cnt; i++ )
                [mString appendFormat:@"%@%@", planetSymbol([planets objectAtIndex:i]), (i<cnt-1) ? @"\n" : @""];
            text = [VText text:mString base:NSMakePoint(x, y) color:color angle:0.0
                       center:cp lineHeight:cellSize font:symbolFont];
            r = [text bounds];
#ifdef __APPLE__
            if (r.size.height > cnt * cellSize + 2)	// workaround Apple -> reduce lineHeight
            {   text = [VText text:mString base:NSMakePoint(x, y) color:color angle:0.0
                           center:cp lineHeight:cellSize-1 font:symbolFont];
                r = [text bounds];
            }
#endif
            [subGroup addObject:text];
            [text moveTo:NSMakePoint(r.origin.x + (cellSize - r.size.width)/2.0, y)];

            /* horicontal lines for planet symbols */
            for (i=0; i<=(int)[planets count] /*+1*/; i++ )
            {
                p0 = NSMakePoint(x, y + i * cellSize);
                p1 = NSMakePoint(x + cellSize, p0.y);
                line = [VLine lineWithPoints:p0 :p1];
                [subGroup addObject:line];
            }
            /* vertical line for planet symbols */
            p0 = NSMakePoint(x, y);
            p1 = NSMakePoint(p0.x, y + [planets count] * cellSize);
            line = [VLine lineWithPoints:p0 :p1];
            [subGroup addObject:line];

            x += w + cellSize;
        }
        if ( showCombiAspects || show2ndSymbolRow )
        {
            /* composite symbols vertical (2nd column) */
            [mString setString:@""];
            for ( i=0, cnt=(int)[planets count]; i<cnt; i++ )
                [mString appendFormat:@"%@%@", planetSymbol([planets objectAtIndex:i]), (i<cnt-1) ? @"\n" : @""];
            text = [VText text:mString base:NSMakePoint(x, y) color:color angle:0.0
                       center:cp lineHeight:cellSize font:symbolFont];
            r = [text bounds];
#ifdef __APPLE__
            if (r.size.height > cnt * cellSize + 2)	// workaround Apple -> reduce lineHeight
            {   text = [VText text:mString base:NSMakePoint(x, y) color:color angle:0.0
                           center:cp lineHeight:cellSize-1 font:symbolFont];
                r = [text bounds];
            }
#endif
            [text moveTo:NSMakePoint(r.origin.x + (cellSize - r.size.width)/2.0, y)];
            [subGroup addObject:text];

            /* horicontal lines for planet symbols */
            for (i=0; i<=(int)[planets count] /*+1*/; i++ )
            {
                p0 = NSMakePoint(x, y + i * cellSize);
                p1 = NSMakePoint(x + cellSize, p0.y);
                line = [VLine lineWithPoints:p0 :p1];
                [subGroup addObject:line];
            }
            /* vertical line for planet symbols */
            p0 = NSMakePoint(x+cellSize, y);
            p1 = NSMakePoint(p0.x, y + [planets count] * cellSize);
            line = [VLine lineWithPoints:p0 :p1];
            [subGroup addObject:line];
        }

        if ( showCombiAspects )
        {
            /* diagonal seperating lines */
            for (i=1; i<=(int)[planetsC count]; i++ )	// vertical-direction
            {
                p0 = NSMakePoint(x + cellSize, y + i * cellSize);
                p1 = NSMakePoint(p0.x + i * cellSize, y);
                line = [VLine lineWithPoints:p0 :p1];
                [line setColor:[NSColor lightGrayColor]];
                [subGroup addObject:line];

                p0 = NSMakePoint(x + cellSize + i*cellSize, y + [planetsC count] * cellSize);
                p1 = NSMakePoint(x + cellSize + [planetsH count] * cellSize, y + i*cellSize);
                line = [VLine lineWithPoints:p0 :p1];
                [line setColor:[NSColor lightGrayColor]];
                [subGroup addObject:line];
            }

            /*
             p0 = NSMakePoint(x + cellSize, y + [planetsC count] * cellSize);
             p1 = NSMakePoint(p0.x + [planetsH count] * cellSize, y);
             line = [VLine lineWithPoints:p0 :p1];
             [line setColor:[NSColor grayColor]];
             [subGroup addObject:line];
             */
            /* aspect symbols */
            w = cellSize;				// width of vertical symbols (cellSize)
            h = cellSize * [planetsC count];	// height of all composite planet entries

            /* table grid */
            for (i=0; i<=(int)[planetsH count] /*+1*/; i++ )
            {
                /* horicontal lines */
                p0 = NSMakePoint(x, y + i * cellSize);
                p1 = NSMakePoint(x + w + [planetsH count] * cellSize, p0.y);
                line = [VLine lineWithPoints:p0 :p1];
                [subGroup addObject:line];

                /* vertical lines */
                p0 = NSMakePoint(x + ((i <= (int)[planetsH count]) ? (w+i*cellSize) : 0.0), y);
                p1 = NSMakePoint(p0.x, y + ([planetsC count]+1) * cellSize);
                line = [VLine lineWithPoints:p0 :p1];
                [subGroup addObject:line];
            }

            /* aspect symbols */
        // w = cellSize;				// width of vertical symbols (cellSize)
        // h = cellSize * [planetsC count];	// height of all composite planet entries
            for (i=0, cnt=(int)[planetsH count]; i<cnt; i++ )	// rows
            {   float	deg1 = [degreeDictH floatForKey:[planetsH objectAtIndex:i]];

                /* horoscope symbols horicontal (one by one) */
                mString = [NSMutableString string];
                [mString appendFormat:@"%@", planetSymbol([planetsH objectAtIndex:i])];
                text = [VText text:mString
                           center:NSMakePoint(x + w + i*cellSize+cellSize/2.0, y + h + cellSize/2.0)
                            color:color angle:0.0 center:cp lineHeight:LINE_HEIGHT font:symbolFont];
                [subGroup addObject:text];

                /* table columns */
                for (j=0, cnt=(int)[planetsC count]; j<cnt; j++)	// columns
                {   float	deg2 = [degreeDictC floatForKey:[planetsC objectAtIndex:j]];
                    NSColor	*orbCol;

                    if ( (string = aspectSymbol(angle(deg1, deg2), orb)) )
                    {
                        orbCol = orbColor(angle(deg1, deg2), 1.0, 1.5);
                        text = [VText text:string
                                   center:NSMakePoint(x + w + i*cellSize+cellSize/4.0,
                                                      y + h - j*cellSize - cellSize*3.0/4.0)
                                    color:orbCol angle:0.0 center:cp lineHeight:cellSize-1 font:symbolFontMini];
                        [subGroup addObject:text];
                    }
                    if ( (string = aspectSymbol(mirror(deg1, deg2), orb)) )
                    {
                        orbCol = orbColor(mirror(deg1, deg2), 1.0, 1.5);
                        text = [VText text:string
                                   center:NSMakePoint(x + w + i*cellSize+cellSize*3.0/4.0,
                                                      y + h - j*cellSize - cellSize/4.0)
                                    color:orbCol angle:0.0 center:cp lineHeight:cellSize-1 font:symbolFontMini];
                        [subGroup addObject:text];
                    }
                }
            }
        }
        return subGroup; // [group addObject:subGroup];
    }
}

- (void)addEventsAt:(NSPoint)cp radius:(float)radius layer:(int)layer color:(NSColor*)color gmt:(NSCalendarDate*)gmt
{   NSArray		*layerList = [view layerList];
    NSString		*resourcePath = [[NSBundle mainBundle] resourcePath];
    NSDictionary	*weatherDict = [NSDictionary dictionaryWithContentsOfFile:[resourcePath stringByAppendingString:@"/weather.dict"]], *dayDict;
    NSMutableArray	*pressureKeys;
    int			pr, i;
    NSFont		*systemFont10 = [NSFont systemFontOfSize:10.0];
    NSFont		*systemFont12 = [NSFont systemFontOfSize:12.0];

    dayDict = [weatherDict objectForKey:[gmt descriptionWithCalendarFormat:@"%Y.%m.%d-%H:%M"]];
    pressureKeys = [[[dayDict allKeys] mutableCopy] autorelease];
    [pressureKeys sortUsingFunction:sortAsNumbers context:nil];
    for ( pr=0; pr<(int)[pressureKeys count]; pr++ )
    {
        NSString	*pressureKey = [pressureKeys objectAtIndex:pr];
        NSArray		*locArray = [dayDict objectForKey:pressureKey];
        NSMutableArray	*list = [[layerList objectAtIndex:layer+pr] list];

        for ( i=0; i<(int)[locArray count]; i++ )
        {   NSArray		*loc = [locArray objectAtIndex:i];
            float		lat = [[loc objectAtIndex:0] floatValue], lng = [[loc objectAtIndex:1] floatValue];
            VLine		*line;
            NSCalendarDate	*sideralTime = [view sideralTime:gmt longitude:lng];
            NSPoint		p0, p1;

            /* AC */
            {   float	ac;

                ac = [view ac:sideralTime latitude:lat];
//NSLog(@"sid:%@ long:%f lat:%f ac:%f", sideralTime, lng, lat, ac);
                line = [[[VLine allocWithZone:[self zone]] init] autorelease];
                p0 = NSMakePoint(cp.x-radius, cp.y);
                p1 = NSMakePoint(cp.x+radius+(radius/8.0), cp.y);
                [line setVertices:p0 :p1];
                [line setAngle:-ac withCenter:cp];
                [line setColor:color];
                [list addObject:line];

                [list addObject:[VText text:[NSString stringWithFormat:@"AC lat=%.1f lng=%.1f ", lat, lng] 	
                                      base:NSMakePoint(p1.x+(radius/6.0), p1.y) color:[NSColor blackColor]
                                     angle:ac center:cp font:systemFont10]];
            }

            /* MC */
            {   float	mc;

                mc = [view mc:sideralTime];
                line = [[[VLine allocWithZone:[self zone]] init] autorelease];
                p0 = NSMakePoint(cp.x-radius, cp.y);
                p1 = NSMakePoint(cp.x+radius+(radius/8.0), cp.y);
                [line setVertices:p0 :p1];
                [line setAngle:-mc withCenter:cp];
                [line setColor:color];
                [list addObject:line];

                [list addObject:[VText text:[NSString stringWithFormat:@"MC\nlat=%.1f\nlng=%.1f ", lat, lng]
                                      base:NSMakePoint(p1.x+(radius/4.5), p1.y) color:[NSColor blackColor]
                                     angle:mc center:cp  font:systemFont10]];
            }

            [list addObject:[VText text:[NSString stringWithFormat:@"%@ ", pressureKey]
                                  base:NSMakePoint(50.0, 50.0) color:color
                                 angle:0.0 center:cp font:systemFont12]];
        }
    }
}

- (void)addEventToHoroscopeAt:(NSPoint)cp radius:(float)radius layer:(int)layer color:(NSColor*)color
                          lat:(float)latitude long:(float)longitude date:(NSCalendarDate*)gmt
{   NSArray		*layerList = [view layerList];
    NSCalendarDate	*sideralTime;
    NSPoint		p0, p1;
    NSFont		*systemFont10 = [NSFont systemFontOfSize:10.0];

    sideralTime = [view sideralTime:gmt longitude:longitude];

    {   VLine		*line;
        NSMutableArray	*list = [[layerList objectAtIndex:layer] list];

        /* AC */
        {   float	ac;

            ac = [view ac:sideralTime latitude:latitude];
//NSLog(@"sid:%@ long:%f lat:%f ac:%f", sideralTime, longitude, latitude, ac);
            line = [[[VLine allocWithZone:[self zone]] init] autorelease];
            p0 = NSMakePoint(cp.x-radius, cp.y);
            p1 = NSMakePoint(cp.x+radius+(radius/8.0), cp.y);
            [line setVertices:p0 :p1];
            [line setAngle:-ac withCenter:cp];
            [line setColor:color];
            [list addObject:line];

            [list addObject:[VText text:[NSString stringWithFormat:@"AC\nlat=%.1f\nlng=%.1f", latitude, longitude]
                                  base:NSMakePoint(p1.x+(radius/10.0), p1.y) color:[NSColor blackColor]
                                 angle:ac center:cp font:systemFont10]];
        }

        /* MC */
        {   float	mc;

            mc = [view mc:sideralTime];
            line = [[[VLine allocWithZone:[self zone]] init] autorelease];
            p0 = NSMakePoint(cp.x-radius, cp.y);
            p1 = NSMakePoint(cp.x+radius+(radius/8.0), cp.y);
            [line setVertices:p0 :p1];
            [line setAngle:-mc withCenter:cp];
            [line setColor:color];
            [list addObject:line];

            [list addObject:[VText text:[NSString stringWithFormat:@"MC\nlat=%.1f\nlng=%.1f", latitude, longitude]
                                  base:NSMakePoint(p1.x+(radius/10.0), p1.y) color:[NSColor blackColor]
                                 angle:mc center:cp font:systemFont10]];
        }
    }
}


- (void)setCompositeLayerEnabled:(BOOL)enabled
{
    [[[view layerList] objectAtIndex:ixComposite] setState:enabled];
    if (ixMaskComposite >= 0 && !Prefs_ACFix)
        [[[view layerList] objectAtIndex:ixMaskComposite] setState:enabled];
    [view cache:NSZeroRect];
    [[NSNotificationCenter defaultCenter] postNotificationName:DocLayerListHasChanged object:view];
}
#if 0	// deprecated
- (BOOL)getMarksFromTemplate:(NSArray*)list
                      center:(NSPoint*)center
                      radius:(float*)radius                        start:(NSPoint*)start
             radiusComposite:(float*)radiusComposite      startComposite:(NSPoint*)startComposite
                 tableOrigin:(NSPoint*)tableOrigin  tableOriginComposite:(NSPoint*)tableOriginC
{   int		i, okNum = 0;
    BOOL	okCenter = NO, okStart = NO, okStartC = NO, okTable = NO, okTableC = NO;

    for (i=0; i<[list count]; i++)
    {   VMark	*g = [list objectAtIndex:i];

        if ([g isKindOfClass:[VMark class]])
        {
            if ([[g name] isEqual:@"CENTER"])
            {   *center = [g origin]; okCenter = YES; }
            else if ([[g name] isEqual:@"START"])
            {
                *start = [g origin];
                *radius = sqrt( SqrDistPoints(*center, [g origin]) );
                okStart = YES;
            }
            else if ([[g name] isEqual:@"START_C"])
            {
                *startComposite = [g origin];
                *radiusComposite = sqrt( SqrDistPoints(*center, [g origin]) );
                okStartC = YES;
            }
            else if ([[g name] isEqual:@"TABLE"])
            {   *tableOrigin = [g origin]; okTable = YES; }
            else if ([[g name] isEqual:@"TABLE_C"])
            {   *tableOriginC = [g origin]; okTableC = YES; }
            if ((++okNum) >= 5)
                break;
        }
    }
    if (!okCenter)
        NSLog(@"Mark with name 'CENTER' missing on template layer");
    if (!okStart)
        NSLog(@"Mark with name 'START' missing on template layer");
    if (!okStartC)
        NSLog(@"Mark with name 'START_C' missing on template layer");
    if (!okTable)
        NSLog(@"Mark with name 'TABLE' missing on template layer");
    if (!okTableC)
        NSLog(@"Mark with name 'TABLE_C' missing on template layer");
    return okCenter && okStart && okStartC && okTable && okTableC;
}
#endif

/* read template
 * Marks:
 *   CENTER	center mark for chart
 *   CENTER_#	center of interval chart
 *   START	radius and start angle of chart
 *   START_#	radius and start angle of interval charts
 *   START_C	same for composite chart
 *   TABLE	table origin
 *   TABLE_C	table origin for composite chart
 *   INFO	info position
 * Text:
 *   TITLE
 *   NAME
 *   LOCATION
 *   DATE
 *   TIME_#	time text for interval chart, optinally with interval (-#, +# in minutes)
 */
- (ChartTemplate)chartTemplate
{
    return chartTemplate;
}
- (ChartTemplate)getGetTemplate:(NSArray*)list
{   int		i, ix, maxInterval = MAX_CHARTINTERVALS;
    NSRange	range;
    VText	*text;
    BOOL	templateOK = NO;

    chartTemplate.center = NSZeroPoint;
    chartTemplate.start  = chartTemplate.startC = NSZeroPoint;
    chartTemplate.radius = chartTemplate.radiusC = 0.0;
    chartTemplate.table  = chartTemplate.tableC = NSZeroPoint;
    chartTemplate.info   = NSZeroPoint;
    for (i=0; i<maxInterval; i++)
    {
        chartTemplate.centerI[i] = NSZeroPoint;
        chartTemplate.startI[i]  = NSZeroPoint;
        chartTemplate.radiusI[i] = 0.0;
    }
    chartTemplate.components = 0;
    chartTemplate.layerMask   = -1;
    chartTemplate.layerChart  = -1;
    chartTemplate.layerMaskC  = -1;
    chartTemplate.layerChartC = -1;

    for (i=0; i<[list count]; i++)
    {   VMark	*g = [list objectAtIndex:i];

        if ([g isKindOfClass:[VMark class]])
        {
            if ([[g name] isEqual:@"CENTER"])
                chartTemplate.center = [g origin];
            else if ([[g name] isEqual:@"START"])
                chartTemplate.start = [g origin];
            else if ([[g name] isEqual:@"START_C"])
                chartTemplate.startC = [g origin];
            else if ([[g name] isEqual:@"TABLE"])
                chartTemplate.table = [g origin];
            else if ([[g name] isEqual:@"TABLE_C"])
                chartTemplate.tableC = [g origin];
            else if ([[g name] isEqual:@"INFO"])
                chartTemplate.info = [g origin];
            else if ([[g name] hasPrefix:@"CENTER_"])
            {
                range = [[g name] rangeOfString:@"_"];
                if (range.length)
                {   ix = (range.length) ? ([[[g name] substringFromIndex:range.location+1] intValue]-1) : -1;
                    if (ix < maxInterval)
                        chartTemplate.centerI[ix] = [g origin];
                }
            }
            else if ([[g name] hasPrefix:@"START_"])
            {
                range = [[g name] rangeOfString:@"_"];
                if (range.length)
                {   ix = (range.length) ? ([[[g name] substringFromIndex:range.location+1] intValue]-1) : -1;
                    if (ix < maxInterval)
                        chartTemplate.startI[ix] = [g origin];
                }
            }
            else
                continue;
            templateOK = YES;	// found some contents
        }
    }

    text = [self textWithSubstring:@"#COMPONENTS" fromTemplate:list];
    if (text)
    {   NSString	*string = [text string];

        chartTemplate.components = 0;
        if ([string rangeOfString:@"ZODIAC"].length)
            chartTemplate.components |= CHART_ZODIAC;
        if ([string rangeOfString:@"SCALE"].length)
            chartTemplate.components |= CHART_SCALE;
        if ([string rangeOfString:@"OBJECTS"].length)
            chartTemplate.components |= CHART_OBJECTS;
        if ([string rangeOfString:@"ASPECTLINES"].length)
            chartTemplate.components |= CHART_ASPECTLINES;
        if ([string rangeOfString:@"DECLINATION"].length)
            chartTemplate.components |= CHART_DECLINATION;
        if ([string rangeOfString:@"HOUSES"].length)
            chartTemplate.components |= CHART_HOUSES;
        if ([string rangeOfString:@"TOPOSCALE"].length)
            chartTemplate.components |= CHART_TOPOSCALE;
        if ([string rangeOfString:@"STAR"].length)
            chartTemplate.components |= CHART_STAR;
        if ([string rangeOfString:@"WAVE"].length)
            chartTemplate.components |= CHART_WAVE;
    }
    else
    {
        if ( DiffPoint(chartTemplate.center, NSZeroPoint) > 0.0 )
            chartTemplate.components = 0xFFFFFF;	// all components
    }

    /* layer indexe */
    text = [self textWithSubstring:@"#LAYERS" fromTemplate:list];
    if (text)
    {   NSString	*string = [text string];
        NSRange		range;

        range = [string rangeOfString:@"MASK="];
        if (range.length)
            chartTemplate.layerMask   = [[string substringFromIndex:range.location+range.length] intValue];
        range = [string rangeOfString:@"CHART="];
        if (range.length)
            chartTemplate.layerChart  = [[string substringFromIndex:range.location+range.length] intValue];
        range = [string rangeOfString:@"MASKC="];
        if (range.length)
            chartTemplate.layerMaskC  = [[string substringFromIndex:range.location+range.length] intValue];
        range = [string rangeOfString:@"CHARTC="];
        if (range.length)
            chartTemplate.layerChartC = [[string substringFromIndex:range.location+range.length] intValue];
    }

    if ( DiffPoint(chartTemplate.center, NSZeroPoint) > 0.0)
    {
        if ( DiffPoint(chartTemplate.start,  NSZeroPoint) > 0.0 )
            chartTemplate.radius = sqrt( SqrDistPoints(chartTemplate.center, chartTemplate.start) );
        if ( DiffPoint(chartTemplate.startC, NSZeroPoint) > 0.0 )
            chartTemplate.radiusC = sqrt( SqrDistPoints(chartTemplate.center, chartTemplate.startC) );
    }
    for (i=0; i<maxInterval; i++)
        if ( DiffPoint(chartTemplate.centerI[i], NSZeroPoint) > 0.0)
        {
            if ( DiffPoint(chartTemplate.startI[i],  NSZeroPoint) > 0.0 )
                chartTemplate.radiusI[i] = sqrt( SqrDistPoints(chartTemplate.centerI[i], chartTemplate.startI[i]) );
        }

    /* no template -> set default values */
    if (!templateOK)
    {
        chartTemplate.center  = NSMakePoint(296.0, 535.0);
        chartTemplate.radius  = 204.0;
        chartTemplate.radiusC = 144.0;
        chartTemplate.table   = NSMakePoint(20.0, 37.0);
        chartTemplate.tableC  = NSMakePoint(573.0, 37.0);
        chartTemplate.components = 0xFFFFFF;	// all components
    }

    return chartTemplate;
}

/* modified: 2006-06-03
 */
- (void)radix:(NSCalendarDate*)gmt
    longitude:(float)longitude latitude:(float)latitude
         city:(NSString*)city
        title:(NSString*)title composite:(NSDictionary*)composite
{   NSArray			*layerList = [view layerList];
    LayerObject			*layer;
    VGraphic			*g;
    NSPoint			cp, tableOrigin, tableOriginC;
    float			radius, radiusC, rotateAC = 0.0;
    int				ixMaskRadix = -1, ixRadix = 0, ixTemplate, i;
    NSArray			*template = nil;	// list of marks and text showing positions
    NSMutableDictionary		*infoDict = nil;
    NSNotificationCenter	*notificationCenter = [NSNotificationCenter defaultCenter];

    if (!gmt || ![[[view window] title] hasPrefix:CHART_PREFIX])
        return;

    ephemeris = [Astro_Principal ephemeris];

    ixMaskComposite = -1;

    ixTemplate = [layerList count]-1;		// default = last layer
    for (i=[layerList count]-1; i>=0;  i--)	// get template layer
        if ( [(LayerObject*)[layerList objectAtIndex:i] type] == LAYER_TEMPLATE )
            ixTemplate = i;

    /* get positions
     */
    if ([layerList count] <= 1)		// no way
    {
        NSLog(@"Not enough layers (%d) to calculate chart !", [layerList count]);
        return;
    }
    else if ([layerList count] == 2)	// use defaults
    {
        ixRadix = 0;
        ixComposite = 1;
    }
    else if ([layerList count] < 5)	// use template and calculate mask defaults
    {
        ixRadix = 0;
        ixComposite = 1;
    }
    else				// use template and masks
    {
        ixMaskRadix = 0;
        ixRadix = 1;
        ixMaskComposite = 2;
        ixComposite = 3;
    }
    template = [[layerList objectAtIndex:ixTemplate] list];
    chartTemplate = [self getGetTemplate:template];

    /* shortcuts */
    cp = chartTemplate.center;
    radius  = chartTemplate.radius;
    radiusC = chartTemplate.radiusC;
    tableOrigin  = chartTemplate.table;
    tableOriginC = chartTemplate.tableC;
    if (chartTemplate.layerMask   >= 0 && chartTemplate.layerMask   < [layerList count])
        ixMaskRadix     = chartTemplate.layerMask;
    if (chartTemplate.layerChart  >= 0 && chartTemplate.layerChart  < [layerList count])
    {   ixRadix         = chartTemplate.layerChart;
        ixMaskComposite = ixComposite = -1;	// reset composite, as layers are set by template
    }
    if (chartTemplate.layerMaskC  >= 0 && chartTemplate.layerMaskC  < [layerList count])
        ixMaskComposite = chartTemplate.layerMaskC;
    if (chartTemplate.layerChartC >= 0 && chartTemplate.layerChartC < [layerList count])
        ixComposite     = chartTemplate.layerChartC;
    /* retain defaults */
    chartTemplate.layerMask   = ixRadix;
    chartTemplate.layerChart  = ixMaskRadix;
    chartTemplate.layerMaskC  = ixMaskComposite;
    chartTemplate.layerChartC = ixComposite;

    [astroFont release];
    astroFont = [Prefs_AstroFont retain];
    if (![astroFont length])
        astroFont = @"vhfAstro";
/*#ifdef GNUSTEP_BASE_LIBRARY	// Workaround: GNUstep (X-Backend) makes the first char of the name capital
    if (![NSFont fontWithName:astroFont size:12.0])
        astroFont = [[[[Prefs_AstroFont substringToIndex:1] uppercaseString]
                     stringByAppendingString:[Prefs_AstroFont substringFromIndex:1]] retain];
#endif*/

    /* chart
     */
    layer = [layerList objectAtIndex:ixRadix];
    [layer removeAllObjects];
    if (Prefs_ACFix)
        rotateAC = 180.0 - ac(sideralTime(gmt, longitude), latitude);

    /* no mask -> calculate basics
     */
    if (ixMaskRadix < 0)		// calculate basics
    {
        /* center star */
        if (chartTemplate.components & CHART_CENTER)
            [layer addObject:[self starWithCenter:cp radius:radius*STAR_SIZE rotate:rotateAC]];

        /* Ecliptic circle: global houses and signs */
        if (chartTemplate.components & CHART_ZODIAC)
            [layer addObject:[self eclipticWithCenter:cp
                                              radius1:radius*RADIUS_1 radius2:radius*RADIUS_2
                                           symbolSize:radius*SYMBOL_SIZE rotate:rotateAC]];

        /* scale */
        if (chartTemplate.components & CHART_SCALE)
            [layer addObject:[self scaleWithCenter:cp radius:radius*RADIUS_2
                                        scaleSize1:radius*SCALE_MIN scaleSize2:radius*SCALE_MAX
                                        degreeType:SCALEDEG_360 rotate:rotateAC]];
    }
    /* rotate AC to the left
     * we create a rotated copy of the mask and turn off the mask
     */
    else if (rotateAC)	// rotate Mask
    {   NSArray	*list = [[layerList objectAtIndex:ixMaskRadix] list];

        [[layerList objectAtIndex:ixMaskRadix] setState:0];
        for (i=0; i<(int)[list count]; i++)
        {   VGraphic	*g = [[[list objectAtIndex:i] copy] autorelease];

            if (![g isLocked])
            {   [g setAngle:-rotateAC withCenter:cp];
                if ([g isKindOfClass:[VText class]] && [(VText*)g isSerialNumber])
                    [g rotate:[g rotAngle]];
            }
            [layer addObject:g];
        }
    }
    else		// no rotation -> enable mask
        [[layerList objectAtIndex:ixMaskRadix] setState:1];

    g = [self horoscopeWithCenter:cp radius:radius tableOrigin:tableOrigin
                         template:template color:[NSColor blackColor]
                            title:title lat:latitude lon:longitude city:city gmt:gmt];
    [layer addObjectsFromArray:[(VGroup*)g list]];

    /* give other modules a chance to add their stuff
     */
    infoDict = [NSMutableDictionary dictionaryWithObjectsAndKeys:
                    layerList,                                  @"layerList",
                    layer,                                      @"layerChart",
                    [NSNumber numberWithFloat:rotateAC],        @"rotateAC",
                    title,                                      @"title",
                    gmt,                                        @"date",
                    [NSNumber numberWithFloat:latitude],        @"lat",
                    [NSNumber numberWithFloat:longitude],       @"lon",
                    propertyListFromNSPoint(cp),                @"center",
                    [NSNumber numberWithFloat:radius],          @"radius",
                    [NSNumber numberWithFloat:radius*RADIUS_3], @"radiusComposite",
                    [NSNumber numberWithFloat:radius*RADIUS_4], @"radiusCenter",
                    composite,                                  @"composite", nil];
    [notificationCenter postNotificationName:AstroAddChartParts
                                      object:self userInfo:infoDict];

    /* Create Interval Chart
     */
    if ( DiffPoint(chartTemplate.centerI[0], NSZeroPoint) > 0.0 )
    {   int	i;
        int	min = Prefs_AstroInterval;

        for (i=0; DiffPoint(chartTemplate.centerI[i], NSZeroPoint) > 0.0; i++)
        {   NSString		*strFormat;
            NSCalendarDate	*dateI = gmt;
            NSPoint		cp;
            float		rad;
            VText		*text;

            cp    = chartTemplate.centerI[i];
            rad   = chartTemplate.radiusI[i];

            strFormat = [NSString stringWithFormat:@"#TIME_%d", i+1];
            if ( (text = [self textWithSubstring:strFormat fromTemplate:template]) )
            {   int	minutes;

                minutes = [[[text string] substringFromIndex:[strFormat length]] intValue];
                // FIXME: overwrite n * minimum interval to allow -10, -5, 0, +5, +10
                if (min != 0 && minutes != 0)	// override interval from prefs
                    minutes = (minutes>0) ? min : -min;
                dateI = [gmt dateByAddingYears:0 months:0 days:0 hours:0 minutes:minutes seconds:0];
                [text replaceTextWithString:[dateI descriptionWithCalendarFormat:@"%H:%M"]];
                [layer addObject:text];
            }

            [infoDict setObject:dateI                          forKey:@"date"];
            [infoDict setObject:propertyListFromNSPoint(cp)    forKey:@"center"];
            [infoDict setObject:[NSNumber numberWithFloat:rad] forKey:@"radius"];
            [infoDict setObject:[NSNumber numberWithFloat:rad] forKey:@"radiusComposite"];
            [infoDict setObject:[NSNumber numberWithFloat:rad] forKey:@"radiusCenter"];
            [infoDict setObject:@"1"                           forKey:@"isIntervalChart"];
            [notificationCenter postNotificationName:AstroAddChartParts
                                              object:self userInfo:infoDict];
        }
    }

    /* composite */
    if ( [composite intForKey:@"doCalculate"] )
    {   NSCalendarDate	*gmtH = gmt;	// save date of outside chart for composite table

        layer = [layerList objectAtIndex:ixComposite];
        [layer removeAllObjects];

        gmt = [composite objectForKey:@"date"];
        if ([gmt isKindOfClass:[NSString class]])
            gmt = [NSCalendarDate dateWithString:[composite stringForKey:@"date"]
                                  calendarFormat:@"%Y-%m-%d %H:%M %Z"];
        if (!gmt)
            gmt = [NSCalendarDate date];
        [gmt setTimeZone:[NSTimeZone timeZoneForSecondsFromGMT:0]];
        if (Prefs_ACFix)
            rotateAC = 180.0 - ac(sideralTime(gmt, [composite floatForKey:@"lon"]), [composite floatForKey:@"lat"]);

        if (ixMaskComposite < 0)	// calculate basics
        {   NSMutableArray	*list;
            VPath		*path;
            VArc		*arc;

            /* filled ring */
            path = [VPath path];
            [path setFillColor:GRAY_COLOR];
            [path setFilled:YES optimize:NO];
            list = [path list];
            arc = [VArc arc];
            [arc setCenter:cp start:NSMakePoint(cp.x+radius*RADIUS_3, cp.y) angle:360.0];
            [list addObject:arc];
            arc = [VArc arc];
            [arc setCenter:cp start:NSMakePoint(cp.x+radius*RADIUS_4, cp.y) angle:360.0];
            [list addObject:arc];
            [layer addObject:path];

            /* black border */
            arc = [VArc arc];
            [arc setCenter:cp start:NSMakePoint(cp.x+radius*RADIUS_3, cp.y) angle:360.0];
            [layer addObject:arc];
            arc = [VArc arc];
            [arc setCenter:cp start:NSMakePoint(cp.x+radius*RADIUS_4, cp.y) angle:360.0];
            [layer addObject:arc];

            /* scale */
            [layer addObject:[self scaleWithCenter:cp radius:radius*RADIUS_3
                                        scaleSize1:radius*SCALE_MIN scaleSize2:radius*SCALE_MAX
                                        degreeType:SCALEDEG_NONE rotate:rotateAC]];
        }
        /* rotate AC to the left
         * we create a rotated copy of the mask and turn off the mask
         */
        else if (rotateAC)	// rotate Mask
        {   NSArray	*list = [[layerList objectAtIndex:ixMaskComposite] list];

            [[layerList objectAtIndex:ixMaskComposite] setState:0];
            for (i=0; i<(int)[list count]; i++)
            {   VGraphic	*g = [[[list objectAtIndex:i] copy] autorelease];

                if (![g isLocked])
                {   [g setAngle:-rotateAC withCenter:cp];
                    if ([g isKindOfClass:[VText class]] && [(VText*)g isSerialNumber])
                        [g rotate:[g rotAngle]];
                }
                [layer addObject:g];
            }
        }
        else			// no rotation -> enable mask
            [[layerList objectAtIndex:ixMaskComposite] setState:1];

        /* add composite chart */
        g = [self compositeWithCenter:cp radius:radius template:template color:[NSColor blackColor]
                                title:[composite stringForKey:@"name"]
                                  lat:[composite floatForKey:@"lat"] lon:[composite floatForKey:@"lon"]
                                 city:[composite stringForKey:@"city"] gmt:gmt];
        [layer addObject:g];

        /* add composite table */
        g = [self compositeTable:cp tableOrigin:(NSPoint)tableOriginC
                            latH:latitude                       lonH:longitude                      gmtH:gmtH
                            latC:[composite floatForKey:@"lat"] lonC:[composite floatForKey:@"lon"] gmtC:gmt];
        [layer addObject:g];

        [layer setState:1];	// make composite layer visible
    }
    [view drawAndDisplay];
}

- (void)dealloc
{
    [view release];
    [astroFont release];
    [super dealloc];
}

@end
